<?php
/**
 * Клас за админ панела
 */

if (!defined('ABSPATH')) {
    exit;
}

class ReviewXpress_Admin {
    
    /**
     * Конструктор
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_reviewxpress_approve_review', array($this, 'approve_review'));
        add_action('wp_ajax_reviewxpress_reject_review', array($this, 'reject_review'));
        add_action('wp_ajax_reviewxpress_delete_review', array($this, 'delete_review'));
        add_action('wp_ajax_reviewxpress_get_stats', array($this, 'get_stats'));
        add_action('wp_ajax_reviewxpress_export_reviews', array($this, 'export_reviews'));
        
        // Admin-post handler за запазване на настройките
        add_action('admin_post_reviewxpress_save_settings', array($this, 'save_settings'));
        
        // Admin notices за показване на съобщения след redirect
        add_action('admin_notices', array($this, 'show_admin_notices'));
    }
    
    /**
     * Добавяне на меню в админ панела
     */
    public function add_admin_menu() {
        // Главно меню
        add_menu_page(
            esc_html__('ReviewXpress', 'reviewxpress'),
            esc_html__('ReviewXpress', 'reviewxpress'),
            'manage_options',
            'reviewxpress',
            array($this, 'admin_page'),
            'dashicons-star-filled',
            30
        );
        
        // All Reviews (leads to main page)
        add_submenu_page(
            'reviewxpress',
            esc_html__('All Reviews', 'reviewxpress'),
            esc_html__('All Reviews', 'reviewxpress'),
            'manage_options',
            'reviewxpress',
            array($this, 'admin_page')
        );

        // Settings - submenu
        add_submenu_page(
            'reviewxpress',
            esc_html__('Settings', 'reviewxpress'),
            esc_html__('Settings', 'reviewxpress'),
            'manage_options',
            'reviewxpress-settings-general',
            array($this, 'settings_page_general')
        );

        // Emails - submenu
        add_submenu_page(
            'reviewxpress',
            esc_html__('Emails', 'reviewxpress'),
            esc_html__('Emails', 'reviewxpress'),
            'manage_options',
            'reviewxpress-settings-emails',
            array($this, 'settings_page_emails')
        );

        // Translations - submenu
        add_submenu_page(
            'reviewxpress',
            esc_html__('Translations & Design', 'reviewxpress'),
            esc_html__('Translations & Design', 'reviewxpress'),
            'manage_options',
            'reviewxpress-settings-translations',
            array($this, 'settings_page_translations')
        );
        
        // Shortcodes - submenu
        add_submenu_page(
            'reviewxpress',
            esc_html__('Shortcodes', 'reviewxpress'),
            esc_html__('Shortcodes', 'reviewxpress'),
            'manage_options',
            'reviewxpress-settings-defaults',
            array($this, 'settings_page_defaults')
        );
        
        // About - submenu
        add_submenu_page(
            'reviewxpress',
            esc_html__('About', 'reviewxpress'),
            esc_html__('About', 'reviewxpress'),
            'manage_options',
            'reviewxpress-about',
            array($this, 'about_page')
        );
    }
    
    /**
     * Регистриране на настройки
     */
    public function register_settings() {
        register_setting('reviewxpress_settings', 'reviewxpress_settings', array($this, 'sanitize_settings'));
    }
    
    /**
     * Почистване на настройки
     */
    public function sanitize_settings($input) {
        $sanitized = array();
        
        $sanitized['auto_approve'] = !empty($input['auto_approve']) ? 1 : 0;
        $sanitized['require_woocommerce_order'] = !empty($input['require_woocommerce_order']) ? 1 : 0;
        $sanitized['allow_guest_reviews'] = !empty($input['allow_guest_reviews']) ? 1 : 0;
        $sanitized['max_images'] = intval($input['max_images']);
        $sanitized['max_image_size'] = intval($input['max_image_size']);
        $sanitized['max_video_size'] = intval($input['max_video_size']);
        $sanitized['allowed_image_types'] = sanitize_text_field($input['allowed_image_types']);
        // $sanitized['auto_email_customer'] = ... (Про версия)
        // $sanitized['email_template'] = ... (Про версия)
        
        return $sanitized;
    }
    
    /**
     * Главна страница на админ панела
     */
    public function admin_page() {
        // Check user capabilities - MUST be first
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'reviewxpress'));
        }

        // If nonce is missing, add it and redirect (do not break menu navigation)
        if (!isset($_GET['review_nonce'])) {
            $url = add_query_arg(
                array(
                    'page' => 'reviewxpress',
                    'review_nonce' => wp_create_nonce('reviewxpress_admin'),
                ),
                admin_url('admin.php')
            );
            
            wp_safe_redirect($url);
            exit;
        }

        // Now enforce nonce validity (fail early)
        if (!isset($_GET['review_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['review_nonce'])), 'reviewxpress_admin')) {
            wp_die(esc_html__('Security check failed.', 'reviewxpress'));
        }
        
        $database = new ReviewXpress_Database();
        
        // Now safe to read $_GET parameters (nonce verified)
        $status = isset($_GET['status']) ? sanitize_text_field(wp_unslash($_GET['status'])) : 'all';
        $product_id = isset($_GET['product_id']) ? absint($_GET['product_id']) : 0;
        $search = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';
        
        // Пагинация
        $per_page = 20;
        $current_page = isset($_GET['paged']) ? max(1, absint($_GET['paged'])) : 1;
        $offset = ($current_page - 1) * $per_page;
        
        // Сортиране
        // Validate orderby parameter - strict whitelist
        $allowed_orderby = array('created_at', 'name', 'email', 'rating', 'status', 'id');
        $orderby_raw = isset($_GET['orderby']) ? sanitize_key(wp_unslash($_GET['orderby'])) : 'created_at';
        $orderby = in_array($orderby_raw, $allowed_orderby, true) ? $orderby_raw : 'created_at';
        
        // Validate order parameter - strict whitelist (ASC or DESC only)
        $order_raw = isset($_GET['order']) ? sanitize_key(wp_unslash($_GET['order'])) : 'DESC';
        $order = (strtoupper($order_raw) === 'ASC') ? 'ASC' : 'DESC';
        
        // Вземане на ревюта
        $args = array(
            'status' => $status,
            'limit' => $per_page,
            'offset' => $offset,
            'orderby' => $orderby,
            'order' => $order
        );
        
        // Clear cache if needed (when filters change)
        if (isset($_GET['status']) || isset($_GET['product_id']) || isset($_GET['s']) || isset($_GET['paged'])) {
            wp_cache_flush_group('reviewxpress');
        }
        
        global $wpdb;
        $reviews_table = $wpdb->prefix . 'reviewxpress_reviews';
        $reviews_table_escaped = esc_sql($reviews_table);
        $sql = "SELECT COUNT(*) FROM `{$reviews_table_escaped}`";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $total_reviews = $wpdb->get_var($sql);
        
        // If no reviews in database, return empty array
        if ($total_reviews == 0) {
            $reviews = array();
        } else {
            if ($product_id > 0) {
                $reviews = $database->get_reviews($product_id, $args);
            } else {
                $reviews = $this->get_all_reviews($args, $search);
            }
        }
        
        // Ensure reviews is always an array
        if (!is_array($reviews)) {
            $reviews = array();
        }
        
        // Статистики
        $stats = $this->get_review_stats();
        
        include REVIEWXPRESS_PLUGIN_PATH . 'templates/admin/reviews-main.php';
    }
    
    /**
     * Страница с настройки
     */
    public function settings_page_general() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'reviewxpress'));
        }
        
        $settings = get_option('reviewxpress_settings', array());
        include REVIEWXPRESS_PLUGIN_PATH . 'templates/admin/settings-general.php';
    }

    public function settings_page_emails() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'reviewxpress'));
        }
        
        $settings = get_option('reviewxpress_settings', array());
        include REVIEWXPRESS_PLUGIN_PATH . 'templates/admin/settings-emails.php';
    }

    public function settings_page_translations() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'reviewxpress'));
        }
        
        $settings = get_option('reviewxpress_settings', array());
        include REVIEWXPRESS_PLUGIN_PATH . 'templates/admin/settings-translations.php';
    }
    
    public function settings_page_defaults() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'reviewxpress'));
        }
        
        $settings = get_option('reviewxpress_settings', array());
        include REVIEWXPRESS_PLUGIN_PATH . 'templates/admin/settings-shortcodes.php';
    }
    
    /**
     * Страница "За нас"
     */
    public function about_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'reviewxpress'));
        }
        
        include REVIEWXPRESS_PLUGIN_PATH . 'templates/admin/about.php';
    }

    /**
     * Вземане на всички ревюта
     */
    private function get_all_reviews($args, $search = '') {
        global $wpdb;
        
        $reviews_table = $wpdb->prefix . 'reviewxpress_reviews';
        
        $where_clauses = array();
        $where_values = array();
        
        if ($args['status'] !== 'all') {
            $where_clauses[] = "status = %s";
            $where_values[] = sanitize_text_field($args['status']);
        }
        
        if (!empty($search)) {
            // Sanitize search term
            $search_raw = wp_unslash($search);
            $search = sanitize_text_field($search_raw);
            $where_clauses[] = "(name LIKE %s OR email LIKE %s OR review_text LIKE %s)";
            // Search term is sanitized and escaped with esc_like(), then passed to prepare() which is safe
            $like = '%' . $wpdb->esc_like($search) . '%';
            $where_values[] = $like;
            $where_values[] = $like;
            $where_values[] = $like;
        }
        
        // Validate orderby parameter - strict whitelist
        $allowed_orderby = array('created_at', 'name', 'email', 'rating', 'status', 'id');
        $orderby_raw = isset($args['orderby']) ? sanitize_key($args['orderby']) : 'created_at';
        $orderby = in_array($orderby_raw, $allowed_orderby, true) ? $orderby_raw : 'created_at';
        
        // Validate order parameter - strict whitelist (ASC or DESC only)
        $order_raw = isset($args['order']) ? sanitize_key($args['order']) : 'DESC';
        $order = (strtoupper($order_raw) === 'ASC') ? 'ASC' : 'DESC';
        
        // Build ORDER BY clause using switch statement (no dynamic column names)
        $orderby_clause = '';
        switch ($orderby) {
            case 'id':
                $orderby_clause = ($order === 'ASC') ? 'ORDER BY r.id ASC' : 'ORDER BY r.id DESC';
                break;
            case 'created_at':
                $orderby_clause = ($order === 'ASC') ? 'ORDER BY r.created_at ASC' : 'ORDER BY r.created_at DESC';
                break;
            case 'name':
                $orderby_clause = ($order === 'ASC') ? 'ORDER BY r.name ASC, r.created_at DESC' : 'ORDER BY r.name DESC, r.created_at DESC';
                break;
            case 'email':
                $orderby_clause = ($order === 'ASC') ? 'ORDER BY r.email ASC, r.created_at DESC' : 'ORDER BY r.email DESC, r.created_at DESC';
                break;
            case 'rating':
                $orderby_clause = ($order === 'ASC') ? 'ORDER BY r.rating ASC, r.created_at DESC' : 'ORDER BY r.rating DESC, r.created_at DESC';
                break;
            case 'status':
                $orderby_clause = ($order === 'ASC') ? 'ORDER BY r.status ASC, r.created_at DESC' : 'ORDER BY r.status DESC, r.created_at DESC';
                break;
            default:
                $orderby_clause = 'ORDER BY r.created_at DESC';
                break;
        }
        
        // Build SQL with proper escaping - use whitelist for table/column names
        $reviews_table_escaped = esc_sql($reviews_table);
        $posts_table_escaped = esc_sql($wpdb->prefix . 'posts');
        $media_table_escaped = esc_sql($wpdb->prefix . 'reviewxpress_media');
        
        // Build WHERE clause - always include WHERE 1=1 for consistency
        if (!empty($where_clauses)) {
            $where_sql = 'WHERE ' . implode(' AND ', $where_clauses);
        } else {
            $where_sql = 'WHERE 1=1';
        }
        
        // Sanitize all WHERE values
        $where_values = array_map('sanitize_text_field', $where_values);
        
        // Sanitize limit and offset - ensure they are positive integers
        $limit = absint($args['limit']);
        $offset = absint($args['offset']);
        if ($limit <= 0) {
            $limit = 20; // Default per page
        }
        
        // Build base SQL query
        // Table names are escaped, ORDER BY uses predefined strings from switch statement
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $base_sql = "SELECT r.*, 
                    p.post_title as product_name,
                    GROUP_CONCAT(m.file_url ORDER BY m.media_type, m.id) as media_urls,
                    GROUP_CONCAT(m.media_type ORDER BY m.media_type, m.id) as media_types
             FROM `{$reviews_table_escaped}` r
             LEFT JOIN `{$posts_table_escaped}` p ON r.product_id = p.ID
             LEFT JOIN `{$media_table_escaped}` m ON r.id = m.review_id
             {$where_sql}
             GROUP BY r.id
             {$orderby_clause}
             LIMIT %d OFFSET %d";
        
        // Use caching for better performance
        $cache_key = 'reviewxpress_reviews_' . md5(serialize($args) . $search);
        $results = wp_cache_get($cache_key, 'reviewxpress');
        
        if (false === $results) {
            // Table and column names are escaped with esc_sql(), so safe to use in prepare()
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $results = $wpdb->get_results($wpdb->prepare($base_sql, ...array_merge($where_values, array($limit, $offset))));
            
            // Ensure results is always an array
            if (!is_array($results)) {
                $results = array();
            }
            
            wp_cache_set($cache_key, $results, 'reviewxpress', 300); // Cache for 5 minutes
        }
        
        // Обработка на медиите
        if (!empty($results)) {
            foreach ($results as &$review) {
                $review->media = array();
                if (!empty($review->media_urls)) {
                    $urls = explode(',', $review->media_urls);
                    $types = explode(',', $review->media_types);
                    
                    for ($i = 0; $i < count($urls); $i++) {
                        $review->media[] = array(
                            'url' => $urls[$i],
                            'type' => $types[$i]
                        );
                    }
                }
                
                unset($review->media_urls, $review->media_types);
            }
        }
        
        return $results;
    }
    
    /**
     * Вземане на статистики
     */
    private function get_review_stats() {
        global $wpdb;
        
        $reviews_table = $wpdb->prefix . 'reviewxpress_reviews';
        
        // Build SQL with proper escaping
        $reviews_table_escaped = esc_sql($reviews_table);
        
        // Table name is escaped with esc_sql(), so safe to interpolate
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $sql = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                    SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
                    SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected,
                    COALESCE(AVG(rating), 0) as average_rating
                FROM `{$reviews_table_escaped}`";
        
        // Use caching for better performance
        $cache_key = 'reviewxpress_stats';
        $stats = wp_cache_get($cache_key, 'reviewxpress');
        
        if (false === $stats) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
            $stats = $wpdb->get_row($sql);
            wp_cache_set($cache_key, $stats, 'reviewxpress', 600); // Cache for 10 minutes
        }
        
        // Ensure all values are set to 0 if NULL
        if (!$stats) {
            $stats = (object) array(
                'total' => 0,
                'pending' => 0,
                'approved' => 0,
                'rejected' => 0,
                'average_rating' => 0
            );
        } else {
            $stats->total = intval($stats->total);
            $stats->pending = intval($stats->pending);
            $stats->approved = intval($stats->approved);
            $stats->rejected = intval($stats->rejected);
            $stats->average_rating = floatval($stats->average_rating);
        }
        
        return $stats;
    }
    
    /**
     * Одобряване на ревю
     */
    public function approve_review() {
        try {
            // Verify nonce and check permissions
            check_ajax_referer('reviewxpress_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(array('message' => esc_html__('You do not have permission for this action.', 'reviewxpress')));
                return;
            }
            
            // Extract and sanitize only needed keys from $_POST (after nonce and capability check)
            $review_id = isset($_POST['review_id']) ? absint($_POST['review_id']) : 0;
            if ($review_id <= 0) {
                wp_send_json_error(esc_html__('Invalid review ID.', 'reviewxpress'));
                return;
            }
            
            $database = new ReviewXpress_Database();
            $result = $database->update_review_status($review_id, 'approved');
            
            if ($result !== false) {
                wp_send_json_success(esc_html__('Review approved successfully.', 'reviewxpress'));
            } else {
                wp_send_json_error(esc_html__('An error occurred while approving the review.', 'reviewxpress'));
            }
        } catch (Exception $e) {
            wp_send_json_error(esc_html__('An error occurred while approving the review.', 'reviewxpress'));
        }
    }
    
    /**
     * Отхвърляне на ревю
     */
    public function reject_review() {
        try {
            // Verify nonce and check permissions
            check_ajax_referer('reviewxpress_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(array('message' => esc_html__('You do not have permission for this action.', 'reviewxpress')));
                return;
            }
            
            // Extract and sanitize only needed keys from $_POST (after nonce and capability check)
            $review_id = isset($_POST['review_id']) ? absint($_POST['review_id']) : 0;
            if ($review_id <= 0) {
                wp_send_json_error(esc_html__('Invalid review ID.', 'reviewxpress'));
                return;
            }
            
            $database = new ReviewXpress_Database();
            $result = $database->update_review_status($review_id, 'rejected');
            
            if ($result !== false) {
                wp_send_json_success(esc_html__('Review rejected.', 'reviewxpress'));
            } else {
                wp_send_json_error(esc_html__('An error occurred while rejecting the review.', 'reviewxpress'));
            }
        } catch (Exception $e) {
            wp_send_json_error(esc_html__('An error occurred while rejecting the review.', 'reviewxpress'));
        }
    }
    
    /**
     * Изтриване на ревю
     */
    public function delete_review() {
        try {
            // Verify nonce and check permissions
            check_ajax_referer('reviewxpress_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(array('message' => esc_html__('You do not have permission for this action.', 'reviewxpress')));
                return;
            }
            
            // Extract and sanitize only needed keys from $_POST (after nonce and capability check)
            $review_id = isset($_POST['review_id']) ? absint($_POST['review_id']) : 0;
            if ($review_id <= 0) {
                wp_send_json_error(esc_html__('Invalid review ID.', 'reviewxpress'));
                return;
            }
            
            $database = new ReviewXpress_Database();
            $result = $database->delete_review($review_id);
            
            if ($result !== false) {
                wp_send_json_success(esc_html__('Review deleted successfully.', 'reviewxpress'));
            } else {
                wp_send_json_error(esc_html__('An error occurred while deleting the review.', 'reviewxpress'));
            }
        } catch (Exception $e) {
            wp_send_json_error(esc_html__('Възникна грешка при изтриването на ревюто.', 'reviewxpress'));
        }
    }
    
    /**
     * Получаване на статистики
     */
    public function get_stats() {
        try {
            // Verify nonce and check permissions
            check_ajax_referer('reviewxpress_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(array('message' => esc_html__('You do not have permission for this action.', 'reviewxpress')));
                return;
            }
            
            $stats = $this->get_review_stats();
            
            if ($stats) {
                wp_send_json_success($stats);
            } else {
                wp_send_json_error(esc_html__('An error occurred while retrieving statistics.', 'reviewxpress'));
            }
        } catch (Exception $e) {
            wp_send_json_error(esc_html__('Възникна грешка при получаването на статистиките.', 'reviewxpress'));
        }
    }
    
    /**
     * Запазване на настройките (admin-post handler)
     */
    public function save_settings() {
        
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission for this action.', 'reviewxpress'));
        }
        
        if (!isset($_POST['reviewxpress_settings_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['reviewxpress_settings_nonce'])), 'reviewxpress_settings')) {
            wp_die(esc_html__('Invalid request.', 'reviewxpress'));
        }
        
        // Изчистваме кеша преди да заредим настройките
        wp_cache_delete('reviewxpress_settings', 'options');
        
        // Extract and sanitize only needed keys from $_POST (after nonce verification)
        $rx_section = isset($_POST['rx_section']) ? sanitize_text_field(wp_unslash($_POST['rx_section'])) : 'general';
        
        // Зареждаме текущите настройки
        $old_settings = get_option('reviewxpress_settings', array());
        $old_settings = is_array($old_settings) ? $old_settings : array();
        
        // Копираме старите настройки - използваме ги като база, не стойности по подразбиране
        // Това гарантира, че настройките не се нулират когато се променя само една секция
        $settings = $old_settings;
        
        // Добавяме стойности по подразбиране само ако липсват
        // Това гарантира, че всички настройки имат валидни стойности
        $defaults = array(
            // General settings
            'form_position' => 'after_reviews',
            'form_template' => 'default',
            'reviews_position' => 'after_description',
            'reviews_template' => 'default',
            'form_display_type' => 'position',
            'form_placement' => 'after_tab',
            'reviews_display_type' => 'woocommerce_tab',
            'show_form_in_reviews_section' => 0,
            'show_reviews_section' => 0,
            'show_reviews' => 0,
            'auto_approve' => 0,
            'review_mode' => 'disabled',
            'reviews_per_page' => 5,
            'form_page_url' => '',
            'content_moderation' => 'standard',
            'require_woocommerce_order' => 0,
            'allow_guest_reviews' => 0,
            'max_images' => 5,
            'max_image_size' => 5,
            'max_video_size' => 50,
            'allowed_image_types' => 'jpg,jpeg,png',
            // Email settings
            'email_from_name' => '',
            'email_from_address' => '',
            // Translations - labels
            'tr_label_name' => '',
            'tr_label_email' => '',
            'tr_label_rating' => '',
            'tr_label_review' => '',
            'tr_label_submit' => '',
            'tr_label_media' => '',
            'tr_label_avatar' => '',
            'tr_label_drag_image' => '',
            'tr_label_avatar_help' => '',
            'tr_label_email_help' => '',
            'tr_label_review_placeholder' => '',
            'tr_label_helpful' => '',
            'tr_label_load_more' => '',
            'tr_label_sort_by' => '',
            'tr_label_no_reviews' => '',
            'tr_label_rating_summary' => '',
            'tr_label_reviews_count' => '',
            'tr_label_anonymous' => '',
            // Translations - messages
            'tr_msg_success' => '',
            'tr_msg_error' => '',
            'tr_msg_required' => '',
            'tr_msg_duplicate' => '',
            'tr_msg_moderation' => '',
            'tr_msg_woocommerce_required' => '',
            'tr_msg_thank_you' => '',
            'tr_msg_review_submitted' => '',
            // Translations - statuses
            'tr_status_pending' => '',
            'tr_status_approved' => '',
            'tr_status_rejected' => '',
            // Design - form
            'form_background_color' => '#ffffff',
            'form_text_color' => '#111111',
            'form_border_color' => '#e5e7eb',
            'form_button_color' => '#ffd600',
            'form_button_text_color' => '#161616',
            'form_border_radius' => 8,
            // Design - reviews
            'reviews_background_color' => '#ffffff',
            'reviews_text_color' => '#111111',
            'reviews_border_color' => '#e5e7eb',
            'reviews_button_color' => '#ffd600',
            'reviews_button_text_color' => '#161616',
            'reviews_border_radius' => 8,
            // Shortcodes
            'default_avatar_id' => 0,
            'default_avatar_url' => '',
            'shortcode_button_color' => '#0073aa',
            'shortcode_background_color' => '#ffffff',
            'shortcode_text_color' => '#333333',
            'shortcode_border_color' => '#e5e7eb',
            'shortcode_border_radius' => 8,
            'shortcode_font_family' => 'default',
            // Integrations
            'page_builder' => 'default',
            'custom_css_enabled' => 0,
            'custom_css_class' => '',
            'custom_css_id' => ''
        );
        
        foreach ($defaults as $key => $default_value) {
            if (!isset($settings[$key])) {
                $settings[$key] = $default_value;
            }
        }

        if ($rx_section === 'general') {
            
            // Form display type - запазваме старата стойност ако не е изпратено
            if (isset($_POST['form_display_type'])) {
                $form_display_type_raw = sanitize_text_field(wp_unslash($_POST['form_display_type']));
                $valid_form_types = array('shortcode', 'position');
                if (in_array($form_display_type_raw, $valid_form_types, true)) {
                    $settings['form_display_type'] = $form_display_type_raw;
                } else {
                    $settings['form_display_type'] = isset($old_settings['form_display_type']) ? $old_settings['form_display_type'] : 'position';
                }
            } else {
                $settings['form_display_type'] = isset($old_settings['form_display_type']) ? $old_settings['form_display_type'] : 'position';
            }
            
            // Form placement - запазваме старата стойност ако не е изпратено
            if (isset($_POST['form_placement'])) {
                $form_placement_raw = sanitize_text_field(wp_unslash($_POST['form_placement']));
                $valid_placements = array('before_tab', 'after_tab', 'before_description', 'after_description', 'with_reviews');
                if (in_array($form_placement_raw, $valid_placements, true)) {
                    $settings['form_placement'] = $form_placement_raw;
                } else {
                    $settings['form_placement'] = isset($old_settings['form_placement']) ? $old_settings['form_placement'] : 'after_tab';
                }
            } else {
                $settings['form_placement'] = isset($old_settings['form_placement']) ? $old_settings['form_placement'] : 'after_tab';
            }
            
            
            // Review mode settings
            $review_mode = isset($_POST['review_mode']) ? sanitize_text_field(wp_unslash($_POST['review_mode'])) : 'disabled';
            
            // Ако show_reviews_section е включено и review_mode е disabled, автоматично активираме review_mode
            // Това е важно за Divi тема и други автоматични позиционирания
            if (isset($_POST['show_reviews_section']) && ($_POST['show_reviews_section'] === '1' || $_POST['show_reviews_section'] === 1)) {
                if ($review_mode === 'disabled') {
                    // Ако review_mode е disabled но show_reviews_section е включено, активираме manual_approve режим
                    $review_mode = 'manual_approve';
                }
            }
            
            $settings['review_mode'] = $review_mode; // Запазваме review_mode в настройките
            
            // Review mode processing
            
            switch ($review_mode) {
                case 'disabled':
                    $settings['auto_approve'] = 0;
                    $settings['show_reviews'] = 0;
                    break;
                case 'manual_approve':
                    $settings['auto_approve'] = 0;
                    $settings['show_reviews'] = 1;
                    break;
                case 'auto_approve':
                    $settings['auto_approve'] = 1;
                    $settings['show_reviews'] = 1;
                    break;
                case 'hidden_auto':
                    $settings['auto_approve'] = 1;
                    $settings['show_reviews'] = 0;
                    break;
                default:
                    $settings['auto_approve'] = 0;
                    $settings['show_reviews'] = 0;
            }
            
            
            // Review mode settings processed
            
            // Reviews display settings (moved to "Показване на ревюта" section)
            // Checkbox винаги се изпраща благодарение на JavaScript който добавя hidden поле
            // Ако checkbox-ът е избран, стойността ще бъде "1", иначе ще бъде "0" от hidden полето
            // Проверяваме дали има стойност в POST (може да е "1" от checkbox или "0" от hidden)
            if (isset($_POST['show_reviews_section'])) {
                $settings['show_reviews_section'] = ($_POST['show_reviews_section'] === '1' || $_POST['show_reviews_section'] === 1) ? 1 : 0;
            } else {
                // Ако по някаква причина не е изпратено, запазваме старата стойност
                $settings['show_reviews_section'] = isset($old_settings['show_reviews_section']) ? intval($old_settings['show_reviews_section']) : 0;
            }
            // Reviews display type - запазваме старата стойност ако не е изпратено
            if (isset($_POST['reviews_display_type'])) {
                $reviews_display_type_raw = sanitize_text_field(wp_unslash($_POST['reviews_display_type']));
                $valid_types = array('shortcode', 'position', 'woocommerce_tab');
                if (in_array($reviews_display_type_raw, $valid_types, true)) {
                    $settings['reviews_display_type'] = $reviews_display_type_raw;
                } else {
                    // Ако е невалидна стойност, запазваме старата
                    $settings['reviews_display_type'] = isset($old_settings['reviews_display_type']) ? $old_settings['reviews_display_type'] : 'woocommerce_tab';
                }
            } else {
                // Ако не е изпратено, запазваме старата стойност
                $settings['reviews_display_type'] = isset($old_settings['reviews_display_type']) ? $old_settings['reviews_display_type'] : 'woocommerce_tab';
            }
            
            // Reviews position - запазваме старата стойност ако не е изпратено
            if (isset($_POST['reviews_position'])) {
                $reviews_position_raw = sanitize_text_field(wp_unslash($_POST['reviews_position']));
                $valid_positions = array('before_tab', 'after_tab', 'before_description', 'after_description');
                if (in_array($reviews_position_raw, $valid_positions, true)) {
                    $settings['reviews_position'] = $reviews_position_raw;
                } else {
                    $settings['reviews_position'] = isset($old_settings['reviews_position']) ? $old_settings['reviews_position'] : 'after_description';
                }
            } else {
                $settings['reviews_position'] = isset($old_settings['reviews_position']) ? $old_settings['reviews_position'] : 'after_description';
            }
            
            // Reviews template - запазваме старата стойност ако не е изпратено
            if (isset($_POST['reviews_template'])) {
                $reviews_template_raw = sanitize_text_field(wp_unslash($_POST['reviews_template']));
                $valid_templates = array('default', 'side_by_side');
                if (in_array($reviews_template_raw, $valid_templates, true)) {
                    $settings['reviews_template'] = $reviews_template_raw;
                } else {
                    $settings['reviews_template'] = isset($old_settings['reviews_template']) ? $old_settings['reviews_template'] : 'default';
                }
            } else {
                $settings['reviews_template'] = isset($old_settings['reviews_template']) ? $old_settings['reviews_template'] : 'default';
            }
            
            // Reviews per page setting - запазваме старата стойност ако не е изпратено
            if (isset($_POST['reviews_per_page'])) {
                $reviews_per_page = absint($_POST['reviews_per_page']);
                $valid_per_page = array(3, 5, 10, 15, 20, 25);
                if (in_array($reviews_per_page, $valid_per_page, true)) {
                    $settings['reviews_per_page'] = $reviews_per_page;
                } else {
                    $settings['reviews_per_page'] = isset($old_settings['reviews_per_page']) ? intval($old_settings['reviews_per_page']) : 5;
                }
            } else {
                $settings['reviews_per_page'] = isset($old_settings['reviews_per_page']) ? intval($old_settings['reviews_per_page']) : 5;
            }
            
            // Reviews display settings processed
            
            // Form display settings (moved to "Форма за ревю" section)
            // Checkbox винаги се изпраща благодарение на JavaScript който добавя hidden поле
            if (isset($_POST['show_form_in_reviews_section'])) {
                $settings['show_form_in_reviews_section'] = ($_POST['show_form_in_reviews_section'] === '1' || $_POST['show_form_in_reviews_section'] === 1) ? 1 : 0;
            } else {
                // Ако по някаква причина не е изпратено, запазваме старата стойност
                $settings['show_form_in_reviews_section'] = isset($old_settings['show_form_in_reviews_section']) ? intval($old_settings['show_form_in_reviews_section']) : 0;
            }
            $settings['form_page_url'] = isset($_POST['form_page_url']) && !empty($_POST['form_page_url']) ? esc_url_raw(wp_unslash($_POST['form_page_url'])) : '';
            $content_moderation_raw = isset($_POST['content_moderation']) ? sanitize_text_field(wp_unslash($_POST['content_moderation'])) : 'standard';
            $settings['content_moderation'] = in_array($content_moderation_raw, array('standard','strict','none'), true) ? $content_moderation_raw : 'standard';
            
            // Купони за награди и автоматични имейли са в Про версията - не обработваме ги
            // $settings['reward_coupon_enabled'] = ... (Про версия)
            // $settings['reward_coupon_type'] = ... (Про версия)
            // $settings['reward_coupon_amount'] = ... (Про версия)
            // $settings['reward_coupon_days'] = ... (Про версия)
            // $settings['reward_coupon_prefix'] = ... (Про версия)
            // $settings['reward_coupon_delay_days'] = ... (Про версия)
            // $settings['email_delay_days'] = ... (Про версия)
            // $settings['auto_email_customer'] = ... (Про версия)
            
            // WooCommerce integration settings - checkbox винаги се изпраща благодарение на JavaScript
            if (isset($_POST['require_woocommerce_order'])) {
                $settings['require_woocommerce_order'] = ($_POST['require_woocommerce_order'] === '1' || $_POST['require_woocommerce_order'] === 1) ? 1 : 0;
            } else {
                $settings['require_woocommerce_order'] = isset($old_settings['require_woocommerce_order']) ? intval($old_settings['require_woocommerce_order']) : 0;
            }
            
            if (isset($_POST['allow_guest_reviews'])) {
                $settings['allow_guest_reviews'] = ($_POST['allow_guest_reviews'] === '1' || $_POST['allow_guest_reviews'] === 1) ? 1 : 0;
            } else {
                $settings['allow_guest_reviews'] = isset($old_settings['allow_guest_reviews']) ? intval($old_settings['allow_guest_reviews']) : 0;
            }
            $settings['max_images'] = isset($_POST['max_images']) ? absint($_POST['max_images']) : 5;
            $settings['max_image_size'] = isset($_POST['max_image_size']) ? absint($_POST['max_image_size']) : 5;
            $settings['max_video_size'] = isset($_POST['max_video_size']) ? absint($_POST['max_video_size']) : 50;
            $settings['allowed_image_types'] = isset($_POST['allowed_image_types']) ? sanitize_text_field(wp_unslash($_POST['allowed_image_types'])) : 'jpg,jpeg,png';
            
            // Настройките се обработват правилно
        } elseif ($rx_section === 'emails') {
            // Process email settings - handle empty values properly
            $settings['email_from_name'] = isset($_POST['email_from_name']) ? sanitize_text_field(wp_unslash($_POST['email_from_name'])) : '';
            $settings['email_from_address'] = isset($_POST['email_from_address']) ? sanitize_email(wp_unslash($_POST['email_from_address'])) : '';
            
            // SMTP и имейл шаблоните са в Про версията - не обработваме ги
            // $settings['email_coupon_subject'] = ... (Про версия)
            // $settings['email_coupon_body'] = ... (Про версия)
            // $settings['email_request_subject'] = ... (Про версия)
            // $settings['email_request_body'] = ... (Про версия)
            // $settings['smtp_enabled'] = ... (Про версия)
            // $settings['smtp_host'] = ... (Про версия)
            // $settings['smtp_port'] = ... (Про версия)
            // $settings['smtp_username'] = ... (Про версия)
            // $settings['smtp_password'] = ... (Про версия)
            // $settings['smtp_encryption'] = ... (Про версия)
            // Имейл настройките се обработват правилно
        } elseif ($rx_section === 'translations') {
            // Етикети за формата за ревю
            $settings['tr_label_name'] = isset($_POST['tr_label_name']) ? sanitize_text_field(wp_unslash($_POST['tr_label_name'])) : '';
            $settings['tr_label_email'] = isset($_POST['tr_label_email']) ? sanitize_text_field(wp_unslash($_POST['tr_label_email'])) : '';
            $settings['tr_label_rating'] = isset($_POST['tr_label_rating']) ? sanitize_text_field(wp_unslash($_POST['tr_label_rating'])) : '';
            $settings['tr_label_review'] = isset($_POST['tr_label_review']) ? sanitize_text_field(wp_unslash($_POST['tr_label_review'])) : '';
            $settings['tr_label_submit'] = isset($_POST['tr_label_submit']) ? sanitize_text_field(wp_unslash($_POST['tr_label_submit'])) : '';
            $settings['tr_label_media'] = isset($_POST['tr_label_media']) ? sanitize_text_field(wp_unslash($_POST['tr_label_media'])) : '';
            $settings['tr_label_avatar'] = isset($_POST['tr_label_avatar']) ? sanitize_text_field(wp_unslash($_POST['tr_label_avatar'])) : '';
            $settings['tr_label_drag_image'] = isset($_POST['tr_label_drag_image']) ? sanitize_text_field(wp_unslash($_POST['tr_label_drag_image'])) : '';
            $settings['tr_label_avatar_help'] = isset($_POST['tr_label_avatar_help']) ? sanitize_text_field(wp_unslash($_POST['tr_label_avatar_help'])) : '';
            $settings['tr_label_email_help'] = isset($_POST['tr_label_email_help']) ? sanitize_text_field(wp_unslash($_POST['tr_label_email_help'])) : '';
            $settings['tr_label_review_placeholder'] = isset($_POST['tr_label_review_placeholder']) ? sanitize_text_field(wp_unslash($_POST['tr_label_review_placeholder'])) : '';
            
            // Етикети за списъка с ревюта
            $settings['tr_label_helpful'] = isset($_POST['tr_label_helpful']) ? sanitize_text_field(wp_unslash($_POST['tr_label_helpful'])) : '';
            $settings['tr_label_load_more'] = isset($_POST['tr_label_load_more']) ? sanitize_text_field(wp_unslash($_POST['tr_label_load_more'])) : '';
            $settings['tr_label_sort_by'] = isset($_POST['tr_label_sort_by']) ? sanitize_text_field(wp_unslash($_POST['tr_label_sort_by'])) : '';
            $settings['tr_label_no_reviews'] = isset($_POST['tr_label_no_reviews']) ? sanitize_text_field(wp_unslash($_POST['tr_label_no_reviews'])) : '';
            $settings['tr_label_rating_summary'] = isset($_POST['tr_label_rating_summary']) ? sanitize_text_field(wp_unslash($_POST['tr_label_rating_summary'])) : '';
            $settings['tr_label_reviews_count'] = isset($_POST['tr_label_reviews_count']) ? sanitize_text_field(wp_unslash($_POST['tr_label_reviews_count'])) : '';
            
            // Съобщения и уведомления
            $settings['tr_msg_success'] = isset($_POST['tr_msg_success']) ? sanitize_text_field(wp_unslash($_POST['tr_msg_success'])) : '';
            $settings['tr_msg_error'] = isset($_POST['tr_msg_error']) ? sanitize_text_field(wp_unslash($_POST['tr_msg_error'])) : '';
            $settings['tr_msg_required'] = isset($_POST['tr_msg_required']) ? sanitize_text_field(wp_unslash($_POST['tr_msg_required'])) : '';
            $settings['tr_msg_duplicate'] = isset($_POST['tr_msg_duplicate']) ? sanitize_text_field(wp_unslash($_POST['tr_msg_duplicate'])) : '';
            $settings['tr_msg_moderation'] = isset($_POST['tr_msg_moderation']) ? sanitize_text_field(wp_unslash($_POST['tr_msg_moderation'])) : '';
            $settings['tr_msg_woocommerce_required'] = isset($_POST['tr_msg_woocommerce_required']) ? sanitize_text_field(wp_unslash($_POST['tr_msg_woocommerce_required'])) : '';
            $settings['tr_msg_thank_you'] = isset($_POST['tr_msg_thank_you']) ? sanitize_text_field(wp_unslash($_POST['tr_msg_thank_you'])) : '';
            $settings['tr_msg_review_submitted'] = isset($_POST['tr_msg_review_submitted']) ? sanitize_text_field(wp_unslash($_POST['tr_msg_review_submitted'])) : '';
            
            // Статуси на ревюта
            $settings['tr_status_pending'] = isset($_POST['tr_status_pending']) ? sanitize_text_field(wp_unslash($_POST['tr_status_pending'])) : '';
            $settings['tr_status_approved'] = isset($_POST['tr_status_approved']) ? sanitize_text_field(wp_unslash($_POST['tr_status_approved'])) : '';
            $settings['tr_status_rejected'] = isset($_POST['tr_status_rejected']) ? sanitize_text_field(wp_unslash($_POST['tr_status_rejected'])) : '';
            $settings['tr_label_anonymous'] = isset($_POST['tr_label_anonymous']) ? sanitize_text_field(wp_unslash($_POST['tr_label_anonymous'])) : '';
            
            // Дизайн настройки за формата за изпращане
            $form_bg = isset($_POST['form_background_color']) ? sanitize_hex_color(wp_unslash($_POST['form_background_color'])) : '#ffffff';
            $settings['form_background_color'] = $form_bg ? $form_bg : '#ffffff';
            
            $form_text = isset($_POST['form_text_color']) ? sanitize_hex_color(wp_unslash($_POST['form_text_color'])) : '#111111';
            $settings['form_text_color'] = $form_text ? $form_text : '#111111';
            
            $form_border = isset($_POST['form_border_color']) ? sanitize_hex_color(wp_unslash($_POST['form_border_color'])) : '#e5e7eb';
            $settings['form_border_color'] = $form_border ? $form_border : '#e5e7eb';
            
            $form_btn = isset($_POST['form_button_color']) ? sanitize_hex_color(wp_unslash($_POST['form_button_color'])) : '#ffd600';
            $settings['form_button_color'] = $form_btn ? $form_btn : '#ffd600';
            
            $form_btn_text = isset($_POST['form_button_text_color']) ? sanitize_hex_color(wp_unslash($_POST['form_button_text_color'])) : '#161616';
            $settings['form_button_text_color'] = $form_btn_text ? $form_btn_text : '#161616';
            
            $settings['form_border_radius'] = isset($_POST['form_border_radius']) ? absint($_POST['form_border_radius']) : 8;
            if ($settings['form_border_radius'] > 50) {
                $settings['form_border_radius'] = 50;
            }
            
            // Дизайн настройки за формата за показване на ревютата
            $rev_bg = isset($_POST['reviews_background_color']) ? sanitize_hex_color(wp_unslash($_POST['reviews_background_color'])) : '#ffffff';
            $settings['reviews_background_color'] = $rev_bg ? $rev_bg : '#ffffff';
            
            $rev_text = isset($_POST['reviews_text_color']) ? sanitize_hex_color(wp_unslash($_POST['reviews_text_color'])) : '#111111';
            $settings['reviews_text_color'] = $rev_text ? $rev_text : '#111111';
            
            $rev_border = isset($_POST['reviews_border_color']) ? sanitize_hex_color(wp_unslash($_POST['reviews_border_color'])) : '#e5e7eb';
            $settings['reviews_border_color'] = $rev_border ? $rev_border : '#e5e7eb';
            
            $rev_btn = isset($_POST['reviews_button_color']) ? sanitize_hex_color(wp_unslash($_POST['reviews_button_color'])) : '#ffd600';
            $settings['reviews_button_color'] = $rev_btn ? $rev_btn : '#ffd600';
            
            $rev_btn_text = isset($_POST['reviews_button_text_color']) ? sanitize_hex_color(wp_unslash($_POST['reviews_button_text_color'])) : '#161616';
            $settings['reviews_button_text_color'] = $rev_btn_text ? $rev_btn_text : '#161616';
            
            $settings['reviews_border_radius'] = isset($_POST['reviews_border_radius']) ? absint($_POST['reviews_border_radius']) : 8;
            if ($settings['reviews_border_radius'] > 50) {
                $settings['reviews_border_radius'] = 50;
            }
            
            // Дизайн на имейли е в Про версията - не обработваме настройките
            // $settings['email_background_color'] = ... (Про версия)
            // $settings['email_text_color'] = ... (Про версия)
            // $settings['email_header_color'] = ... (Про версия)
            // $settings['email_button_color'] = ... (Про версия)
            // $settings['email_button_text_color'] = ... (Про версия)
            // $settings['email_font_family'] = ... (Про версия)
            
            // Преводните и дизайн настройките се обработват правилно
        } elseif ($rx_section === 'defaults' || $rx_section === 'shortcodes') {
            $settings['default_avatar_id'] = isset($_POST['default_avatar_id']) ? absint($_POST['default_avatar_id']) : 0;
            $settings['default_avatar_url'] = isset($_POST['default_avatar_url']) ? esc_url_raw(wp_unslash($_POST['default_avatar_url'])) : '';
            
            // Стилове за шорткодове
            $sc_btn = isset($_POST['shortcode_button_color']) ? sanitize_hex_color(wp_unslash($_POST['shortcode_button_color'])) : '#0073aa';
            $settings['shortcode_button_color'] = $sc_btn ? $sc_btn : '#0073aa';
            
            $sc_bg = isset($_POST['shortcode_background_color']) ? sanitize_hex_color(wp_unslash($_POST['shortcode_background_color'])) : '#ffffff';
            $settings['shortcode_background_color'] = $sc_bg ? $sc_bg : '#ffffff';
            
            $sc_text = isset($_POST['shortcode_text_color']) ? sanitize_hex_color(wp_unslash($_POST['shortcode_text_color'])) : '#333333';
            $settings['shortcode_text_color'] = $sc_text ? $sc_text : '#333333';
            
            $sc_border = isset($_POST['shortcode_border_color']) ? sanitize_hex_color(wp_unslash($_POST['shortcode_border_color'])) : '#e5e7eb';
            $settings['shortcode_border_color'] = $sc_border ? $sc_border : '#e5e7eb';
            
            $settings['shortcode_border_radius'] = isset($_POST['shortcode_border_radius']) ? absint($_POST['shortcode_border_radius']) : 8;
            if ($settings['shortcode_border_radius'] > 50) {
                $settings['shortcode_border_radius'] = 50;
            }
            
            $settings['shortcode_font_family'] = isset($_POST['shortcode_font_family']) ? sanitize_text_field(wp_unslash($_POST['shortcode_font_family'])) : 'default';
            // Шорткод настройките се обработват правилно
        }

        // Интеграционни настройки (винаги се обработват)
        // Важно: page_builder трябва да се запазва правилно, дори когато не е изпратено в POST
        if (isset($_POST['page_builder'])) {
            $page_builder_raw = sanitize_text_field(wp_unslash($_POST['page_builder']));
            $valid_page_builders = array('default', 'divi', 'elementor', 'flatsome', 'beaver', 'avada', 'storefront', 'astra', 'generatepress', 'oceanwp');
            $settings['page_builder'] = in_array($page_builder_raw, $valid_page_builders, true) ? $page_builder_raw : 'default';
        } else {
            // Ако не е изпратено, запазваме старата стойност (не нулираме)
            $settings['page_builder'] = isset($old_settings['page_builder']) && !empty($old_settings['page_builder']) ? $old_settings['page_builder'] : 'default';
        }
        // Custom CSS enabled - checkbox винаги се изпраща благодарение на JavaScript
        if (isset($_POST['custom_css_enabled'])) {
            $settings['custom_css_enabled'] = ($_POST['custom_css_enabled'] === '1' || $_POST['custom_css_enabled'] === 1) ? 1 : 0;
        } else {
            $settings['custom_css_enabled'] = isset($old_settings['custom_css_enabled']) ? intval($old_settings['custom_css_enabled']) : 0;
        }
        $settings['custom_css_class'] = isset($_POST['custom_css_class']) ? sanitize_text_field(wp_unslash($_POST['custom_css_class'])) : '';
        $settings['custom_css_id'] = isset($_POST['custom_css_id']) ? sanitize_text_field(wp_unslash($_POST['custom_css_id'])) : '';
        // Custom CSS code functionality is available in Pro version only

        // Нормализираме checkbox стойностите за правилно сравнение (int вместо string)
        // Това гарантира, че checkbox полетата винаги са 0 или 1, независимо от секцията
        $checkbox_fields = array(
            'show_reviews_section',
            'show_form_in_reviews_section',
            'show_reviews',
            'auto_approve',
            'require_woocommerce_order',
            'allow_guest_reviews',
            'custom_css_enabled'
        );
        
        foreach ($checkbox_fields as $field) {
            // Нормализираме и новите и старите настройки
            $settings[$field] = !empty($settings[$field]) ? 1 : 0;
            $old_settings[$field] = !empty($old_settings[$field]) ? 1 : 0;
        }
        
        // Финален преглед - гарантираме, че всички настройки от defaults са налични
        // Това предотвратява проблеми когато се запаметява само една секция
        // и някои настройки не са обработени
        // Важно: Не презаписваме checkbox полетата и page_builder които вече са обработени
        $processed_checkboxes = array('show_reviews_section', 'show_form_in_reviews_section', 'show_reviews', 'auto_approve', 'require_woocommerce_order', 'allow_guest_reviews', 'custom_css_enabled');
        $processed_fields = array_merge($processed_checkboxes, array('page_builder', 'reviews_display_type', 'form_display_type', 'form_placement', 'reviews_position', 'reviews_template'));
        
        foreach ($defaults as $key => $default_value) {
            // Пропускаме полетата които вече са обработени
            if (in_array($key, $processed_fields, true) && isset($settings[$key])) {
                continue;
            }
            
            if (!isset($settings[$key]) || $settings[$key] === '') {
                // Ако настройката липсва или е празна, използваме default или старата стойност
                if (isset($old_settings[$key]) && $old_settings[$key] !== '') {
                    $settings[$key] = $old_settings[$key];
                } else {
                    $settings[$key] = $default_value;
                }
            }
        }
        
        // Сравняваме със старите настройки, за да отчетем реална промяна
        $changed = serialize($settings) !== serialize($old_settings);
        
        // Запазване на настройките
        $ok = true;
        
        if ($changed) {
            try {
                // Изчистваме кеша преди запис и след запис
                wp_cache_delete('reviewxpress_settings', 'options');
                wp_cache_flush();
                
                $option_exists = get_option('reviewxpress_settings') !== false;
                
                if ($option_exists) {
                    // Опитваме се с update_option
                    // Използваме autoload = 'no' за да не се кешира автоматично
                    $result = update_option('reviewxpress_settings', $settings, 'no');
                    
                    // ВИНАГИ изчистваме кеша след update_option, дори и да върне TRUE
                    wp_cache_delete('reviewxpress_settings', 'options');
                    wp_cache_flush();
                    
                    // Проверяваме дали настройките наистина са запазени
                    $verify_settings = get_option('reviewxpress_settings', array());
                    if (empty($verify_settings) || !isset($verify_settings['page_builder'])) {
                        $result = false; // Форсираме директно DB обновяване
                    }
                    
                    // Ако update_option върне false или верификацията не успее, опитваме се директно с базата данни
                    if ($result === false) {
                        global $wpdb;
                        
                        // Сериализираме данните
                        $serialized = maybe_serialize($settings);
                        
                        // Опитваме се директно да обновим в базата данни
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $db_result = $wpdb->update(
                            $wpdb->options,
                            array('option_value' => $serialized),
                            array('option_name' => 'reviewxpress_settings'),
                            array('%s'),
                            array('%s')
                        );
                        
                        if ($db_result === false) {
                            // Проверяваме дали има реална грешка или просто няма промяна
                            wp_cache_delete('reviewxpress_settings', 'options');
                            $saved_settings = get_option('reviewxpress_settings', array());
                            
                            // Ако данните все още са различни, има грешка
                            if (serialize($saved_settings) !== serialize($settings)) {
                                $ok = false;
                            }
                            // Ако са еднакви, значи няма проблем
                        } else {
                            // Успешно обновяване директно в базата
                            wp_cache_delete('reviewxpress_settings', 'options');
                        }
                    }
                } else {
                    // Опитваме се с add_option
                    $result = add_option('reviewxpress_settings', $settings, '', 'no');
                    
                    if ($result === false) {
                        // Ако add_option не работи, опитваме се директно
                        global $wpdb;
                        $serialized = maybe_serialize($settings);
                        
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $db_result = $wpdb->insert(
                            $wpdb->options,
                            array(
                                'option_name' => 'reviewxpress_settings',
                                'option_value' => $serialized,
                                'autoload' => 'no'
                            ),
                            array('%s', '%s', '%s')
                        );
                        
                        if ($db_result === false) {
                            $ok = false;
                        } else {
                            wp_cache_delete('reviewxpress_settings', 'options');
                        }
                    }
                }
                
                // Изчистваме кеша след запис
                wp_cache_delete('reviewxpress_settings', 'options');
                
            } catch (Exception $e) {
                // Грешка при запис
                $ok = false;
            } catch (Error $e) {
                // PHP 7+ Fatal errors
                $ok = false;
            }
        }
        
        // Изчистваме кеша след запазване за да се уверя че новите настройки се зареждат правилно
        wp_cache_delete('reviewxpress_settings', 'options');
        wp_cache_flush();

        // Redirect обратно (PRG), с флаг за съобщение
        $url = add_query_arg(array(
            'page' => 'reviewxpress-settings-' . $rx_section,
            'rx_saved' => $ok ? '1' : '0',
            'rx_changed' => $changed ? '1' : '0',
            '_wpnonce' => wp_create_nonce('reviewxpress_settings_nonce')
        ), admin_url('admin.php'));
        
        wp_safe_redirect($url);
        exit;
    }
    
    /**
     * Показване на admin notices след redirect
     */
    public function show_admin_notices() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // If rx_saved or rx_changed is provided, nonce must be present and valid
        if (!isset($_GET['rx_saved']) && !isset($_GET['rx_changed'])) {
            return;
        }
        
        // Verify nonce - if missing or invalid, ignore notices
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'reviewxpress_settings_nonce')) {
            return;
        }
        
        // Now safe to read $_GET parameters (nonce verified)
        $changed = isset($_GET['rx_changed']) && sanitize_key(wp_unslash($_GET['rx_changed'])) === '1';
        $saved = isset($_GET['rx_saved']) && sanitize_key(wp_unslash($_GET['rx_saved'])) === '1';
        
        if ($saved) {
            if ($changed) {
                echo '<div class="notice notice-success is-dismissible"><p><strong>' . esc_html__('Success!', 'reviewxpress') . '</strong> ' . esc_html__('Settings saved successfully!', 'reviewxpress') . '</p></div>';
            } else {
                echo '<div class="notice notice-info is-dismissible"><p><strong>' . esc_html__('Info:', 'reviewxpress') . '</strong> ' . esc_html__('No changes to save.', 'reviewxpress') . '</p></div>';
            }
        } else {
            echo '<div class="notice notice-error is-dismissible"><p><strong>' . esc_html__('Error!', 'reviewxpress') . '</strong> ' . esc_html__('An error occurred while saving settings.', 'reviewxpress') . '</p></div>';
        }
    }
    
    /**
     * Експорт на ревюта в CSV
     */
    public function export_reviews() {
        // Verify nonce and check permissions
        check_ajax_referer('reviewxpress_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('You do not have permission for this operation.', 'reviewxpress')));
            return;
        }
        
        global $wpdb;
        
        // Extract and sanitize only needed keys from $_POST (after nonce and capability check)
        $status = isset($_POST['status']) ? sanitize_text_field(wp_unslash($_POST['status'])) : 'all';
        $product_id = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        $search = isset($_POST['search']) ? sanitize_text_field(wp_unslash($_POST['search'])) : '';
        
        // Build SQL with placeholders and always use prepare()
        $reviews_table = $wpdb->prefix . 'reviewxpress_reviews';
        $reviews_table_esc = esc_sql($reviews_table);
        $sql = "SELECT * FROM `{$reviews_table_esc}` WHERE 1=%d";
        $sql_args = array(1);

        if ($status !== 'all') {
            $sql .= " AND status = %s";
            $sql_args[] = $status;
        }

        if ($product_id > 0) {
            $sql .= " AND product_id = %d";
            $sql_args[] = $product_id;
        }

        if (!empty($search)) {
            $like = '%' . $wpdb->esc_like($search) . '%';
            $sql .= " AND (name LIKE %s OR email LIKE %s OR review_text LIKE %s)";
            $sql_args[] = $like;
            $sql_args[] = $like;
            $sql_args[] = $like;
        }

        $sql .= " ORDER BY created_at DESC";

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPlaceholders, WordPress.DB.PreparedSQL.NotPrepared
        $reviews = $wpdb->get_results($wpdb->prepare($sql, $sql_args));
        
        // Stream CSV directly (avoid writing PII to disk)
        nocache_headers();
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=reviewxpress-export-' . gmdate('Y-m-d-H-i-s') . '.csv');
        
        $out = fopen('php://output', 'w');
        fputcsv($out, array('ID','Product ID','Product Name','Author Name','Email','Rating','Review Text','Status','Created At'));
        
        foreach ($reviews as $review) {
            $product_name = get_the_title($review->product_id);
            $product_name = $product_name !== '' ? $product_name : 'N/A';
            
            fputcsv($out, array(
                (int) $review->id,
                (int) $review->product_id,
                (string) $product_name,
                (string) $review->name,
                (string) $review->email,
                (int) $review->rating,
                (string) $review->review_text,
                (string) $review->status,
                (string) $review->created_at
            ));
        }
        
        // php://output doesn't need to be closed
        wp_die();
    }
}

