jQuery(document).ready(function($) {
    'use strict';

    // Проверка дали е инициализиран
    if (window.reviewxpressInitialized) {
        return;
    }
    window.reviewxpressInitialized = true;

    // Инициализация на формата за ревю
    function initReviewForm() {
        const $form = $('.rx-review-form');
        if ($form.length === 0 || $form.data('rx-initialized')) {
            return;
        }
        $form.data('rx-initialized', true);

        // Инициализация на drag & drop за файлове
        initAvatarUpload();
        initImagesUpload();
        initVideoUpload();
        
        // Инициализация на медия секцията
        initMediaSection();
        
        // Инициализация на звездите за рейтинг
        initRatingStars();
        
        // Обработка на изпращане на формата
        handleFormSubmit();
    }

    // Аватар качване и превю
    function initAvatarUpload() {
        const $area = $('#rx-avatar-upload');
        const $input = $('#rx-avatar');
        const $preview = $('#rx-uploaded-avatar');
        const $placeholder = $area.find('.rx-upload-placeholder');

        if ($input.length === 0) return;

        // Премахваме стари/външни бутони за премахване, ако съществуват
        $('.rx-remove-avatar').remove();

        $area.on('click', function(e) {
            if ($(e.target).is('input[type="file"]')) return;
            e.preventDefault();
            e.stopImmediatePropagation();
            if ($input[0]) $input[0].click();
        });

        $area.on('dragover', function(e) { e.preventDefault(); $area.addClass('rx-drag-over'); });
        $area.on('dragleave', function(e) { e.preventDefault(); $area.removeClass('rx-drag-over'); });
        $area.on('drop', function(e) {
            e.preventDefault();
            $area.removeClass('rx-drag-over');
            const files = e.originalEvent.dataTransfer.files;
            if (files && files.length) {
                $input[0].files = files;
                renderAvatarPreview(files[0]);
            }
        });

        $input.on('change', async function() {
            const file = this.files && this.files[0];
            if (!file) return;
            const resized = await resizeImageToCanvasBlob(file, 200, 200, 0.9);
            if (resized && resized.blob) {
                const resizedFile = new File([resized.blob], file.name, { type: 'image/jpeg' });
                const dt = new DataTransfer();
                dt.items.add(resizedFile);
                this.files = dt.files;
                renderAvatarPreview(resizedFile);
            } else {
                renderAvatarPreview(file);
            }
        });

        function renderAvatarPreview(file) {
            if (!file.type.startsWith('image/')) {
                showMessage(reviewxpress_ajax.msgs?.avatar_invalid || 'Please select an image for avatar.', 'error');
                return;
            }
            
            const maxSize = 2 * 1024 * 1024; // 2MB
            if (file.size > maxSize) {
                showMessage(reviewxpress_ajax.msgs?.avatar_too_large || 'Avatar is too large. Maximum size: 2MB.', 'error');
                return;
            }
            
            const reader = new FileReader();
            reader.onload = function(e) {
                // Показваме само снимката и съобщението в preview
                $preview.html('<div class="rx-avatar-preview"><img src="' + e.target.result + '" alt="avatar" class="rx-avatar-preview-img"/><div class="rx-avatar-limit-message">✓ Profile picture added</div></div>');
                $placeholder.hide();
                // Премахваме евентуални външни бутони и показваме само един бутон за премахване в upload полето
                $('.rx-remove-avatar').remove();
                $area.append('<button type="button" class="rx-remove-avatar-in-field">🗑️ Remove avatar</button>');
                // Компактен режим на полето след качване
                $area.addClass('rx-has-avatar');
            };
            reader.readAsDataURL(file);
        }

        // Обработка за премахване на аватар
        $(document).on('click', '.rx-remove-avatar, .rx-remove-avatar-in-field', function() {
            $preview.empty();
            $input.val('');
            $placeholder.show().html('<span class="rx-upload-text">+ Upload profile picture</span>');
            // Премахваме бутона от полето
            $area.find('.rx-remove-avatar-in-field').remove();
            // Връщаме нормален режим и височина на полето
            $area.removeClass('rx-has-avatar');
        });
    }

    // Снимки качване и превю (мулти)
    function initImagesUpload() {
        const $area = $('#rx-image-upload');
        const $input = $('#rx-images');
        const $preview = $('#rx-uploaded-images');
        const $placeholder = $area.find('.rx-upload-placeholder');
        const maxImages = parseInt($area.data('max-images')) || 5;
        if ($input.length === 0) return;

        $area.on('click', function(e) { if ($(e.target).is('input[type="file"]')) return; e.preventDefault(); if ($input[0]) $input[0].click(); });
        $area.on('dragover', function(e) { e.preventDefault(); $area.addClass('rx-drag-over'); });
        $area.on('dragleave', function(e) { e.preventDefault(); $area.removeClass('rx-drag-over'); });
        $area.on('drop', function(e) {
            e.preventDefault();
            $area.removeClass('rx-drag-over');
            const files = e.originalEvent.dataTransfer.files;
            if (files && files.length) {
                appendImages(files);
            }
        });

        $input.on('change', function() { appendImages(this.files); this.value=''; });

        function getCurrentCount() { return $preview.find('.rx-uploaded-image').length; }

        function appendImages(files) {
            if (!files || !files.length) return;
            let current = getCurrentCount();
            const maxSize = 5 * 1024 * 1024; // 5MB
            
            Array.from(files).forEach(function(file) {
                if (current >= maxImages) {
                    showMessage('Maximum number of images reached (' + maxImages + ').', 'error');
                    return;
                }
                if (!file.type.startsWith('image/')) {
                    showMessage('Please select images only.', 'error');
                    return;
                }
                if (file.size > maxSize) {
                    showMessage('File "' + file.name + '" is too large. Maximum size: 5MB.', 'error');
                    return;
                }
                
                const reader = new FileReader();
                reader.onload = function(e) {
                    $preview.append('<div class="rx-uploaded-image"><img src="' + e.target.result + '"/><button type="button" class="rx-remove-image">×</button></div>');
                };
                reader.readAsDataURL(file);
                current++;
            });
            
            // Показваме placeholder с "добави още" ако не сме достигнали лимита
            if (getCurrentCount() < maxImages) {
                $placeholder.show().html('<span class="rx-add-more-text">+ Add more images</span>');
            } else {
                $placeholder.hide();
            }
        }

        $(document).on('click', '.rx-remove-image', function() {
            $(this).parent().remove();
            // Показваме "добави още" ако не сме достигнали лимита
            if (getCurrentCount() < maxImages) {
                $placeholder.show().html('<span class="rx-add-more-text">+ Add more images</span>');
            }
        });
    }

    // Видео качване и превю (име)
    function initVideoUpload() {
        const $area = $('#rx-video-upload');
        const $input = $('#rx-video');
        const $preview = $('#rx-uploaded-video');
        const $placeholder = $area.find('.rx-upload-placeholder');
        if ($input.length === 0) return;

        $area.on('click', function(e) { if ($(e.target).is('input[type="file"]')) return; e.preventDefault(); if ($input[0]) $input[0].click(); });
        $area.on('dragover', function(e) { e.preventDefault(); $area.addClass('rx-drag-over'); });
        $area.on('dragleave', function(e) { e.preventDefault(); $area.removeClass('rx-drag-over'); });
        $area.on('drop', function(e) {
            e.preventDefault();
            $area.removeClass('rx-drag-over');
            const files = e.originalEvent.dataTransfer.files;
            if (files && files.length) {
                $input[0].files = files;
                renderVideo(files[0]);
            }
        });

        $input.on('change', function() { const f = this.files && this.files[0]; if (f) renderVideo(f); });

        function renderVideo(file) {
            const maxSize = 50 * 1024 * 1024; // 50MB
            const allowedTypes = ['video/mp4', 'video/avi', 'video/mov'];
            
            if (!allowedTypes.includes(file.type)) {
                showMessage('Please select a video file in MP4, AVI or MOV format.', 'error');
                return;
            }
            
            if (file.size > maxSize) {
                showMessage('Video file is too large. Maximum size: 50MB.', 'error');
                return;
            }
            
            $preview.html('<div class="rx-uploaded-video-item"><span>' + file.name + '</span><button type="button" class="rx-remove-video">×</button></div>');
            $placeholder.hide();
        }

        $(document).on('click', '.rx-remove-video', function() {
            $(this).parent().remove();
            $input.val('');
        });
    }

    // Инициализация на медия секцията (toggle и табове)
    function initMediaSection() {
        // Toggle функционалност
        $('.rx-toggle-btn').on('click', function() {
            const $btn = $(this);
            const target = $btn.data('target');
            const $content = $('#' + target);
            
            if ($content.is(':visible')) {
                $content.slideUp(300);
                $btn.removeClass('active');
            } else {
                $content.slideDown(300);
                $btn.addClass('active');
            }
        });
        
        // Таб функционалност
        $('.rx-tab-btn').on('click', function() {
            const $btn = $(this);
            const tabName = $btn.data('tab');
            
            // Премахваме активния клас от всички табове и панели
            $('.rx-tab-btn').removeClass('active');
            $('.rx-tab-pane').removeClass('active');
            
            // Добавяме активния клас към избрания таб и панел
            $btn.addClass('active');
            $('#rx-tab-' + tabName).addClass('active');
        });
    }

    // Инициализация на звездите за рейтинг
    function initRatingStars() {
        const $wrap = $('.rx-rating-input');
        if ($wrap.length === 0) return;

        const $radios = $wrap.find('input[type="radio"][name="rating"]');
        const $labels = $wrap.find('label.rx-star-label');

        function setVisual(val) {
            const rating = parseInt(val) || 0;
            $labels.removeClass('rx-star-selected rx-star-hover');
            $labels.each(function(index){
                if (index + 1 <= rating) $(this).addClass('rx-star-selected');
            });
        }

        $labels.on('click', function(e){
            e.preventDefault();
            const forId = $(this).attr('for');
            const $radio = $('#' + forId);
            if ($radio.length) {
                $radio.prop('checked', true);
                setVisual($radio.val());
            }
        });

        $labels.on('mouseenter', function(){
            const forId = $(this).attr('for');
            const $radio = $('#' + forId);
            setVisual($radio.val());
        });
        $wrap.on('mouseleave', function(){
            const $checked = $radios.filter(':checked');
            setVisual($checked.val());
        });

        const $checked = $radios.filter(':checked');
        setVisual($checked.val());
    }

    // Обработка на изпращане на формата
    function handleFormSubmit() {
        // Премахваме старите event handlers за да избегнем множествени handlers
        $('.rx-review-form, .reviewxpress-form').off('submit.reviewxpress');
        
        // Прикачваме нов handler с namespace - само за нашите форми
        $('.rx-review-form, .reviewxpress-form').on('submit.reviewxpress', function(e) {
            // Проверка дали това е нашата форма
            const $form = $(this);
            if (!$form.hasClass('rx-review-form') && !$form.hasClass('reviewxpress-form')) {
                return; // Не е нашата форма, не правим нищо
            }
            
            // Проверка дали има задължителни полета на нашата форма
            if ($form.find('input[name="name"]').length === 0 || 
                $form.find('input[name="email"]').length === 0 ||
                $form.find('input[name="rating"]').length === 0) {
                return; // Не е нашата форма
            }
            
            e.preventDefault();
            e.stopImmediatePropagation(); // Спираме други handlers
            e.stopPropagation(); // Спираме bubbling
            const $submitBtn = $form.find('.rx-submit-btn');
            const $btnText = $submitBtn.find('.rx-btn-text');
            const originalText = $btnText.length ? $btnText.text() : $submitBtn.text();
            
            // Валидация на задължителни полета
            const name = $form.find('input[name="name"]').val().trim();
            const email = $form.find('input[name="email"]').val().trim();
            const ratingChecked = $form.find('input[name="rating"]:checked').length > 0;
            const reviewText = $form.find('textarea[name="review_text"]').val().trim();
            
            if (!name) {
                showMessage(reviewxpress_ajax.messages.name_required || 'Please enter your name.', 'error');
                $form.find('input[name="name"]').focus();
                return;
            }
            
            if (!email) {
                showMessage(reviewxpress_ajax.messages.email_required || 'Please enter your email address.', 'error');
                $form.find('input[name="email"]').focus();
                return;
            }
            
            // Валидация на имейл формат
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(email)) {
                showMessage(reviewxpress_ajax.messages.email_invalid || 'Please enter a valid email address.', 'error');
                $form.find('input[name="email"]').focus();
                return;
            }
            
            if (!ratingChecked) {
                showMessage(reviewxpress_ajax.messages.rating_required || 'Please select a rating.', 'error');
                // Скрол до рейтинга
                const $ratingWrap = $form.find('.rx-rating-input');
                if ($ratingWrap.length && $ratingWrap.offset()) {
                    $('html, body').animate({ scrollTop: $ratingWrap.offset().top - 100 }, 300);
                }
                return;
            }
            
            if (!reviewText) {
                showMessage(reviewxpress_ajax.messages.review_required || 'Please write a review.', 'error');
                $form.find('textarea[name="review_text"]').focus();
                return;
            }
            
            // Показване на loading състояние
            $submitBtn.prop('disabled', true);
            if ($btnText.length) {
                $btnText.text('Submitting...');
            } else {
                $submitBtn.text('Submitting...');
            }
            
            // Валидация на YouTube URL
            const videoUrl = $form.find('input[name="video_url"]').val().trim();
            if (videoUrl) {
                const youtubeRegex = /^(https?:\/\/)?(www\.)?(youtube\.com\/watch\?v=|youtu\.be\/)[\w-]+/;
                if (!youtubeRegex.test(videoUrl)) {
                    showMessage(reviewxpress_ajax.messages.youtube_invalid || 'Please enter a valid YouTube URL.', 'error');
                    $form.find('input[name="video_url"]').focus();
                    return;
                }
            }
            
            // Подготовка на FormData
            const formData = new FormData(this);
            
            // Добавяме всички показани снимки като файлове (dataURL -> Blob)
            const images = [];
            $form.find('#rx-uploaded-images .rx-uploaded-image img').each(function(){ 
                images.push($(this).attr('src')); 
            });
            
            // Подготвяме масиви за $_FILES структура
            const imageNames = [];
            const imageTypes = [];
            const imageTmpNames = [];
            const imageErrors = [];
            const imageSizes = [];
            
            images.forEach(function(src, idx){
                const blob = dataURLtoBlob(src);
                if (blob) {
                    // Определяме типа на изображението от data URL
                    let mimeType = 'image/jpeg';
                    if (src.indexOf('data:') === 0) {
                        const matches = src.match(/data:([^;]+);/);
                        if (matches && matches[1]) {
                            mimeType = matches[1];
                        }
                    }
                    
                    // Определяме разширението на файла
                    let extension = 'jpg';
                    if (mimeType === 'image/png') extension = 'png';
                    else if (mimeType === 'image/gif') extension = 'gif';
                    else if (mimeType === 'image/webp') extension = 'webp';
                    else if (mimeType === 'image/svg+xml') extension = 'svg';
                    
                    const fileName = 'image_' + (idx + 1) + '.' + extension;
                    
                    // Създаваме File обект вместо Blob за да запазим MIME типа
                    const file = new File([blob], fileName, { type: mimeType });
                    
                    // Добавяме файла директно към FormData - PHP ще го види като истински файл
                    formData.append('images[]', file);
                }
            });
            
            // Добавяме аватар ако има
            const avatarSrc = $form.find('#rx-uploaded-avatar img').attr('src');
            if (avatarSrc) {
                const avatarBlob = dataURLtoBlob(avatarSrc);
                if (avatarBlob) {
                    formData.append('avatar[name]', 'avatar.jpg');
                    formData.append('avatar[type]', 'image/jpeg');
                    formData.append('avatar[tmp_name]', avatarBlob);
                    formData.append('avatar[error]', 0);
                    formData.append('avatar[size]', avatarBlob.size);
                }
            }
            
            // Добавяме видео ако има
            const videoSrc = $form.find('#rx-uploaded-video video source').attr('src');
            if (videoSrc) {
                formData.append('video[name]', 'video.mp4');
                formData.append('video[type]', 'video/mp4');
                formData.append('video[tmp_name]', videoSrc);
                formData.append('video[error]', 0);
                formData.append('video[size]', 0);
            }
            
            // Проверка за валиден AJAX URL и nonce
            if (typeof reviewxpress_ajax === 'undefined' || !reviewxpress_ajax.ajax_url || !reviewxpress_ajax.nonce) {
                showMessage('AJAX configuration error. Please refresh the page.', 'error');
                $submitBtn.prop('disabled', false);
                if ($btnText.length) {
                    $btnText.text(originalText);
                } else {
                    $submitBtn.text(originalText);
                }
                return;
            }
            
            // Взимаме nonce от формата (приоритет) или от reviewxpress_ajax
            const formNonce = $form.find('input[name="reviewxpress_nonce"]').val();
            const ajaxNonce = reviewxpress_ajax.nonce;
            const finalNonce = formNonce || ajaxNonce;
            
            // Ajax заявка
            formData.append('action', 'reviewxpress_submit_review');
            
            // Уверяваме се че nonce е в FormData (от формата)
            if (formData.has('reviewxpress_nonce')) {
                const existingNonce = formData.get('reviewxpress_nonce');
                if (existingNonce !== finalNonce) {
                    formData.delete('reviewxpress_nonce');
                    formData.append('reviewxpress_nonce', finalNonce);
                }
            } else {
                formData.append('reviewxpress_nonce', finalNonce);
            }
            
            // Премахваме дублирания nonce ако има
            const allNonces = formData.getAll('reviewxpress_nonce');
            if (allNonces.length > 1) {
                formData.delete('reviewxpress_nonce');
                formData.append('reviewxpress_nonce', finalNonce);
            }
            
            // Защита от множествени заявки
            if ($form.data('rx-submitting')) {
                return; // Вече се изпраща
            }
            $form.data('rx-submitting', true);
            
            $.ajax({
                url: reviewxpress_ajax.ajax_url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                timeout: 60000, // 60 секунди timeout
                success: function(response) {
                    var getMsg = function(res){
                        if (!res) return '';
                        if (typeof res === 'string') return res;
                        if (typeof res.data === 'string') return res.data;
                        if (res.data && typeof res.data.message === 'string') return res.data.message;
                        if (typeof res.message === 'string') return res.message;
                        try { return JSON.stringify(res); } catch(e) { return ''; }
                    };

                    if (response && response.success) {
                        // Показване на съобщение за успех
                        var okMsg = (response.data && response.data.message) ? response.data.message : (getMsg(response) || 'Review submitted successfully!');
                        // Скриваме формата и показваме Thank You екран
                        var $container = $form.closest('.reviewxpress-form-container');
                        if ($container.length) {
                            $form.hide();
                            var couponNote = '';
                            // Free version - no coupons, use translated message
                            if (typeof reviewxpress_ajax !== 'undefined' && reviewxpress_ajax.messages && reviewxpress_ajax.messages.review_submitted) {
                                couponNote = '<p class="rx-thanks-note">' + reviewxpress_ajax.messages.review_submitted + '</p>';
                            } else {
                                couponNote = '<p class="rx-thanks-note">Your review has been submitted for approval. Thank you for your feedback!</p>';
                            }
                            var thankYouText = '';
                            if (typeof reviewxpress_ajax !== 'undefined' && reviewxpress_ajax.messages && reviewxpress_ajax.messages.thank_you) {
                                thankYouText = reviewxpress_ajax.messages.thank_you;
                            } else {
                                thankYouText = 'Thank you for leaving a review!';
                            }
                            var thankHtml = '<div class="rx-thankyou">'+
                                '<div class="rx-thank-check">✓</div>'+ 
                                '<h3>' + thankYouText + '</h3>'+ 
                                couponNote +
                                '</div>';
                            $container.find('.rx-thankyou').remove();
                            $container.append(thankHtml);
                        } else {
                            showMessage(okMsg, 'success');
                        }

                        // Изчистване на формата
                        $form[0].reset();
                        $form.find('.rx-uploaded-avatar').empty();
                        $form.find('.rx-uploaded-images').empty();
                        $form.find('.rx-uploaded-video').empty();
                        $form.find('.rx-upload-placeholder').show().html('<span class="rx-upload-text">+ Upload files</span>');
                        $form.find('.rx-star-label').removeClass('rx-star-selected rx-star-hover');
                        $form.find('.rx-media-content').hide();
                        $form.find('.rx-toggle-btn').removeClass('active');

                        // Скрол до началото на формата
                        if ($form.offset()) {
                            $('html, body').animate({ scrollTop: $form.offset().top - 100 }, 300);
                        }

                        // Презареждане на ревютата
                        setTimeout(function() { loadReviews(); }, 500);
                    } else {
                        var errMsg = getMsg(response) || 'An error occurred while submitting the review.';
                        showMessage(errMsg, 'error');
                        $form.data('rx-submitting', false);
                    }
                    
                    // Възстановяване на бутона
                    $submitBtn.prop('disabled', false);
                    if ($btnText.length) {
                        $btnText.text(originalText);
                    } else {
                        $submitBtn.text(originalText);
                    }
                },
                complete: function() {
                    // Премахваме флага за изпращане
                    $form.data('rx-submitting', false);
                    $submitBtn.prop('disabled', false);
                    if ($btnText.length) {
                        $btnText.text(originalText);
                    } else {
                        $submitBtn.text(originalText);
                    }
                },
                error: function(xhr, status, error) {
                    $form.data('rx-submitting', false);
                    var msg = 'An error occurred while submitting the review.';
                    
                    // По-специфични съобщения за различни типове грешки
                    if (status === 'timeout') {
                        msg = 'Request timeout. Please try again.';
                    } else if (status === 'abort') {
                        msg = 'Request was cancelled.';
                        return; // Не показваме съобщение ако е отменено
                    } else if (xhr && xhr.status === 0) {
                        msg = 'Network error. Please check your connection.';
                    } else if (xhr && xhr.status >= 500) {
                        msg = 'Server error. Please try again later.';
                    } else if (xhr && xhr.status === 403) {
                        msg = 'Access denied. Please refresh the page.';
                    } else if (xhr && xhr.status === 404) {
                        msg = 'AJAX endpoint not found. Please contact administrator.';
                    }
                    
                    // Проверка за nonce грешка
                    if (xhr && xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        if (xhr.responseJSON.data.message.indexOf('nonce') !== -1 || 
                            xhr.responseJSON.data.message.indexOf('Nonce') !== -1 ||
                            xhr.responseJSON.data.message.indexOf('невалиден') !== -1) {
                            msg = 'Session expired. Please refresh the page and try again.';
                        }
                    }
                    
                    try {
                        if (xhr && xhr.responseJSON) {
                            if (typeof xhr.responseJSON === 'string') {
                                msg = xhr.responseJSON;
                            } else if (xhr.responseJSON.data && typeof xhr.responseJSON.data.message === 'string') {
                                msg = xhr.responseJSON.data.message;
                            } else if (typeof xhr.responseJSON.message === 'string') {
                                msg = xhr.responseJSON.message;
                            } else if (typeof xhr.responseJSON.data === 'string') {
                                msg = xhr.responseJSON.data;
                            }
                        } else if (xhr && xhr.responseText) {
                            try {
                                var response = JSON.parse(xhr.responseText);
                                if (response.data && response.data.message) {
                                    msg = response.data.message;
                                } else if (response.message) {
                                    msg = response.message;
                                }
                            } catch (e) {
                                msg = xhr.responseText;
                            }
                        }
        } catch (e) {
            msg = 'An unexpected error occurred while submitting the review.';
        }
                    showMessage(msg, 'error');
                },
                complete: function() {
                    // Възстановяване на бутона
                    $submitBtn.prop('disabled', false);
                    if ($btnText.length) {
                        $btnText.text(originalText);
                    } else {
                        $submitBtn.text(originalText);
                    }
                }
            });
        });
    }

    // Зареждане на ревюта
    function loadReviews() {
        const $container = $('.rx-reviews-container');
        if ($container.length === 0) return;

        const productId = $container.data('product-id');
        if (!productId || productId <= 0) {
            showMessage('Invalid product ID for loading reviews.', 'error');
            return;
        }

        $.ajax({
            url: reviewxpress_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'reviewxpress_load_reviews',
                product_id: productId,
                reviewxpress_nonce: reviewxpress_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    $container.html(response.data.html || response.data);
                } else {
                    showMessage('Error loading reviews.', 'error');
                }
            },
            error: function(xhr, status, error) {
                showMessage('Грешка при зареждане на ревютата.', 'error');
            }
        });
    }

    // Показване на съобщения
    function showMessage(message, type) {
        // Премахваме старите съобщения
        $('.rx-message').remove();
        
        const $message = $('<div class="rx-message rx-message-' + type + '">' + message + '</div>');
        $('.rx-review-form').before($message);
        
        // Скрол до съобщението
        if ($message.offset()) {
            $('html, body').animate({ scrollTop: $message.offset().top - 100 }, 300);
        }
        
        setTimeout(function() {
            $message.fadeOut(function() {
                $message.remove();
            });
        }, type === 'error' ? 8000 : 5000);
    }

    // Помощни функции за обработка на изображения
    function resizeImageToCanvasBlob(file, maxW, maxH, quality) {
        return new Promise((resolve) => {
            try {
                const img = new Image();
                const reader = new FileReader();
                reader.onload = function(e) {
                    img.onload = function() {
                        const canvas = document.createElement('canvas');
                        let w = img.width;
                        let h = img.height;
                        const ratio = Math.min(maxW / w, maxH / h, 1);
                        w = Math.round(w * ratio);
                        h = Math.round(h * ratio);
                        canvas.width = maxW;
                        canvas.height = maxH;
                        const ctx = canvas.getContext('2d');
                        ctx.fillStyle = '#fff';
                        ctx.fillRect(0, 0, canvas.width, canvas.height);
                        const x = Math.round((maxW - w) / 2);
                        const y = Math.round((maxH - h) / 2);
                        ctx.drawImage(img, x, y, w, h);
                        canvas.toBlob(function(blob) {
                            resolve({ blob: blob, type: file.type || 'image/jpeg' });
                        }, 'image/jpeg', quality || 0.9);
                    };
                    img.src = e.target.result;
                };
                reader.readAsDataURL(file);
            } catch (err) {
                resolve(null);
            }
        });
    }

    function dataURLtoBlob(dataurl) {
        try {
            const arr = dataurl.split(',');
            const mime = arr[0].match(/:(.*?);/)[1];
            const bstr = atob(arr[1]);
            let n = bstr.length;
            const u8arr = new Uint8Array(n);
            while (n--) { 
                u8arr[n] = bstr.charCodeAt(n); 
            }
            return new Blob([u8arr], { type: mime });
        } catch (e) {
            return null;
        }
    }

    // Инициализация на lightbox за снимки
    function initImageLightbox() {
        // Проверяваме дали вече е инициализиран
        if (window.reviewxpressLightboxInitialized) {
            return;
        }
        
        // Създаваме lightbox HTML ако не съществува
        if ($('#rx-lightbox').length === 0) {
            $('body').append(`
                <div id="rx-lightbox" class="rx-lightbox">
                    <div class="rx-lightbox-content">
                        <button class="rx-lightbox-close">&times;</button>
                        <button class="rx-lightbox-nav rx-lightbox-prev">&larr;</button>
                        <button class="rx-lightbox-nav rx-lightbox-next">&rarr;</button>
                        <img class="rx-lightbox-image" src="" alt="">
                    </div>
                </div>
            `);
        }

        const $lightbox = $('#rx-lightbox');
        const $lightboxImg = $lightbox.find('.rx-lightbox-image');
        const $closeBtn = $lightbox.find('.rx-lightbox-close');
        const $prevBtn = $lightbox.find('.rx-lightbox-prev');
        const $nextBtn = $lightbox.find('.rx-lightbox-next');
        
        let currentImages = [];
        let currentIndex = 0;

        // Клик върху снимка за отваряне на lightbox (използваме namespace за да избегнем конфликти)
        $(document).off('click.reviewxpress-lightbox', '.rx-clickable-image').on('click.reviewxpress-lightbox', '.rx-clickable-image', function(e) {
            e.preventDefault();
            e.stopImmediatePropagation();
            
            const $img = $(this);
            const fullImageUrl = $img.data('full-image') || $img.attr('src');
            
            if (!fullImageUrl) {
                return;
            }
            
            // Събираме всички снимки от текущия ревю
            const $review = $img.closest('.rx-review-item');
            currentImages = [];
            
            // Събираме всички снимки от ревюто (включително от media секцията)
            $review.find('.rx-clickable-image').each(function() {
                const imgUrl = $(this).data('full-image') || $(this).attr('src');
                if (imgUrl && currentImages.indexOf(imgUrl) === -1) {
                    currentImages.push(imgUrl);
                }
            });
            
            // Ако няма снимки, добавяме текущата
            if (currentImages.length === 0) {
                currentImages.push(fullImageUrl);
            }
            
            currentIndex = currentImages.indexOf(fullImageUrl);
            if (currentIndex === -1) currentIndex = 0;
            
            openLightbox(fullImageUrl);
        });

        // Отваряне на lightbox
        function openLightbox(imageUrl) {
            if (!$lightbox.length || !$lightboxImg.length) {
                return;
            }
            
            $lightboxImg.attr('src', imageUrl);
            $lightbox.addClass('active');
            $('body').addClass('rx-lightbox-open');
            
            // Показваме/скриваме навигационни бутони
            if (currentImages.length > 1) {
                $prevBtn.show();
                $nextBtn.show();
            } else {
                $prevBtn.hide();
                $nextBtn.hide();
            }
        }

        // Затваряне на lightbox
        function closeLightbox() {
            $lightbox.removeClass('active');
            $('body').removeClass('rx-lightbox-open');
        }

        // Следваща снимка
        function nextImage() {
            if (currentImages.length <= 1) return;
            currentIndex = (currentIndex + 1) % currentImages.length;
            $lightboxImg.attr('src', currentImages[currentIndex]);
        }

        // Предишна снимка
        function prevImage() {
            if (currentImages.length <= 1) return;
            currentIndex = (currentIndex - 1 + currentImages.length) % currentImages.length;
            $lightboxImg.attr('src', currentImages[currentIndex]);
        }

        // Event listeners
        $closeBtn.on('click', closeLightbox);
        $nextBtn.on('click', nextImage);
        $prevBtn.on('click', prevImage);
        
        // Клик върху фона за затваряне
        $lightbox.on('click', function(e) {
            if (e.target === this) {
                closeLightbox();
            }
        });

        // Keyboard навигация
        $(document).on('keydown', function(e) {
            if (!$lightbox.hasClass('active')) return;
            
            switch(e.key) {
                case 'Escape':
                    closeLightbox();
                    break;
                case 'ArrowLeft':
                    prevImage();
                    break;
                case 'ArrowRight':
                    nextImage();
                    break;
            }
        });

        // Предотвратяване на скрол когато lightbox е отворен
        $('body').on('wheel.reviewxpress-lightbox', function(e) {
            if ($lightbox.hasClass('active')) {
                e.preventDefault();
            }
        });
        
        // Маркираме като инициализиран
        window.reviewxpressLightboxInitialized = true;
    }

        // Инициализация на lightbox за снимки
        initImageLightbox();
        
        // Реинициализация след AJAX зареждане на ревюта
        $(document).on('rx-reviews-loaded', function() {
            // Не реинициализираме, защото event handler-ът е глобален
        });
        
        // Глобален event handler за Helpful бутона (работи навсякъде)
        $(document).on('click', '.rx-helpful-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const $btn = $(this);
            const reviewId = $btn.data('review-id');
            const $countSpan = $btn.find('.rx-helpful-count');
            
            if (!reviewId) {
                return;
            }
            
            // Проверяваме дали AJAX обектът е наличен
            if (typeof reviewxpress_ajax === 'undefined') {
                return;
            }
            
            const isActive = $btn.hasClass('rx-helpful-active');
            const helpfulValue = isActive ? 0 : 1;
            
            // Деактивираме бутона докато се изпълнява заявката
            $btn.prop('disabled', true);
            
            $.ajax({
                url: reviewxpress_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'reviewxpress_toggle_helpful',
                    review_id: reviewId,
                    helpful: helpfulValue,
                    nonce: reviewxpress_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        if (helpfulValue === 1) {
                            $btn.addClass('rx-helpful-active');
                        } else {
                            $btn.removeClass('rx-helpful-active');
                        }
                        if ($countSpan.length) {
                            $countSpan.text('(' + response.data.helpful_count + ')');
                        }
                    }
                },
                error: function() {
                    // Silent error handling
                },
                complete: function() {
                    // Активираме бутона отново
                    $btn.prop('disabled', false);
                }
            });
        });

        // Инициализация при зареждане на страницата
        try {
            initReviewForm();
            
            // Автоматично поправяне на празни лейбъли
            setTimeout(function() {
                const $forms = $('.rx-review-form, .reviewxpress-form');
                
                $forms.each(function() {
                    const $form = $(this);
                    const $labels = $form.find('.rx-form-group label');
                    
                    $labels.each(function() {
                        const $label = $(this);
                        const $labelText = $label.find('.rx-label-text');
                        
                        // Пропускаме звездите за рейтинг и други специални лейбъли
                        if ($label.hasClass('rx-star-label') || $label.attr('for') === 'rx-avatar') {
                            return;
                        }
                        
                        // Проверяваме само ако текстът е празен
                        if ($labelText.length > 0 && (!$labelText.text() || $labelText.text().trim() === '')) {
                            // Опитваме се да определим текста от for атрибута
                            const forAttr = $label.attr('for');
                            let fallbackText = '';
                            if (forAttr === 'rx-name') {
                                fallbackText = 'Name';
                            } else if (forAttr === 'rx-email') {
                                fallbackText = 'Email';
                            } else if (forAttr === 'rx-review-text') {
                                fallbackText = 'Review';
                            } else if ($label.find('.rx-required-asterisk').length > 0 && !forAttr) {
                                fallbackText = 'Rating';
                            }
                            
                            if (fallbackText) {
                                $labelText.text(fallbackText);
                            }
                        }
                    });
                });
            }, 1000);
        } catch (e) {
            showMessage('Error initializing review form.', 'error');
        }
        
        // Инициализация при динамично зареждане на съдържание
        if (window.MutationObserver) {
            try {
                const observer = new MutationObserver(function(mutations) {
                    mutations.forEach(function(mutation) {
                        if (mutation.type === 'childList') {
                            mutation.addedNodes.forEach(function(node) {
                                if (node.nodeType === 1) {
                                    const $node = $(node);
                                    if ($node.find('.rx-review-form').length > 0 || $node.hasClass('rx-review-form')) {
                                        try {
                                            initReviewForm();
                                        } catch (e) {
                                        }
                                    }
                                }
                            });
                        }
                    });
                });
                observer.observe(document.body, { childList: true, subtree: true });
            } catch (e) {
            }
        }
});

/**
 * Review List functionality
 */
jQuery(document).ready(function($) {
    // Initialize review list functionality
    function initReviewList($container) {
        if ($container.length === 0 || $container.data('rx-initialized')) {
            return;
        }
        $container.data('rx-initialized', true);
        
        var productId = parseInt($container.data('product-id')) || 0;
        var orderby = $container.data('orderby') || 'date';
        var order = $container.data('order') || 'DESC';
        var currentPage = 1;
        var loading = false;
        
        // Sorting
        $container.find('#rx-sort-select').on('change', function() {
            orderby = $(this).val();
            currentPage = 1;
            loadReviews();
        });
        
        // Load more reviews
        $container.find('#rx-load-more').on('click', function() {
            if (!loading) {
                currentPage++;
                loadReviews();
            }
        });
        
        // Load reviews function
        function loadReviews() {
            if (loading) return;
            
            loading = true;
            var loadMoreBtn = $container.find('#rx-load-more');
            var originalText = loadMoreBtn.text();
            
            if (currentPage === 1) {
                loadMoreBtn.text(reviewxpress_ajax.messages.loading || 'Loading...');
            }
            
            $.ajax({
                url: reviewxpress_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'reviewxpress_load_reviews',
                    product_id: productId,
                    page: currentPage,
                    orderby: orderby,
                    order: order,
                    nonce: reviewxpress_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        if (currentPage === 1) {
                            $container.find('#rx-reviews-list').html(response.data.html);
                        } else {
                            $container.find('#rx-reviews-list').append(response.data.html);
                        }
                        
                        if (!response.data.has_more) {
                            $container.find('.rx-load-more-container').hide();
                        }
                    }
                },
                error: function() {
                    alert(reviewxpress_ajax.messages.error || 'An error occurred while loading reviews.');
                },
                complete: function() {
                    loading = false;
                    loadMoreBtn.text(originalText);
                }
            });
        }
        
        // Helpful button се обработва глобално (виж по-долу в кода)
        
        // Gallery functionality for side-by-side template
        if ($container.find('.rx-reviews-list.side-by-side').length > 0) {
            initSideBySideGallery($container);
        }
        
        // Pagination
        var totalReviews = parseInt($container.data('total-reviews')) || 0;
        var reviewsPerPage = parseInt($container.data('reviews-per-page')) || 5;
        
        if (totalReviews > reviewsPerPage) {
            updatePagination($container, {
                current_page: 1,
                total_pages: Math.ceil(totalReviews / reviewsPerPage),
                has_prev: false,
                has_next: totalReviews > reviewsPerPage
            });
        }
    }
    
    // Side-by-side gallery functionality
    function initSideBySideGallery($container) {
        if ($('#rx-media-modal').length === 0) {
            $('body').append('<div id="rx-media-modal" class="rx-media-modal"><div class="rx-media-modal-content"><span class="rx-media-modal-close">&times;</span><div class="rx-media-modal-prev">‹</div><div class="rx-media-modal-next">›</div></div></div>');
        }
        
        var currentMediaIndex = 0;
        var currentMediaList = [];
        
        $container.on('click', '.review-media-main, .review-media-thumbnail', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            var $this = $(this);
            var $reviewItem = $this.closest('.rx-review-item');
            var $allMedia = $reviewItem.find('.review-media-main, .review-media-thumbnail');
            
            currentMediaList = [];
            $allMedia.each(function() {
                currentMediaList.push({
                    url: $(this).data('full-image'),
                    type: $(this).data('media-type')
                });
            });
            
            currentMediaIndex = parseInt($this.data('media-index')) || 0;
            showModal();
        });
        
        function showModal() {
            if (currentMediaList.length === 0) return;
            
            var media = currentMediaList[currentMediaIndex];
            var $modal = $('#rx-media-modal');
            var $content = $modal.find('.rx-media-modal-content');
            
            $content.find('img, video').remove();
            
            if (media.type === 'video') {
                $content.append('<video controls autoplay><source src="' + media.url + '" type="video/mp4"></video>');
            } else {
                $content.append('<img src="' + media.url + '" alt="Review Media">');
            }
            
            if (currentMediaList.length > 1) {
                $modal.find('.rx-media-modal-prev, .rx-media-modal-next').removeClass('hidden');
            } else {
                $modal.find('.rx-media-modal-prev, .rx-media-modal-next').addClass('hidden');
            }
            
            $modal.addClass('show');
            $('body').css('overflow', 'hidden');
        }
        
        $(document).on('click', '.rx-media-modal-close, .rx-media-modal', function(e) {
            if (e.target === this || $(e.target).hasClass('rx-media-modal-close')) {
                $('#rx-media-modal').removeClass('show');
                $('body').css('overflow', '');
            }
        });
        
        $(document).on('click', '.rx-media-modal-prev', function(e) {
            e.stopPropagation();
            currentMediaIndex = (currentMediaIndex - 1 + currentMediaList.length) % currentMediaList.length;
            showModal();
        });
        
        $(document).on('click', '.rx-media-modal-next', function(e) {
            e.stopPropagation();
            currentMediaIndex = (currentMediaIndex + 1) % currentMediaList.length;
            showModal();
        });
        
        $(document).on('keydown', function(e) {
            if ($('#rx-media-modal').hasClass('show')) {
                if (e.key === 'Escape') {
                    $('#rx-media-modal').removeClass('show');
                    $('body').css('overflow', '');
                } else if (e.key === 'ArrowLeft') {
                    $('.rx-media-modal-prev').trigger('click');
                } else if (e.key === 'ArrowRight') {
                    $('.rx-media-modal-next').trigger('click');
                }
            }
        });
    }
    
    // Pagination functions
    window.loadReviewsPage = function(page) {
        var $container = $('.reviewxpress-list-container').first();
        var productId = parseInt($container.data('product-id')) || 0;
        var orderby = $container.find('.rx-sorting select[name="orderby"]').val() || 'created_at';
        var order = $container.find('.rx-sorting select[name="order"]').val() || 'DESC';
        
        $.ajax({
            url: reviewxpress_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'reviewxpress_load_reviews_paginated',
                product_id: productId,
                page: page,
                orderby: orderby,
                order: order,
                nonce: reviewxpress_ajax.nonce
            },
            beforeSend: function() {
                $container.find('.rx-reviews-list').html('<div class="rx-loading">' + (reviewxpress_ajax.messages.loading || 'Loading...') + '</div>');
            },
            success: function(response) {
                if (response.success) {
                    $container.find('.rx-reviews-list').html(response.data.html);
                    updatePagination($container, response.data);
                } else {
                    $container.find('.rx-reviews-list').html('<div class="rx-error">' + (reviewxpress_ajax.messages.error || 'Error loading reviews.') + '</div>');
                }
            },
            error: function() {
                $container.find('.rx-reviews-list').html('<div class="rx-error">' + (reviewxpress_ajax.messages.error || 'Error loading reviews.') + '</div>');
            }
        });
    };
    
    function updatePagination($container, data) {
        var paginationHtml = '<div class="rx-pagination">';
        var prevText = reviewxpress_ajax.messages.previous || 'Previous';
        var nextText = reviewxpress_ajax.messages.next || 'Next';
        var pageText = reviewxpress_ajax.messages.page || 'Page';
        var ofText = reviewxpress_ajax.messages.of || 'of';
        
        if (data.has_prev) {
            paginationHtml += '<button class="rx-pagination-btn" onclick="loadReviewsPage(' + (data.current_page - 1) + ')">← ' + prevText + '</button>';
        }
        
        paginationHtml += '<span class="rx-pagination-info">' + pageText + ' ' + data.current_page + ' ' + ofText + ' ' + data.total_pages + '</span>';
        
        if (data.has_next) {
            paginationHtml += '<button class="rx-pagination-btn" onclick="loadReviewsPage(' + (data.current_page + 1) + ')">' + nextText + ' →</button>';
        }
        
        paginationHtml += '</div>';
        
        $container.find('.rx-pagination').remove();
        $container.find('.rx-reviews-list').after(paginationHtml);
    }
    
    // Initialize all review list containers
    $('.reviewxpress-list-container').each(function() {
        initReviewList($(this));
    });
    
    // Re-initialize on dynamic content load
    if (window.MutationObserver) {
        var observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.type === 'childList') {
                    mutation.addedNodes.forEach(function(node) {
                        if (node.nodeType === 1) {
                            var $node = $(node);
                            if ($node.hasClass('reviewxpress-list-container') || $node.find('.reviewxpress-list-container').length > 0) {
                                $node.find('.reviewxpress-list-container').each(function() {
                                    initReviewList($(this));
                                });
                            }
                        }
                    });
                }
            });
        });
        observer.observe(document.body, { childList: true, subtree: true });
    }
});

/**
 * Divi Theme Builder: Inject reviews with JavaScript
 */
function reviewxpressDiviInjectReviews() {
    if (typeof reviewxpressDivi === 'undefined' || !reviewxpressDivi.reviewsHtml) {
        return;
    }
    
    jQuery(document).ready(function($) {
        // Проверяваме дали има Divi табове
        if ($('.et_pb_wc_tabs').length > 0) {
            // Decode base64 encoded HTML
            var reviewsHtml = '';
            if (reviewxpressDivi.reviewsHtmlEncoded) {
                try {
                    reviewsHtml = atob(reviewxpressDivi.reviewsHtml);
                } catch (e) {
                    reviewsHtml = reviewxpressDivi.reviewsHtml;
                }
            } else {
                reviewsHtml = reviewxpressDivi.reviewsHtml;
            }
            
            // Опитваме различни места за инжектиране
            var injected = false;
            
            // Метод 1: След табовете
            if ($('.et_pb_wc_tabs').length > 0 && !injected) {
                $('.et_pb_wc_tabs').after(reviewsHtml);
                injected = true;
            }
            
            // Метод 2: В началото на "Отзиви" таба (ако има такъв)
            if ($('.et_pb_tab_content').length > 1 && !injected) {
                $('.et_pb_tab_content').eq(1).prepend(reviewsHtml);
                injected = true;
            }
            
            // Метод 3: Преди стандартните WooCommerce ревюта
            if ($('.woocommerce-Reviews').length > 0 && !injected) {
                $('.woocommerce-Reviews').before(reviewsHtml);
                injected = true;
            }
        }
    });
}

/**
 * Divi Theme Builder: Inject form with JavaScript
 */
function reviewxpressDiviInjectForm() {
    if (typeof reviewxpressDivi === 'undefined' || !reviewxpressDivi.formHtml) {
        return;
    }
    
    jQuery(document).ready(function($) {
        // Decode base64 encoded HTML
        var formHtml = '';
        if (reviewxpressDivi.formHtmlEncoded) {
            try {
                formHtml = atob(reviewxpressDivi.formHtml);
            } catch (e) {
                formHtml = reviewxpressDivi.formHtml;
            }
        } else {
            formHtml = reviewxpressDivi.formHtml;
        }
        
        var injected = false;
        var placement = reviewxpressDivi.formPlacement || 'after_tab';
        
        // Чакаме малко за да се заредят Divi елементите
        setTimeout(function() {
            // Метод 1: След Divi табовете (най-често използвано)
            if ($('.et_pb_wc_tabs').length > 0 && !injected) {
                if (placement === 'after_tab' || placement === 'after_description') {
                    $('.et_pb_wc_tabs').after(formHtml);
                    injected = true;
                } else if (placement === 'before_tab') {
                    $('.et_pb_wc_tabs').before(formHtml);
                    injected = true;
                }
            }
            
            // Метод 2: В "Отзиви" таба, ако има
            if ($('.et_pb_tab_content').length > 0 && !injected) {
                var reviewsTab = $('.et_pb_tab_content').filter(function() {
                    return $(this).text().toLowerCase().indexOf('review') !== -1 || 
                           $(this).text().toLowerCase().indexOf('review') !== -1 ||
                           $(this).text().toLowerCase().indexOf('revue') !== -1;
                });
                
                if (reviewsTab.length > 0) {
                    reviewsTab.prepend(formHtml);
                    injected = true;
                }
            }
            
            // Метод 3: След стандартните WooCommerce ревюта
            if ($('.woocommerce-Reviews').length > 0 && !injected) {
                if (placement === 'after_tab' || placement === 'after_description') {
                    $('.woocommerce-Reviews').after(formHtml);
                    injected = true;
                } else if (placement === 'before_tab') {
                    $('.woocommerce-Reviews').before(formHtml);
                    injected = true;
                }
            }
            
            // Метод 4: След product summary (преди табове)
            if ($('.product-summary').length > 0 && !injected) {
                if (placement === 'before_description' || placement === 'before_tab') {
                    $('.product-summary').after(formHtml);
                    injected = true;
                }
            }
            
            // Метод 5: Fallback - след всички product елементи
            if (!injected && $('.et_pb_wc_tabs').length > 0) {
                $('.et_pb_wc_tabs').after(formHtml);
                injected = true;
            }
            
            // Метод 6: Абсолютно fallback - в края на продуктната страница
            if (!injected && $('.woocommerce-product-page').length > 0) {
                $('.woocommerce-product-page').append(formHtml);
                injected = true;
            }
        }, 500); // Даваме време на Divi да зареди всички елементи
    });
}

// Инициализиране на Divi функции ако има локализирани данни
// Използваме отделни проверки за да избегнем проблеми с && encoding
if (typeof reviewxpressDivi !== 'undefined') {
    if (reviewxpressDivi.reviewsHtml) {
        reviewxpressDiviInjectReviews();
    }
    if (reviewxpressDivi.formHtml) {
        reviewxpressDiviInjectForm();
    }
}