/**
 * Админ JavaScript за ReviewXpress
 */

jQuery(document).ready(function($) {
    'use strict';
    
    // Централизиран AJAX URL – локализиран от PHP
    const ajaxURL = (typeof reviewxpress_admin !== 'undefined' && reviewxpress_admin.ajax_url)
        ? reviewxpress_admin.ajax_url
        : (typeof ajaxurl !== 'undefined' ? ajaxurl : '');
    
    // Инициализиране на админ функционалностите
    initAdminFeatures();
    
    /**
     * Инициализиране на админ функционалностите
     */
    function initAdminFeatures() {
        try {
            // Одобряване на ревю
            $('.rx-approve-btn').on('click', function() {
                handleReviewAction($(this), 'approve');
            });
            
            // Отхвърляне на ревю
            $('.rx-reject-btn').on('click', function() {
                handleReviewAction($(this), 'reject');
            });
            
            // Изтриване на ревю
            $('.rx-delete-btn').on('click', function() {
                handleReviewAction($(this), 'delete');
            });
            
            // Преглед на ревю
            $('.rx-view-btn').on('click', function() {
                showReviewModal($(this));
            });
            
            // Затваряне на модала
            $('.rx-modal-close, .rx-modal').on('click', function(e) {
                if (e.target === this) {
                    closeModal();
                }
            });
            
            // Филтриране
            $('.rx-admin-filters form').on('submit', function(e) {
                // Формата ще се изпрати нормално
            });
            
            // Масови действия
            initBulkActions();
        } catch (e) {
        }
    }
    
    /**
     * Обработка на действия с ревюта
     */
    function handleReviewAction(btn, action) {
        try {
            var reviewId = btn.data('review-id');
            var row = btn.closest('tr');
            var actionText = getActionText(action);
            
            if (!reviewId || reviewId <= 0) {
                alert('Invalid review ID.');
                return;
            }
            
            if (confirm('Are you sure you want to ' + actionText.toLowerCase() + ' this review?')) {
            var originalText = btn.text();
            btn.text('Processing...');
            btn.prop('disabled', true);
            
            $.ajax({
                url: ajaxURL,
                type: 'POST',
                data: {
                    action: 'reviewxpress_' + action + '_review',
                    review_id: reviewId,
                    nonce: (typeof reviewxpress_admin !== 'undefined' ? reviewxpress_admin.nonce : '')
                },
                success: function(response) {
                    if (response && response.success) {
                        handleActionSuccess(action, row, reviewId);
                        showToast('Action completed successfully!', 'success');
                    } else {
                        var errorMsg = 'An error occurred during processing.';
                        if (response && response.data && response.data.message) {
                            errorMsg = response.data.message;
                        }
                        alert(errorMsg);
                        showToast(errorMsg, 'error');
                    }
                },
                error: function(xhr, status, error) {
                    var errorMsg = 'An error occurred while sending the request.';
                    try {
                        if (xhr && xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                            errorMsg = xhr.responseJSON.data.message;
                        } else if (xhr && xhr.responseText) {
                            var response = JSON.parse(xhr.responseText);
                            if (response && response.data && response.data.message) {
                                errorMsg = response.data.message;
                            }
                        }
                    } catch (e) {
                    }
                    alert(errorMsg);
                    showToast(errorMsg, 'error');
                },
                complete: function() {
                    btn.text(originalText);
                    btn.prop('disabled', false);
                }
            });
            }
        } catch (e) {
            alert('An error occurred while processing the action.');
        }
    }
    
    /**
     * Вземане на текст за действие
     */
    function getActionText(action) {
        var texts = {
            'approve': 'approve',
            'reject': 'reject',
            'delete': 'delete'
        };
        return texts[action] || action;
    }
    
    /**
     * Обработка на успешно действие
     */
    function handleActionSuccess(action, row, reviewId) {
        switch (action) {
            case 'approve':
                row.find('.rx-status').removeClass('rx-status-pending').addClass('rx-status-approved').text('Approved');
                updateActions(row, reviewId);
                break;
                
            case 'reject':
                row.find('.rx-status').removeClass('rx-status-pending').addClass('rx-status-rejected').text('Rejected');
                updateActions(row, reviewId);
                break;
                
            case 'delete':
                row.fadeOut(300, function() {
                    $(this).remove();
                });
                break;
        }
    }
    
    /**
     * Обновяване на действията в реда
     */
    function updateActions(row, reviewId) {
        var actionsHtml = '<div class="rx-actions">' +
            '<button type="button" class="button button-small rx-view-btn" data-review-id="' + reviewId + '">View</button>' +
            '<button type="button" class="button button-small button-link-delete rx-delete-btn" data-review-id="' + reviewId + '">Delete</button>' +
            '</div>';
        row.find('.rx-actions').html(actionsHtml);
        
        // Добавяне на event listeners за новите бутони
        row.find('.rx-view-btn').on('click', function() {
            showReviewModal($(this));
        });
        
        row.find('.rx-delete-btn').on('click', function() {
            handleReviewAction($(this), 'delete');
        });
    }
    
    /**
     * Показване на модал за преглед на ревю
     */
    function showReviewModal(btn) {
        try {
            var reviewId = btn.data('review-id');
            var row = $('tr[data-review-id="' + reviewId + '"]');
            
            if (!reviewId || reviewId <= 0) {
                alert('Invalid review ID.');
                return;
            }
            
            if (row.length === 0) {
                alert('Review not found.');
                return;
            }
            
            // Събиране на данните от реда
            var reviewData = {
                id: reviewId,
                product: row.find('td:nth-child(2)').text().trim(),
                author: row.find('td:nth-child(3)').text().trim(),
                email: row.find('td:nth-child(4)').text().trim(),
                rating: row.find('.rx-rating-number').text().trim(),
                text: row.find('.rx-review-preview').text().trim(),
                status: row.find('.rx-status').text().trim(),
                date: row.find('td:nth-child(9)').text().trim()
            };
            
            // Генериране на HTML за модала
            var modalContent = generateModalContent(reviewData);
            
            // Показване на модала
            $('#rx-modal-body').html(modalContent);
            $('#rx-review-modal').show();
            
            // Добавяне на event listener за затваряне
            $('.rx-modal-close').off('click').on('click', function() {
                closeModal();
            });
        } catch (e) {
            alert('An error occurred while displaying the modal.');
        }
    }
    
    /**
     * Генериране на съдържание за модала
     */
    function generateModalContent(data) {
        var html = '<div class="rx-review-detail">';
        
        html += '<h3>' + escapeHtml(data.author) + '</h3>';
        html += '<div class="rx-review-meta">';
        html += '<p><strong>Product:</strong> ' + escapeHtml(data.product) + '</p>';
        html += '<p><strong>Email:</strong> ' + escapeHtml(data.email) + '</p>';
        html += '<p><strong>Rating:</strong> ' + escapeHtml(data.rating) + ' \u2605</p>';
        html += '<p><strong>Date:</strong> ' + escapeHtml(data.date) + '</p>';
        html += '<p><strong>Status:</strong> <span class="rx-status rx-status-' + data.status.toLowerCase() + '">' + escapeHtml(data.status) + '</span></p>';
        html += '</div>';
        
        if (data.text && data.text !== 'No text') {
            html += '<div class="rx-review-content">';
            html += '<h4>Review:</h4>';
            html += '<div class="rx-review-text">' + escapeHtml(data.text) + '</div>';
            html += '</div>';
        }
        
        html += '</div>';
        
        return html;
    }
    
    /**
     * Затваряне на модала
     */
    function closeModal() {
        $('#rx-review-modal').hide();
    }
    
    /**
     * Инициализиране на масови действия
     */
    function initBulkActions() {
        // Добавяне на checkbox за всички редове
        if ($('.rx-bulk-checkbox').length === 0) {
            var headerCheckbox = '<th><input type="checkbox" class="rx-bulk-checkbox" id="rx-select-all"></th>';
            $('.wp-list-table thead tr').prepend(headerCheckbox);
            
            $('.wp-list-table tbody tr').each(function() {
                var checkbox = '<td><input type="checkbox" class="rx-review-checkbox" value="' + $(this).data('review-id') + '"></td>';
                $(this).prepend(checkbox);
            });
        }
        
        // Select all functionality
        $('#rx-select-all').on('change', function() {
            var isChecked = $(this).is(':checked');
            $('.rx-review-checkbox').prop('checked', isChecked);
        });
        
        // Individual checkbox change
        $('.rx-review-checkbox').on('change', function() {
            var totalCheckboxes = $('.rx-review-checkbox').length;
            var checkedCheckboxes = $('.rx-review-checkbox:checked').length;
            $('#rx-select-all').prop('checked', totalCheckboxes === checkedCheckboxes);
        });
    }
    
    /**
     * Escape HTML
     */
    function escapeHtml(text) {
        var map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }
    
    /**
     * Показване на toast съобщение
     */
    function showToast(message, type) {
        type = type || 'success';
        
        var toast = $('<div class="rx-toast rx-toast-' + type + '">' + message + '</div>');
        $('body').append(toast);
        
        setTimeout(function() {
            toast.addClass('rx-toast-show');
        }, 100);
        
        setTimeout(function() {
            toast.removeClass('rx-toast-show');
            setTimeout(function() {
                toast.remove();
            }, 300);
        }, 3000);
    }
    
    /**
     * Обновяване на статистиките
     */
    function updateStats() {
        $.ajax({
            url: ajaxURL,
            type: 'POST',
            data: {
                action: 'reviewxpress_get_stats',
                nonce: (typeof reviewxpress_admin !== 'undefined' ? reviewxpress_admin.nonce : '')
            },
            success: function(response) {
                if (response && response.success) {
                    updateStatsDisplay(response.data);
                }
            },
            error: function(xhr, status, error) {
            }
        });
    }
    
    /**
     * Обновяване на показването на статистиките
     */
    function updateStatsDisplay(stats) {
        $('.rx-stat-number').each(function() {
            var type = $(this).parent().find('h3').text().toLowerCase();
            var value = 0;
            
            switch (type) {
                case 'total reviews':
                    value = stats.total;
                    break;
                case 'pending approval':
                    value = stats.pending;
                    break;
                case 'approved':
                    value = stats.approved;
                    break;
                case 'rejected':
                    value = stats.rejected;
                    break;
                case 'average rating':
                    value = stats.average_rating + ' \u2605';
                    break;
            }
            
            $(this).text(value);
        });
    }
    
    // Автоматично обновяване на статистиките всеки 30 секунди (само ако има статистики на страницата)
    if ($('.rx-stat-number').length > 0) {
        setInterval(updateStats, 30000);
    }
    
    // Reviews Main Page specific functionality
    if ($('#rx-list').length > 0 && typeof reviewxpress_reviews_main !== 'undefined') {
        initReviewsMainPage();
    }
});

/**
 * Initialize Reviews Main Page functionality
 */
function initReviewsMainPage() {
    jQuery(document).ready(function($) {
        var strings = reviewxpress_reviews_main.strings;
        var exportData = {
            status: reviewxpress_reviews_main.status,
            product_id: reviewxpress_reviews_main.product_id,
            search: reviewxpress_reviews_main.search
        };
        
        function openModal(){ $('#rx-review-modal').css('display','flex'); }
        function closeModal(){ $('#rx-review-modal').css('display','none'); }
        
        // Одобряване на ревю
        $('.rx-approve-btn').on('click', function() {
            var reviewId = $(this).data('review-id');
            var row = $(this).closest('tr');
            
            if (confirm(strings.confirm_approve)) {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'reviewxpress_approve_review',
                        review_id: reviewId,
                        nonce: reviewxpress_admin.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            row.find('.badge').removeClass('pend').addClass('ok').text(strings.approved);
                            row.find('.actions').html('<button type="button" class="btn xs rx-view-btn" data-review-id="' + reviewId + '">' + strings.view + '</button>');
                        } else {
                            alert(response.data.message);
                        }
                    }
                });
            }
        });
        
        // Отхвърляне на ревю
        $('.rx-reject-btn').on('click', function() {
            var reviewId = $(this).data('review-id');
            var row = $(this).closest('tr');
            
            if (confirm(strings.confirm_reject)) {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'reviewxpress_reject_review',
                        review_id: reviewId,
                        nonce: reviewxpress_admin.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            row.find('.badge').removeClass('pend').addClass('den').text(strings.rejected);
                            row.find('.actions').html('<button type="button" class="btn xs rx-view-btn" data-review-id="' + reviewId + '">' + strings.view + '</button>');
                        } else {
                            alert(response.data.message);
                        }
                    }
                });
            }
        });
        
        // Изтриване на ревю
        $('.rx-delete-btn').on('click', function() {
            var reviewId = $(this).data('review-id');
            var row = $(this).closest('tr');
            
            if (confirm(strings.confirm_delete)) {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'reviewxpress_delete_review',
                        review_id: reviewId,
                        nonce: reviewxpress_admin.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            row.fadeOut();
                        } else {
                            alert(response.data.message);
                        }
                    }
                });
            }
        });
        
        // Преглед на ревю
        $('.rx-view-btn').on('click', function() {
            var reviewId = $(this).data('review-id');
            var row = $('tr[data-review-id="' + reviewId + '"]');
            
            var reviewData = {
                id: reviewId,
                product: row.find('td:nth-child(3)').text().trim(),
                author: row.find('td:nth-child(4)').text().trim(),
                email: row.find('td:nth-child(5)').text().trim(),
                rating: row.find('td:nth-child(6) small').text().trim(),
                text: row.find('td:nth-child(7)').text().trim(),
                media: row.find('td:nth-child(8)').html(),
                status: row.find('td:nth-child(9) .badge').text().trim(),
                date: row.find('td:nth-child(10)').text().trim()
            };
            
            // Генериране на звезди за рейтинга
            var stars = '';
            var ratingNum = parseInt(reviewData.rating) || 0;
            for (var i = 1; i <= 5; i++) {
                stars += i <= ratingNum ? '\u2605' : '\u2606';
            }
            
            var modalContent = '<div class="rx-review-detail">';
            modalContent += '<div class="rx-review-header">';
            modalContent += '<h3>' + reviewData.author + '</h3>';
            modalContent += '<div class="rx-review-rating">' + stars + ' (' + reviewData.rating + '/5)</div>';
            modalContent += '</div>';
            
            modalContent += '<div class="rx-review-info">';
            modalContent += '<p><strong>' + strings.product + '</strong> ' + reviewData.product + '</p>';
            modalContent += '<p><strong>' + strings.email + '</strong> ' + reviewData.email + '</p>';
            modalContent += '<p><strong>' + strings.date + '</strong> ' + reviewData.date + '</p>';
            modalContent += '<p><strong>' + strings.status + '</strong> <span class="badge ' + (reviewData.status === strings.approved ? 'ok' : (reviewData.status === 'ЧАКАЩО' ? 'pend' : 'den')) + '">' + reviewData.status + '</span></p>';
            modalContent += '</div>';
            
            // Показване на медиите
            if (reviewData.media && reviewData.media.trim() !== '' && !reviewData.media.includes('No media')) {
                modalContent += '<div class="rx-review-media-section">';
                modalContent += '<h4>' + strings.media + '</h4>';
                modalContent += '<div class="rx-review-media">' + reviewData.media + '</div>';
                modalContent += '</div>';
            }
            
            // Показване на текста на ревюто
            if (reviewData.text && reviewData.text.trim() !== '' && !reviewData.text.includes('No text')) {
                modalContent += '<div class="rx-review-text-section">';
                modalContent += '<h4>' + strings.review + '</h4>';
                modalContent += '<div class="rx-review-text">' + reviewData.text + '</div>';
                modalContent += '</div>';
            }
            
            modalContent += '</div>';
            
            $('#rx-modal-body').html(modalContent);
            openModal();
        });
        
        // Затваряне на модала
        $('.rx-modal-close, .rx-modal').on('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });

        // Нулиране на филтрите
        $('#rx-reset-btn').on('click', function(){
            var $form = $('#rx-filters-form');
            $form.find('select[name="status"]').val('all');
            $form.find('select[name="product_id"]').val('0');
            $form.find('input[name="s"]').val('');
            $form.trigger('submit');
        });

        // Затваряне с ESC
        $(document).on('keydown', function(e){ if(e.key === 'Escape'){ closeModal(); } });
        
        // Експорт на ревюта
        $('#rx-export-btn').on('click', function() {
            // Създаваме форма за submit за да можем да получим файл директно
            var $form = $('<form>', {
                method: 'POST',
                action: ajaxurl,
                target: '_blank'
            });
            
            $form.append($('<input>', {
                type: 'hidden',
                name: 'action',
                value: 'reviewxpress_export_reviews'
            }));
            
            $form.append($('<input>', {
                type: 'hidden',
                name: 'nonce',
                value: reviewxpress_admin.nonce
            }));
            
            $form.append($('<input>', {
                type: 'hidden',
                name: 'status',
                value: exportData.status || 'all'
            }));
            
            $form.append($('<input>', {
                type: 'hidden',
                name: 'product_id',
                value: exportData.product_id || 0
            }));
            
            $form.append($('<input>', {
                type: 'hidden',
                name: 'search',
                value: exportData.search || ''
            }));
            
            $form.css('display', 'none');
            $('body').append($form);
            $form.submit();
            $form.remove();
        });
    });
}

/**
 * Initialize Settings General Page functionality
 */
function initSettingsGeneralPage() {
    jQuery(document).ready(function($) {
        // Показване/скриване на custom CSS полетата
        $('#custom_css_enabled').on('change', function() {
            if ($(this).is(':checked')) {
                $('.custom-css-group').show();
            } else {
                $('.custom-css-group').hide();
            }
        });
        
        // Инициализиране при зареждане
        $('#custom_css_enabled').trigger('change');
        
        // Обновяване на show_reviews въз основа на review_mode
        $('input[name="review_mode"]').on('change', function() {
            var reviewMode = $(this).val();
            var showReviews = 0;
            
            switch(reviewMode) {
                case 'manual_approve':
                case 'auto_approve':
                    showReviews = 1;
                    break;
                case 'disabled':
                case 'hidden_auto':
                default:
                    showReviews = 0;
                    break;
            }
            
            $('#show_reviews').val(showReviews);
        });
        
        // Инициализиране при зареждане
        $('input[name="review_mode"]:checked').trigger('change');
        
        // Показване/скриване на полетата за позициониране и стил на ревютата
        $('#reviews_display_type').on('change', function() {
            var displayType = $(this).val();
            if (displayType === 'position') {
                $('#reviews_position_group, #reviews_style_group').show();
            } else {
                // Скрываме за 'woocommerce_tab' и 'shortcode'
                $('#reviews_position_group, #reviews_style_group').hide();
            }
        });
        
        // Инициализиране при зареждане
        $('#reviews_display_type').trigger('change');
        
        // Уверяваме се че checkbox полетата винаги се изпращат в POST заявката
        // Добавяме hidden поле преди изпращане на формата
        // Използваме делегиране на събития за по-надеждно закачване
        $(document).on('submit', '#rx-settings form', function(e) {
            var $form = $(this);
            
            // Списък с checkbox полета които трябва винаги да се изпращат
            var checkboxes = ['show_reviews_section', 'show_form_in_reviews_section', 'require_woocommerce_order', 'allow_guest_reviews', 'custom_css_enabled'];
            
            checkboxes.forEach(function(checkboxName) {
                var $checkbox = $form.find('input[name="' + checkboxName + '"][type="checkbox"]');
                if ($checkbox.length > 0) {
                    if (!$checkbox.is(':checked')) {
                        // Ако checkbox-ът не е избран, добавяме hidden поле със стойност 0
                        if ($form.find('input[name="' + checkboxName + '"][type="hidden"]').length === 0) {
                            $form.append($('<input>').attr({
                                type: 'hidden',
                                name: checkboxName,
                                value: '0'
                            }));
                        }
                    } else {
                        // Ако е избран, премахваме hidden полето ако съществува
                        $form.find('input[name="' + checkboxName + '"][type="hidden"]').remove();
                    }
                }
            });
        });
    });
}

// Инициализиране на settings-general страницата
jQuery(document).ready(function($) {
    if ($('#rx-settings').length > 0) {
        initSettingsGeneralPage();
    }
});
