<?php

defined('ABSPATH') || exit;

// Process form submissions
Review_Wall_Form_Handler::process_form_submission();

// Get DB instance to interact with URLs
$db = Review_Wall_Helper::get_db_instance();

// Get all URLs
$urls = $db->get_urls();

// Get global settings
$logo             = get_option('review_wall_logo');
$rating_threshold = get_option('review_wall_rating_threshold');
$default_url      = get_option('review_wall_default_url');

// Define an array of setting keys
$settings_keys = [
    'review_wall_title_step_1',
    'review_wall_description_step_1',
    'review_wall_btn_step_1',
    'review_wall_title_step_2',
    'review_wall_description_step_2',
    'review_wall_comment_label',
    'review_wall_email_label',
    'review_wall_btn_step_2',
    'review_wall_title_step_3',
    'review_wall_btn_step_3',
];

// Load all settings into an associative array
$settings = [];
foreach ($settings_keys as $key) {
    $settings[$key] = get_option($key);
}

$steps = [
    [
        'title'       => __('Step 1 - Initial Rating Screen', 'review-wall'),
        'description' => __('Customize texts for the first screen where users rate their experience.', 'review-wall'),
        'fields' => [
            'review_wall_title_step_1'       => __('Title', 'review-wall'),
            'review_wall_description_step_1' => __('Description', 'review-wall'),
            'review_wall_btn_step_1'         => __('Button Text', 'review-wall'),
        ]
    ],
    [
        'title'       => __('Step 2 - Feedback Form (Negative Rating)', 'review-wall'),
        'description' => __('Customize texts for the second screen shown when a user gives a rating below the threshold.', 'review-wall'),
        'fields' => [
            'review_wall_title_step_2'       => __('Title', 'review-wall'),
            'review_wall_description_step_2' => __('Description', 'review-wall'),
            'review_wall_comment_label'      => __('Comment Field Label', 'review-wall'),
            'review_wall_email_label'        => __('Email Field Label', 'review-wall'),
            'review_wall_btn_step_2'         => __('Button Text', 'review-wall'),
        ]
    ],
    [
        'title'       => __('Step 3 - Thank You Screen', 'review-wall'),
        'description' => __('Customize texts for the final screen shown after submitting feedback.', 'review-wall'),
        'fields' => [
            'review_wall_title_step_3' => __('Title', 'review-wall'),
            'review_wall_btn_step_3'   => __('Button Text', 'review-wall'),
        ]
    ]
];

// Modals
require_once REVIEWS_WALL_PATH . 'includes/views/edit-url-modal.php';
require_once REVIEWS_WALL_PATH . 'includes/views/delete-url-modal.php';

?>

<div class="wrap review-wall-admin-wrap">
    <h1><?php esc_html_e('Review Wall Settings', 'review-wall'); ?></h1>

    <div class="review-wall-admin-container">
        <div class="review-wall-section">
            <h2><?php esc_html_e('Global Settings', 'review-wall'); ?></h2>

            <form class="review-wall-form" method="post" action="">
                <?php wp_nonce_field('review_wall_action', 'review_wall_nonce'); ?>
                <input type="hidden" name="review_wall_submit_action" value="save_global_settings">

                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="review_wall_logo"><?php esc_html_e('Logo', 'review-wall'); ?></label>
                        </th>
                        <td>
                            <input type="hidden" id="review_wall_logo" name="review_wall_logo" value="<?php echo esc_attr($logo); ?>">
                            <button type="button" class="button" id="upload_logo_button"><?php esc_html_e('Upload Logo', 'review-wall'); ?></button>

                            <?php if (!empty($logo)): ?>
                                <button type="button" class="button" id="remove_logo_button"><?php esc_html_e('Remove Logo', 'review-wall'); ?></button>
                            <?php endif; ?>

                            <div id="logo_preview">
                                <?php if (!empty($logo)): ?>
                                    <?php echo wp_get_attachment_image($logo, 'medium', false, ['style' => 'max-width: 200px;']); ?>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="review_wall_rating_threshold"><?php esc_html_e('Rating Threshold', 'review-wall'); ?></label>
                        </th>
                        <td>
                            <select name="review_wall_rating_threshold" id="review_wall_rating_threshold">
                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                    <option value="<?php echo esc_attr($i); ?>" <?php selected($rating_threshold, $i); ?>><?php echo esc_html($i); ?></option>
                                <?php endfor; ?>
                            </select>
                            <p class="description"><?php esc_html_e('Redirect to Google Review at or above this rating.', 'review-wall'); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="review_wall_default_url"><?php esc_html_e('Default Google Review URL', 'review-wall'); ?></label>
                        </th>
                        <td>
                            <input type="url" name="review_wall_default_url" id="review_wall_default_url" class="regular-text" value="<?php echo esc_attr($default_url); ?>" required>
                            <p class="description">
                                <?php esc_html_e('Enter the URL of your primary Google review page here.', 'review-wall'); ?>
                                <br>
                                <?php esc_html_e('This URL will be used by default, and as a fallback if an invalid review link is accessed and multiple URLs are configured below.', 'review-wall'); ?>
                                <br>
                                <?php esc_html_e('Example: https://www.google.com/maps/place/YourBusiness/reviews', 'review-wall'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <input type="submit" name="submit" class="button button-primary" value="<?php esc_html_e('Save Settings', 'review-wall'); ?>">
                </p>
            </form>
        </div>

        <div class="review-wall-section">
            <h2><?php esc_html_e('Text Settings', 'review-wall'); ?></h2>

            <form class="review-wall-form" method="post" action="">
                <?php wp_nonce_field('review_wall_action', 'review_wall_nonce'); ?>
                <input type="hidden" name="review_wall_submit_action" value="save_text_settings">

                <?php foreach ($steps as $step) : ?>
                    <div class="review-wall-card">
                        <h3><?php echo esc_html($step['title']); ?></h3>
                        <p class="description"><?php echo esc_html($step['description']); ?></p>

                        <table class="form-table">
                            <?php foreach ($step['fields'] as $key => $label) : ?>
                                <tr>
                                    <th scope="row">
                                        <label for="<?php echo esc_attr($key); ?>">
                                            <?php echo esc_html($label); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <?php if (strpos($key, 'description') !== false) : ?>
                                            <textarea name="<?php echo esc_attr($key); ?>" id="<?php echo esc_attr($key); ?>" class="large-text" rows="3" required><?php echo esc_textarea($settings[$key]); ?></textarea>
                                        <?php else : ?>
                                            <input type="text" name="<?php echo esc_attr($key); ?>" id="<?php echo esc_attr($key); ?>" class="regular-text" value="<?php echo esc_attr($settings[$key]); ?>" required>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </table>
                    </div>
                <?php endforeach; ?>

                <p class="submit">
                    <input type="submit" name="submit" class="button button-primary" value="<?php esc_html_e('Save Text Settings', 'review-wall'); ?>">
                </p>
            </form>
        </div>

        <div class="review-wall-section">
            <h2><?php esc_html_e('Add New URL', 'review-wall'); ?></h2>

            <form class="review-wall-form" method="post" action="">
                <?php wp_nonce_field('review_wall_action', 'review_wall_nonce'); ?>
                <input type="hidden" name="review_wall_submit_action" value="add_url">

                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="url_name"><?php esc_html_e('Name', 'review-wall'); ?></label>
                        </th>
                        <td>
                            <input type="text" name="url_name" id="url_name" class="regular-text" required>
                            <p class="description"><?php esc_html_e('Name for this URL.', 'review-wall'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="url_key"><?php esc_html_e('URL Key', 'review-wall'); ?></label>
                        </th>
                        <td>
                            <input type="text" name="url_key" id="url_key" class="regular-text" placeholder="<?php esc_html_e('Leave empty to auto-generate', 'review-wall'); ?>">
                            <?php /*<button type="button" class="button" id="generate_key_button"><?php esc_html_e('Generate Key', 'review-wall'); ?></button>*/ ?>
                            <p class="description"><?php esc_html_e('This key will be used in the URL parameter.', 'review-wall'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="url_link"><?php esc_html_e('Google Review URL', 'review-wall'); ?></label>
                        </th>
                        <td>
                            <input type="url" name="url_link" id="url_link" class="regular-text" required>
                            <p class="description"><?php esc_html_e('The URL to the Google Review page.', 'review-wall'); ?></p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <input type="submit" name="submit" id="submit" class="button button-primary" value="<?php esc_html_e('Add URL', 'review-wall'); ?>">
                </p>
            </form>
        </div>

        <div class="review-wall-section">
            <h2><?php esc_html_e('Existing URLs', 'review-wall'); ?></h2>

            <?php if (empty($urls)): ?>
                <p><?php esc_html_e('No URLs have been created yet.', 'review-wall'); ?></p>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('ID', 'review-wall'); ?></th>
                            <th><?php esc_html_e('Name', 'review-wall'); ?></th>
                            <th><?php esc_html_e('URL Key', 'review-wall'); ?></th>
                            <th><?php esc_html_e('Google Review URL', 'review-wall'); ?></th>
                            <th><?php esc_html_e('Created', 'review-wall'); ?></th>
                            <th><?php esc_html_e('Actions', 'review-wall'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($urls as $url): ?>
                            <tr>
                                <td><?php echo esc_html($url->id); ?></td>
                                <td>
                                    <?php echo esc_html($url->url_name); ?>
                                    <br>
                                    <!-- <div class="row-actions">
                                    <span class="view"> -->
                                    <a href="<?php echo esc_url(Review_Wall_Page_Manager::get_page_url() . '?id=' . $url->url_key); ?>" target="_blank"><?php esc_html_e('View', 'review-wall'); ?></a>
                                    <!-- </span>
                                    </div> -->
                                </td>
                                <td><?php echo esc_html($url->url_key); ?></td>
                                <td>
                                    <a href="<?php echo esc_url($url->url_link); ?>" target="_blank"><?php echo esc_url($url->url_link); ?></a>
                                </td>
                                <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($url->created_at))); ?></td>
                                <td>
                                    <a href="#" class="edit-url"
                                        data-id="<?php echo esc_attr($url->id); ?>"
                                        data-name="<?php echo esc_attr($url->url_name); ?>"
                                        data-key="<?php echo esc_attr($url->url_key); ?>"
                                        data-url="<?php echo esc_url($url->url_link); ?>">
                                        <?php esc_html_e('Edit', 'review-wall'); ?>
                                    </a> |
                                    <a href="#" class="delete-url"
                                        data-id="<?php echo esc_attr($url->id); ?>">
                                        <?php esc_html_e('Delete', 'review-wall'); ?>
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</div>