<?php

defined('ABSPATH') || exit;

/**
 * Class Review_Wall_Uninstall
 * 
 * Handles complete removal of the plugin data during uninstall.
 */
class Review_Wall_Uninstall
{
    /**
     * Uninstall the plugin completely.
     * This should be called from uninstall.php
     */
    public static function uninstall()
    {
        // Remove Google Reviews cron first
        self::remove_google_cron();

        // Remove database table
        self::remove_database_table();

        // Remove options
        self::remove_options();

        // Remove all review posts
        self::delete_review_posts();

        // Delete the review wall page
        self::delete_review_page();

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Remove the database tables.
     */
    private static function remove_database_table()
    {
        global $wpdb;
        $table_review_wall_urls = $wpdb->prefix . 'review_wall_urls';
        $table_review_wall_google_reviews = $wpdb->prefix . 'review_wall_google_reviews';

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
        $wpdb->query("DROP TABLE IF EXISTS {$table_review_wall_urls}");

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
        $wpdb->query("DROP TABLE IF EXISTS {$table_review_wall_google_reviews}");
    }

    /**
     * Remove plugin options.
     */
    private static function remove_options()
    {
        require_once REVIEWS_WALL_PATH . 'includes/classes/class-review-wall-settings.php';

        foreach (Review_Wall_Settings::get_options() as $option_name => $default_value) {
            delete_option($option_name);
        }
    }

    /**
     * Delete all review posts.
     */
    private static function delete_review_posts()
    {
        // Get all posts with the review_wall post type
        $reviews = get_posts([
            'post_type' => 'review_wall',
            'numberposts' => -1,
            'post_status' => 'any'
        ]);

        // Delete each post permanently (bypass trash)
        foreach ($reviews as $review) {
            wp_delete_post($review->ID, true);
        }
    }

    /**
     * Delete the review wall page.
     */
    private static function delete_review_page()
    {
        // We need to include the page manager class
        require_once REVIEWS_WALL_PATH . 'includes/classes/class-review-wall-page-manager.php';
        Review_Wall_Page_Manager::delete_page();
    }

    /**
     * Delete the review wall page.
     */
    private static function remove_google_cron()
    {
        // We need to include the page manager class
        require_once REVIEWS_WALL_PATH . 'includes/classes/class-review-wall-google-api.php';
        Review_Wall_Google_Api::remove_cron();
    }
}
