<?php

defined('ABSPATH') || exit;

/**
 * Review_Wall_Google_Api
 *
 * Manages Google Reviews API integration and data synchronization.
 */
class Review_Wall_Google_Api
{
    public const CONNECTION_STATUS_OPTION = 'review_wall_google_connection_status';
    public const LAST_SYNC_OPTION = 'review_wall_google_last_sync';
    public const AVERAGE_RATING_OPTION = 'review_wall_google_average_rating';
    public const TOTAL_REVIEWS_OPTION = 'review_wall_google_total_reviews';

    private $db;

    /**
     * Initialize the class and database instance
     */
    public function __construct()
    {
        $this->db = Review_Wall_Helper::get_db_instance();
    }

    /**
     * Initialize hooks and cron functionality
     */
    public static function init()
    {
        add_action('review_wall_daily_sync', [__CLASS__, 'daily_sync_callback']);
        add_action('init', [__CLASS__, 'setup_cron']);
        register_deactivation_hook(__FILE__, [__CLASS__, 'remove_cron']);
    }

    /**
     * Get API key from remote server
     *
     * @return string|false API key or false on error
     */
    public static function get_api_key()
    {
        $response = wp_remote_post(REVIEW_WALL_API_URL . '/wp-json/wc-software-api/v1/google-api-key', [
            'timeout'     => 15,
            'httpversion' => '1.1',
            'headers'     => [
                'User-Agent' => 'WordPress/' . get_bloginfo('version') . '; ' . home_url(),
                'Accept'     => 'application/json',
            ],
            'body'        => ['site_url' => home_url()],
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            return false;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (json_last_error() !== JSON_ERROR_NONE || empty($data['api_key'])) {
            return false;
        }

        return $data['api_key'];
    }

    /**
     * Extract Place ID from Google Reviews URL
     *
     * @param string $url Google Reviews URL
     * @return string|false Place ID or false on error
     */
    public static function extract_place_id($url)
    {
        if (empty($url) || !filter_var($url, FILTER_VALIDATE_URL)) return false;

        $parsed_url = wp_parse_url($url);
        if (!isset($parsed_url['query'])) return false;

        parse_str($parsed_url['query'], $query_params);
        $place_id = isset($query_params['placeid']) ? sanitize_text_field($query_params['placeid']) : false;

        return ($place_id && preg_match('/^[A-Za-z0-9_-]+$/', $place_id) && strlen($place_id) > 10) ? $place_id : false;
    }

    /**
     * Fetch reviews from Google Places API
     *
     * @param string $place_id Google Place ID
     * @return array|false Reviews data or false on error
     */
    public function fetch_reviews($place_id)
    {
        if (empty($place_id)) {
            update_option(self::CONNECTION_STATUS_OPTION, 'invalid_place_id');
            return false;
        }

        $api_key = self::get_api_key();
        if (!$api_key) {
            update_option(self::CONNECTION_STATUS_OPTION, 'no_api_key');
            return false;
        }

        $fields = 'rating,userRatingCount,reviews';
        $url = sprintf(
            'https://places.googleapis.com/v1/places/%s?fields=%s&key=%s',
            urlencode($place_id),
            urlencode($fields),
            urlencode($api_key)
        );

        $response = wp_remote_get($url, ['timeout' => 30, 'httpversion' => '1.1', 'headers' => ['Accept' => 'application/json']]);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            update_option(self::CONNECTION_STATUS_OPTION, 'api_error');
            return false;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (json_last_error() !== JSON_ERROR_NONE || isset($data['error'])) {
            update_option(self::CONNECTION_STATUS_OPTION, 'invalid_response');
            return false;
        }

        update_option(self::CONNECTION_STATUS_OPTION, 'connected');

        return $data;
    }

    /**
     * Sync reviews with database
     *
     * @param string $place_id Google Place ID
     * @return int|false Number of new reviews added or false on error
     */
    public function sync_reviews($place_id)
    {
        $reviews_data = $this->fetch_reviews($place_id);
        if (!$reviews_data) return false;

        // Update global stats
        if (isset($reviews_data['rating'])) {
            update_option(self::AVERAGE_RATING_OPTION, floatval($reviews_data['rating']));
        }
        if (isset($reviews_data['userRatingCount'])) {
            update_option(self::TOTAL_REVIEWS_OPTION, intval($reviews_data['userRatingCount']));
        }

        $existing_hashes = $this->get_existing_review_hashes();
        $new_count = 0;

        if (!empty($reviews_data['reviews']) && is_array($reviews_data['reviews'])) {
            foreach ($reviews_data['reviews'] as $review) {
                if (!$this->validate_review_data($review)) {
                    continue;
                }

                $review_text = $this->extract_review_text($review);
                $hash = md5($review_text . ($review['authorAttribution']['displayName'] ?? ''));

                if (in_array($hash, $existing_hashes, true)) {
                    continue;
                }

                $author_name = sanitize_text_field($review['authorAttribution']['displayName'] ?? 'Anonymous');
                $author_photo_url = !empty($review['authorAttribution']['photoUri'])
                    ? esc_url_raw($review['authorAttribution']['photoUri'])
                    : null;

                $this->db->add_review($author_name, $author_photo_url, $review_text, intval($review['rating']), $hash);

                $new_count++;
            }
        }

        update_option(self::LAST_SYNC_OPTION, current_time('mysql'));
        return $new_count;
    }

    /**
     * Validate review structure
     */
    private function validate_review_data($review)
    {
        return is_array($review) && isset($review['authorAttribution']['displayName']) && !empty($this->extract_review_text($review));
    }

    /**
     * Extract text from review
     */
    private function extract_review_text($review)
    {
        if (isset($review['text']['text'])) return trim($review['text']['text']);
        if (isset($review['text'])) return trim($review['text']);
        return '';
    }

    /**
     * Get existing review hashes to avoid duplicates
     */
    private function get_existing_review_hashes()
    {
        global $wpdb;
        $table = $wpdb->prefix . 'review_wall_google_reviews';
        $results = $wpdb->get_col("SELECT review_hash FROM {$table} WHERE review_hash IS NOT NULL"); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return is_array($results) ? $results : [];
    }

    /**
     * Check if Google Reviews is connected
     */
    public static function is_connected()
    {
        $status = get_option(self::CONNECTION_STATUS_OPTION);
        $link = get_option('review_wall_google_reviews_link');
        return $status === 'connected' && !empty($link);
    }

    /**
     * Get connection status
     */
    public static function get_connection_status()
    {
        return get_option(self::CONNECTION_STATUS_OPTION, 'not_connected');
    }

    /**
     * Get last sync date
     */
    public static function get_last_sync_date()
    {
        return get_option(self::LAST_SYNC_OPTION, false);
    }

    /**
     * Get average rating
     */
    public static function get_average_rating()
    {
        $rating = get_option(self::AVERAGE_RATING_OPTION);
        return (is_numeric($rating) && $rating >= 0 && $rating <= 5) ? floatval($rating) : false;
    }

    /**
     * Get total reviews count
     */
    public static function get_total_reviews()
    {
        $total = get_option(self::TOTAL_REVIEWS_OPTION);
        return (is_numeric($total) && $total >= 0) ? intval($total) : false;
    }

    /**
     * Setup cron job for daily sync
     */
    public static function setup_cron()
    {
        if (!Review_Wall_License::is_valid() || !self::is_connected()) {
            $timestamp = wp_next_scheduled('review_wall_daily_sync');
            if ($timestamp) wp_unschedule_event($timestamp, 'review_wall_daily_sync');
            return;
        }

        if (!wp_next_scheduled('review_wall_daily_sync')) wp_schedule_event(time(), 'daily', 'review_wall_daily_sync');
    }

    /**
     * Remove cron job
     */
    public static function remove_cron()
    {
        $timestamp = wp_next_scheduled('review_wall_daily_sync');
        if ($timestamp) wp_unschedule_event($timestamp, 'review_wall_daily_sync');
    }

    /**
     * Daily cron callback function
     */
    public static function daily_sync_callback()
    {
        $link = get_option('review_wall_google_reviews_link');
        if (empty($link)) {
            return;
        }

        $place_id = self::extract_place_id($link);
        if (!$place_id) {
            return;
        }

        $api = new self();
        $api->sync_reviews($place_id);
    }

    /**
     * Get API stats
     */
    public static function get_api_stats()
    {
        return [
            'is_connected'      => self::is_connected(),
            'last_sync'         => self::get_last_sync_date(),
            'average_rating'    => self::get_average_rating(),
            'total_reviews'     => self::get_total_reviews(),
            'connection_status' => self::get_connection_status()
        ];
    }
}
