<?php

defined('ABSPATH') || exit;

/**
 * Class Review_Wall_Form_Handler
 *
 * Handles form submissions for the admin panel.
 */
class Review_Wall_Form_Handler
{
    /**
     * Process form submissions.
     */
    public static function process_form_submission()
    {
        // Check if submit action is set
        if (isset($_POST['review_wall_submit_action'])) {
            // Verify nonce for security
            $nonce = isset($_POST['review_wall_nonce']) ? sanitize_text_field(wp_unslash($_POST['review_wall_nonce'])) : '';
            if (!$nonce || !wp_verify_nonce($nonce, 'review_wall_action')) {
                wp_die('Security check failed');
            }

            // Get the database instance
            $db = Review_Wall_Helper::get_db_instance();

            // Sanitize the action
            $action = sanitize_text_field(wp_unslash($_POST['review_wall_submit_action']));

            // Handle different form actions
            if ($action === 'add_url') {
                self::add_url($db);
            } elseif ($action === 'update_url' && isset($_POST['url_id'])) {
                self::update_url($db);
            } elseif ($action === 'delete_url' && isset($_POST['url_id'])) {
                self::delete_url($db);
            } elseif ($action === 'save_global_settings') {
                self::save_global_settings();
            } elseif ($action === 'save_text_settings') {
                self::save_text_settings();
            } elseif ($action === 'save_google_reviews_settings') {
                self::save_google_reviews_settings();
            } elseif ($action === 'add_manual_google_review') {
                self::add_manual_google_review($db);
            } elseif ($action === 'save_api_key_settings') {
                self::save_api_key_settings();
            } elseif ($action === 'manual_sync_google_reviews') {
                self::manual_sync_google_reviews();
            } elseif ($action === 'debug_api_key') {
                self::debug_api_key();
            } elseif ($action === 'debug_place_id') {
                self::debug_place_id();
            } elseif ($action === 'debug_google_api') {
                self::debug_google_api();
            } elseif ($action === 'save_widget_settings') {
                self::save_widget_settings();
            }
        }
    }

    /**
     * Add a new URL.
     *
     * @param Review_Wall_DB $db The database instance.
     */
    private static function add_url($db)
    {
        $url_key  = isset($_POST['url_key']) && !empty($_POST['url_key']) ? sanitize_text_field(wp_unslash($_POST['url_key'])) : $db->generate_url_key(); // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $url_link = isset($_POST['url_link']) ? esc_url_raw(wp_unslash($_POST['url_link'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $url_name = isset($_POST['url_name']) ? sanitize_text_field(wp_unslash($_POST['url_name'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing

        if (empty($url_name)) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Please provide a name for the URL.', 'review-wall') . '</p></div>';
            return;
        }

        if (!empty($url_link)) {
            $db->add_url($url_name, $url_key, $url_link);
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('URL added successfully!', 'review-wall') . '</p></div>';
        } else {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Please provide a valid Google review URL.', 'review-wall') . '</p></div>';
        }
    }

    /**
     * Update an existing URL.
     * 
     * @param Review_Wall_DB $db The database instance.
     */
    private static function update_url($db)
    {
        $id       = isset($_POST['url_id']) ? intval($_POST['url_id']) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $url_key  = isset($_POST['url_key']) ? sanitize_text_field(wp_unslash($_POST['url_key'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $url_link = isset($_POST['url_link']) ? esc_url_raw(wp_unslash($_POST['url_link'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $url_name = isset($_POST['url_name']) ? sanitize_text_field(wp_unslash($_POST['url_name'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing

        if (empty($url_name)) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Please provide a name for the URL.', 'review-wall') . '</p></div>';
            return;
        }

        if (!empty($url_key) && !empty($url_link)) {
            $db->update_url($id, $url_name, $url_key, $url_link);
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('URL updated successfully!', 'review-wall') . '</p></div>';
        } else {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Please provide both a URL key and a valid Google review URL.', 'review-wall') . '</p></div>';
        }
    }

    /**
     * Delete a URL.
     * 
     * @param Review_Wall_DB $db The database instance.
     */
    private static function delete_url($db)
    {
        // Get and sanitize the ID
        $id = isset($_POST['url_id']) ? intval(wp_unslash($_POST['url_id'])) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Missing

        // Delete the URL
        $db->delete_url($id);
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('URL deleted successfully!', 'review-wall') . '</p></div>';
    }

    /**
     * Save global settings.
     */
    private static function save_global_settings()
    {
        $settings_to_save = [
            'review_wall_logo'             => 'intval',
            'review_wall_rating_threshold' => 'intval',
            'review_wall_default_url'      => 'esc_url_raw',
        ];

        foreach ($settings_to_save as $setting => $sanitize_function) {
            if (isset($_POST[$setting])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
                $value = wp_unslash($_POST[$setting]); // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                update_option($setting, $sanitize_function($value));
            }
        }

        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Global settings saved successfully!', 'review-wall') . '</p></div>';
    }

    /**
     * Save text settings.
     */
    private static function save_text_settings()
    {
        $settings_to_save = [
            'review_wall_title_step_1'       => 'sanitize_text_field',
            'review_wall_description_step_1' => 'sanitize_textarea_field',
            'review_wall_btn_step_1'         => 'sanitize_text_field',
            'review_wall_title_step_2'       => 'sanitize_text_field',
            'review_wall_description_step_2' => 'sanitize_textarea_field',
            'review_wall_comment_label'      => 'sanitize_text_field',
            'review_wall_email_label'        => 'sanitize_text_field',
            'review_wall_btn_step_2'         => 'sanitize_text_field',
            'review_wall_title_step_3'       => 'sanitize_text_field',
            'review_wall_btn_step_3'         => 'sanitize_text_field',
        ];

        foreach ($settings_to_save as $setting => $sanitize_function) {
            if (isset($_POST[$setting])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
                $value = wp_unslash($_POST[$setting]); // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                update_option($setting, $sanitize_function($value));
            }
        }

        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Text settings saved successfully!', 'review-wall') . '</p></div>';
    }

    /**
     * Save Google Reviews settings
     */
    private static function save_google_reviews_settings()
    {
        $link = isset($_POST['review_wall_google_reviews_link']) ? esc_url_raw(wp_unslash($_POST['review_wall_google_reviews_link'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing

        if (empty($link)) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Please provide a Google Reviews URL.', 'review-wall') . '</p></div>';
            return;
        }

        update_option('review_wall_google_reviews_link', $link);

        $place_id = Review_Wall_Google_Api::extract_place_id($link);

        if (!$place_id) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Could not extract Place ID from this URL.', 'review-wall') . '</p></div>';
            update_option(Review_Wall_Google_Api::CONNECTION_STATUS_OPTION, 'invalid_place_id');
            return;
        }

        $api = new Review_Wall_Google_Api();
        $data = $api->fetch_reviews($place_id);

        if (empty($data['reviews'])) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Could not connect to Google Reviews. Please check your API key or URL.', 'review-wall') . '</p></div>';
            update_option(Review_Wall_Google_Api::CONNECTION_STATUS_OPTION, 'no_reviews');
            return;
        }

        update_option(Review_Wall_Google_Api::CONNECTION_STATUS_OPTION, 'connected');
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Successfully connected to Google Reviews!', 'review-wall') . '</p></div>';

        $api->sync_reviews($place_id);
    }

    /**
     * Add a manual Google review
     */
    private static function add_manual_google_review($db)
    {
        $author = isset($_POST['review_author']) ? sanitize_text_field(wp_unslash($_POST['review_author'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $photo  = isset($_POST['review_author_photo']) ? intval($_POST['review_author_photo']) : null; // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $text   = isset($_POST['review_content']) ? sanitize_textarea_field(wp_unslash($_POST['review_content'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $rating = isset($_POST['review_rating']) ? intval($_POST['review_rating']) : 5; // phpcs:ignore WordPress.Security.NonceVerification.Missing

        if (empty($author) || empty($text) || $rating < 1 || $rating > 5) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Please fill all required fields.', 'review-wall') . '</p></div>';
            return;
        }

        $db->add_review($author, $photo, $text, $rating);
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Review added successfully!', 'review-wall') . '</p></div>';
    }

    /**
     * Save API key settings
     */
    private static function save_api_key_settings()
    {
        $api_key = isset($_POST['review_wall_api_key']) ? sanitize_text_field(wp_unslash($_POST['review_wall_api_key'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing

        if (empty($api_key)) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Please provide an API Key.', 'review-wall') . '</p></div>';
            return;
        }

        update_option('review_wall_api_key', $api_key);

        $status = Review_Wall_License::check_license($api_key);
        update_option('review_wall_license_status', $status);

        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('API Key saved successfully!', 'review-wall') . '</p></div>';
    }

    /**
     * Manual sync of Google Reviews
     */
    private static function manual_sync_google_reviews()
    {
        $link = get_option('review_wall_google_reviews_link');

        if (empty($link)) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Google Reviews URL is not configured.', 'review-wall') . '</p></div>';
            return;
        }

        $place_id = Review_Wall_Google_Api::extract_place_id($link);

        if (!$place_id) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Invalid Place ID.', 'review-wall') . '</p></div>';
            return;
        }

        $api = new Review_Wall_Google_Api();
        $new_count = $api->sync_reviews($place_id);

        // translators: %d: Number of new reviews added during manual sync.
        $message = $new_count !== false ? sprintf(__('Manual sync completed! %d new reviews added.', 'review-wall'), $new_count) : __('Sync failed. Please check your connection.', 'review-wall');

        $class = strpos($message, 'failed') !== false ? 'notice-error' : 'notice-success';
        echo '<div class="notice ' . esc_attr($class) . ' is-dismissible"><p>' . esc_html($message) . '</p></div>';
    }

    /**
     * API Key debug test
     */
    private static function debug_api_key()
    {
        $debug_info = [
            'timestamp' => current_time('mysql'),
            'action'    => 'API Key Test',
        ];

        $api_key = Review_Wall_Google_Api::get_api_key();

        if ($api_key) {
            $debug_info['status'] = 'SUCCESS';
            $debug_info['api_key'] = substr($api_key, 0, 10) . '...' . substr($api_key, -10);
        } else {
            $debug_info['status'] = 'FAILED';
            $debug_info['error']  = 'Could not retrieve API key';
        }

        set_transient('review_wall_debug_results', wp_json_encode($debug_info, true), 300);

        echo '<div class="notice notice-info is-dismissible"><p>' . esc_html__('API Key test completed. Check results below.', 'review-wall') . '</p></div>';
    }

    /**
     * Place ID debug test
     */
    private static function debug_place_id()
    {
        $debug_info = [
            'timestamp' => current_time('mysql'),
            'action'    => 'Place ID Test',
        ];

        $link = get_option('review_wall_google_reviews_link');
        $debug_info['saved_url'] = $link;

        if (empty($link)) {
            $debug_info['status'] = 'FAILED';
            $debug_info['error'] = 'No Google Reviews URL saved';
        } else {
            $place_id = Review_Wall_Google_Api::extract_place_id($link);
            if ($place_id) {
                $debug_info['status'] = 'SUCCESS';
                $debug_info['place_id'] = $place_id;
            } else {
                $debug_info['status'] = 'FAILED';
                $debug_info['error'] = 'Could not extract Place ID';
            }
        }

        set_transient('review_wall_debug_results', wp_json_encode($debug_info, true), 300);

        echo '<div class="notice notice-info is-dismissible"><p>' . esc_html__('Place ID test completed. Check results below.', 'review-wall') . '</p></div>';
    }

    /**
     * Full Google API debug test
     */
    private static function debug_google_api()
    {
        $debug_info = [
            'timestamp' => current_time('mysql'),
            'action'    => 'Full Google API Test',
        ];

        $link = get_option('review_wall_google_reviews_link');

        if (empty($link)) {
            $debug_info['status'] = 'FAILED';
            $debug_info['error'] = 'No Google Reviews URL saved';
            set_transient('review_wall_debug_results', wp_json_encode($debug_info, true), 300);
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('No Google Reviews URL saved.', 'review-wall') . '</p></div>';
            return;
        }

        $place_id = Review_Wall_Google_Api::extract_place_id($link);

        if (!$place_id) {
            $debug_info['status'] = 'FAILED';
            $debug_info['error'] = 'Could not extract Place ID';
            set_transient('review_wall_debug_results', wp_json_encode($debug_info, true), 300);
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Could not extract Place ID.', 'review-wall') . '</p></div>';
            return;
        }

        $debug_info['place_id'] = $place_id;

        $api_key = Review_Wall_Google_Api::get_api_key();

        if (!$api_key) {
            $debug_info['status'] = 'FAILED';
            $debug_info['error'] = 'Could not retrieve API key';
            set_transient('review_wall_debug_results', wp_json_encode($debug_info, true), 300);
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Could not retrieve API key.', 'review-wall') . '</p></div>';
            return;
        }

        $api = new Review_Wall_Google_Api();
        $reviews_data = $api->fetch_reviews($place_id);

        if (!empty($reviews_data['reviews']) && is_array($reviews_data['reviews'])) {
            $review = $reviews_data['reviews'][0];
            $text = '';

            if (!empty($review['text'])) {
                if (is_array($review['text'])) {
                    foreach ($review['text'] as $block) {
                        if (!empty($block['text'])) $text .= $block['text'] . ' ';
                    }
                    $text = trim($text);
                } else {
                    $text = (string)$review['text'];
                }
            }

            $debug_info['status'] = 'SUCCESS';
            $debug_info['sample_review'] = [
                'author' => $review['authorAttribution']['displayName'] ?? 'N/A',
                'rating' => $review['rating'] ?? 'N/A',
                'text_preview' => substr($text, 0, 100) . '...',
            ];

            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Google API test completed. Sample review available.', 'review-wall') . '</p></div>';
        } else {
            $debug_info['status'] = 'FAILED';
            $debug_info['error'] = 'Google API request failed';
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Google API request failed.', 'review-wall') . '</p></div>';
        }

        set_transient('review_wall_debug_results', wp_json_encode($debug_info, true), 300);
    }

    /**
     * Save widget settings.
     */
    private static function save_widget_settings()
    {
        $settings_to_save = [
            'review_wall_widget_title'                => 'sanitize_text_field',
            'review_wall_widget_btn_text'             => 'sanitize_text_field',
            'review_wall_widget_read_more_text'       => 'sanitize_text_field',
            'review_wall_widget_btn_link'             => 'esc_url',
            'review_wall_widget_bg_color'             => 'sanitize_hex_color',
            'review_wall_widget_btn_text_color'       => 'sanitize_hex_color',
            'review_wall_widget_btn_bg_color'         => 'sanitize_hex_color',
            'review_wall_widget_btn_hover_text_color' => 'sanitize_hex_color',
            'review_wall_widget_btn_hover_bg_color'   => 'sanitize_hex_color',
        ];

        foreach ($settings_to_save as $setting => $sanitize_function) {
            if (isset($_POST[$setting])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
                $value = wp_unslash($_POST[$setting]); // phpcs:ignore WordPress.Security.NonceVerification.Missing
                update_option($setting, $sanitize_function($value));
            }
        }

        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Widget settings saved successfully!', 'review-wall') . '</p></div>';
    }
}
