<?php

defined('ABSPATH') || exit;

/**
 * Class Review_Wall_DB
 * 
 * Handle database operations for the Review Wall plugin.
 */
class Review_Wall_DB
{
    /**
     * The table name for the URLs.
     */
    private $table_review_wall_urls;

    /**
     * The table name for the Google reviews.
     */
    private $table_review_wall_google_reviews;

    /**
     * Initialize the class and set its properties.
     */
    public function __construct()
    {
        global $wpdb;
        $this->table_review_wall_urls    = $wpdb->prefix . 'review_wall_urls';
        $this->table_review_wall_google_reviews = $wpdb->prefix . 'review_wall_google_reviews';
    }

    /**
     * Create the database tables.
     */
    public function create_table()
    {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "
        CREATE TABLE {$this->table_review_wall_urls} (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            url_name varchar(255) NOT NULL,
            url_key varchar(50) NOT NULL,
            url_link text NOT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY url_key (url_key)
        ) $charset_collate;

        CREATE TABLE {$this->table_review_wall_google_reviews} (
            id mediumint(8) unsigned NOT NULL AUTO_INCREMENT,
            review_author varchar(255) NOT NULL,
            review_author_photo text DEFAULT NULL,
            review_content text NOT NULL,
            review_rating tinyint(2) unsigned NOT NULL,
            review_hash varchar(64) DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY review_hash (review_hash)
        ) $charset_collate;
        ";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Add a new URL.
     *
     * @param string $url_name The name for the URL.
     * @param string $url_key The unique key for the URL.
     * @param string $url_link The URL to the Google review.
     * @return int|false The number of rows inserted, or false on error.
     */
    public function add_url($url_name, $url_key, $url_link)
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        return $wpdb->insert(
            $this->table_review_wall_urls,
            array(
                'url_name' => $url_name,
                'url_key'  => $url_key,
                'url_link' => $url_link,
            ),
            array('%s', '%s', '%s')
        );
    }

    /**
     * Update an existing URL.
     *
     * @param int    $id The URL ID.
     * @param string $url_name The name for the URL.
     * @param string $url_key The unique key for the URL.
     * @param string $url_link The URL to the Google review.
     * @return int|false The number of rows updated, or false on error.
     */
    public function update_url($id, $url_name, $url_key, $url_link)
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->update(
            $this->table_review_wall_urls,
            array(
                'url_name' => $url_name,
                'url_key' => $url_key,
                'url_link' => $url_link,
            ),
            array('id' => $id),
            array('%s', '%s', '%s'),
            array('%d')
        );
    }

    /**
     * Delete a URL.
     *
     * @param int $id The URL ID.
     * @return int|false The number of rows deleted, or false on error.
     */
    public function delete_url($id)
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->delete(
            $this->table_review_wall_urls,
            array('id' => $id),
            array('%d')
        );
    }

    /**
     * Get all URLs.
     *
     * @return array An array of URLs.
     */
    public function get_urls()
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->get_results("SELECT * FROM {$this->table_review_wall_urls} ORDER BY id DESC");
    }

    /**
     * Get a single URL by ID.
     *
     * @param int $id The URL ID.
     * @return object|null The URL, or null if not found.
     */
    public function get_url($id)
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM {$this->table_review_wall_urls} WHERE id = %d", $id));
    }

    /**
     * Get a single URL by key.
     *
     * @param string $url_key The unique URL key.
     * @return object|null The URL, or null if not found.
     */
    public function get_url_by_key($url_key)
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM {$this->table_review_wall_urls} WHERE url_key = %s", $url_key));
    }

    /**
     * Generate a unique URL key.
     *
     * @param int $length The length of the key to generate. Default is 7.
     * @return string The generated key.
     */
    public function generate_url_key($length = 7)
    {
        // Create a unique hash based on current time
        $hash = md5(uniqid(wp_rand(), true));
        $short_key = substr($hash, 0, $length);

        // Check if the key already exists
        while ($this->get_url_by_key($short_key)) {
            // If key exists, generate a new one with additional randomness
            $hash = md5(uniqid(wp_rand(), true) . time());
            $short_key = substr($hash, 0, $length);
        }

        return $short_key;
    }

    /**
     * Add a new Google review.
     *
     * @param string $name
     * @param string $photo
     * @param string $text
     * @param int    $rating
     * @param string $hash Unique hash for the review
     * @return int|false Inserted row ID on success, false on failure.
     */
    public function add_review($name, $photo, $text, $rating, $hash = null)
    {
        global $wpdb;

        $data = [
            'review_author'       => $name,
            'review_author_photo' => $photo,
            'review_content'      => $text,
            'review_rating'       => $rating,
        ];

        $format = ['%s', '%s', '%s', '%d'];

        if ($hash) {
            $data['review_hash'] = $hash;
            $format[] = '%s';
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->insert(
            $this->table_review_wall_google_reviews,
            $data,
            $format
        );

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Retrieve all Google reviews ordered by newest first.
     *
     * @return array|object[] Array of review objects.
     */
    public function get_google_reviews()
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->get_results("SELECT * FROM {$this->table_review_wall_google_reviews} ORDER BY id DESC");
    }

    /**
     * Delete a Google review by ID.
     *
     * @param int $id The review ID.
     * @return int|false Number of rows deleted or false on failure.
     */
    public function delete_google_review($id)
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->delete(
            $this->table_review_wall_google_reviews,
            array('id' => $id),
            array('%d')
        );
    }
}
