(function ($) {
    'use strict';

    let googleReviewsSwiper;

    $(document).ready(function () {
        initUserPhotoUploader();
        initCopyShortcode();
        initReviewDelete();
        initGoogleReviewAdminSlider();
        initUrlValidation();
    });

    /**
     * Initializes the WordPress media uploader for selecting a user photo.
     */
    function initUserPhotoUploader() {
        let imageFrame;

        $('#upload_author_photo_button').on('click', function (e) {
            e.preventDefault();

            if (imageFrame) {
                imageFrame.open();
                return;
            }

            imageFrame = wp.media({
                title: 'Select or Upload Photo',
                button: {
                    text: 'Use this photo'
                },
                multiple: false
            });

            imageFrame.on('select', function () {
                const attachment = imageFrame.state().get('selection').first().toJSON();
                $('#review_author_photo').val(attachment.id);

                // Show loading spinner first
                $('#author_photo_preview').html('<div class="spinner is-active" style="float: none; margin: 0 0 10px 0;"></div>');

                // Preview
                const img = new Image();
                img.onload = function () {
                    $('#author_photo_preview').html('<img src="' + attachment.url + '" style="max-width: 100px;">');
                };
                img.onerror = function () {
                    $('#author_photo_preview').html('<p class="error">Error loading image</p>');
                };
                img.src = attachment.url;

                // Show remove button
                $('#remove_author_photo_button').show();
            });

            imageFrame.open();
        });

        $('#remove_author_photo_button').on('click', function () {
            $('#review_author_photo').val('');
            $('#author_photo_preview').html('');
            $(this).hide();
        });
    }

    /**
     * Initializes the copy shortcode button functionality.
     */
    function initCopyShortcode() {
        $('#copy_shortcode_button').on('click', function (e) {
            e.preventDefault();

            const $shortcodeInput = $('#review_shortcode');
            let $msg = $('.copy-hortcode-message');

            if ($msg.length === 0) {
                $msg = $('<p class="copy-hortcode-message"></p>').appendTo('.rw-google-shortcode');
            }

            // Select the shortcode text
            $shortcodeInput.select();
            $shortcodeInput[0].setSelectionRange(0, 99999); // For mobile devices

            try {
                // Execute copy command
                const successful = document.execCommand('copy');
                if (successful) {
                    // On success, update message text and classes
                    $msg
                        .text('Shortcode copied to clipboard!')
                        .removeClass('failed')
                        .addClass('success');
                } else {
                    // On failure, update message text and classes
                    $msg
                        .text('Failed to copy shortcode.')
                        .removeClass('success')
                        .addClass('failed');
                }
            } catch (err) {
                // On exception, update message text and classes
                $msg
                    .text('Browser does not support copying.')
                    .removeClass('success')
                    .addClass('failed');
            }

            // Clear selection
            window.getSelection().removeAllRanges();
        });
    }

    /**
     * Initializes the delete functionality for reviews.
     * Binds a click handler to buttons with class `.delete-review`,
     * sends an AJAX request with nonce, and removes the review element on success.
     */
    function initReviewDelete() {
        // Use event delegation to handle dynamically loaded content
        $(document).on('click', '.rw-google-review-card__delete-review', function (e) {
            e.preventDefault();

            // Get the review ID from the data attribute
            const reviewId = $(this).data('review-id');

            // Confirm before deleting
            if (!reviewId || !confirm('Are you sure you want to delete this review?')) {
                return;
            }

            $('.rw-google-reviews-section').addClass('loading-spin');

            // Send AJAX request to delete the review
            $.ajax({
                url: reviewWallParams.ajax_url,
                method: 'POST',
                data: {
                    action: 'review_wall_delete_review',
                    review_id: reviewId,
                    nonce: reviewWallParams.nonce
                },
                success: function (response) {
                    if (response.success) {
                        // Remove the entire review card from the DOM
                        $(`[data-review-id="${reviewId}"]`).closest('.swiper-slide').remove();

                        // If there are no more review cards left, remove the entire review wall section
                        if ($('.rw-google-review-card').length === 0) {
                            console.log(response);
                            $('.rw-google-reviews').remove();
                            $('.rw-google-reviews-section').html(response.data.empty_reviews);
                            $('.rw-google-shortcode').remove();
                        }

                        $('.rw-google-status__reviews').html(response.data.reviews);
                        $('.rw-google-reviews__reviews').html('(' + response.data.reviews + ')');

                        destroyGoogleReviewsSwiper();
                    } else {
                        alert(response.data?.message || 'Failed to delete review.');
                    }
                },
                error: function () {
                    alert('AJAX request failed. Please try again.');
                },
                complete: function () {
                    initGoogleReviewAdminSlider();
                    $('.rw-google-reviews-section').removeClass('loading-spin');
                }
            });
        });
    }

    /**
     * Initializes the Google Reviews Swiper slider in admin page.
     */
    function initGoogleReviewAdminSlider() {
        destroyGoogleReviewsSwiper();

        if ($('.rw-google-reviews__swiper').length === 0 || typeof Swiper === 'undefined') {
            return;
        }

        googleReviewsSwiper = new Swiper('.rw-google-reviews__swiper', {
            slidesPerView: 1,
            slidesPerGroup: 1,
            spaceBetween: 12,
            speed: 700,
            loop: false,
            navigation: {
                nextEl: '.rw-google-reviews__arrow-next',
                prevEl: '.rw-google-reviews__arrow-prev'
            },
            breakpoints: {
                576: {
                    slidesPerView: 2,
                    slidesPerGroup: 2,
                },
                768: {
                    slidesPerView: 3,
                    slidesPerGroup: 3,
                },
                1200: {
                    slidesPerView: 4,
                    slidesPerGroup: 4,
                },
            }
        });
    }

    /**
     * Destroys the existing Google Reviews Swiper instance if it exists.
     */
    function destroyGoogleReviewsSwiper() {
        if (googleReviewsSwiper && typeof googleReviewsSwiper.destroy === 'function') {
            googleReviewsSwiper.destroy(true, true);
            googleReviewsSwiper = null;
        }
    }

    /**
     * Initializes URL validation for Google Reviews input.
     */
    function initUrlValidation() {
        const $urlInput = $('#review_wall_google_place_id');
        const $warning = $('.rw-google-settings__warning');

        if ($urlInput.length && $warning.length) {
            $urlInput.on('input', function () {
                const url = $(this).val();
                $warning.toggle(url && !url.includes('placeid='));
            });
        }
    }
})(jQuery);