<?php

defined('ABSPATH') || exit;

// Process form submissions
Review_Wall_Form_Handler::process_form_submission();

// Get DB instance
$db = Review_Wall_Helper::get_db_instance();

// Get reviews
$reviews = $db->get_google_reviews();

// Get global settings with defaults
$google_average_rating = get_option('review_wall_google_average_rating');
$google_total_reviews  = get_option('review_wall_google_total_reviews');
$google_reviews_link   = get_option('review_wall_google_reviews_link');

// Display debug results if available
$debug_results = get_transient('review_wall_debug_results');

// Get API stats for better overview
$api_stats = Review_Wall_Google_Api::get_api_stats();

// Get widget settings
$review_wall_widget_title                = get_option('review_wall_widget_title');
$review_wall_widget_btn_text             = get_option('review_wall_widget_btn_text');
$review_wall_widget_btn_link             = get_option('review_wall_widget_btn_link');
$review_wall_widget_read_more_text       = get_option('review_wall_widget_read_more_text');
$review_wall_widget_bg_color             = get_option('review_wall_widget_bg_color');
$review_wall_widget_btn_text_color       = get_option('review_wall_widget_btn_text_color');
$review_wall_widget_btn_bg_color         = get_option('review_wall_widget_btn_bg_color');
$review_wall_widget_btn_hover_text_color = get_option('review_wall_widget_btn_hover_text_color');
$review_wall_widget_btn_hover_bg_color   = get_option('review_wall_widget_btn_hover_bg_color');

?>

<div class="wrap review-wall-admin-wrap">
    <h1><?php esc_html_e('Google Reviews', 'review-wall'); ?></h1>

    <div class="review-wall-admin-container">

        <?php if (!empty($google_reviews_link)) : ?>
            <div class="review-wall-section rw-google-status <?php echo $api_stats['is_connected'] ? 'connected' : 'not-connected'; ?>">
                <div class="rw-google-status__content">
                    <h2 class="rw-google-status__title">
                        <div class="rw-google-status__icon">
                            <?php if ($api_stats['is_connected']) : ?>
                                <span class="dashicons dashicons-yes-alt"></span>
                            <?php else : ?>
                                <span class="dashicons dashicons-warning"></span>
                            <?php endif; ?>
                        </div>

                        <?php echo $api_stats['is_connected'] ? esc_html__('Connected', 'review-wall') : esc_html__('Connection Issues', 'review-wall'); ?>
                    </h2>

                    <?php if ($api_stats['is_connected']) : ?>
                        <p class="description">
                            <?php esc_html_e('Successfully connected to Google Reviews!', 'review-wall'); ?>
                        </p>

                        <div class="rw-google-status__data">
                            <div class="rw-google-status__data-item">
                                <p><?php esc_html_e('Average Rating:', 'review-wall'); ?></p>
                                <span>
                                    <?php
                                    if ($api_stats['average_rating']) {
                                        echo esc_html(number_format(floatval($api_stats['average_rating']), 1) . '/5');
                                    } else {
                                        echo esc_html__('Not available', 'review-wall');
                                    }
                                    ?>
                                </span>
                            </div>

                            <div class="rw-google-status__data-item">
                                <p><?php esc_html_e('Total Reviews:', 'review-wall'); ?></p>
                                <span>
                                    <?php
                                    if ($api_stats['total_reviews'] !== false && $api_stats['total_reviews'] !== null) {
                                        echo esc_html(number_format($api_stats['total_reviews']));
                                    } else {
                                        echo esc_html__('Not available', 'review-wall');
                                    }
                                    ?>
                                </span>
                            </div>

                            <div class="rw-google-status__data-item">
                                <p><?php esc_html_e('Last Sync:', 'review-wall'); ?></p>
                                <span>
                                    <?php
                                    if (!empty($api_stats['last_sync'])) {
                                        echo esc_html(wp_date('M j, Y \a\t g:i A', strtotime($api_stats['last_sync'])));
                                    } else {
                                        echo esc_html__('Not available', 'review-wall');
                                    }
                                    ?>
                                </span>
                            </div>

                            <div class="rw-google-status__data-item">
                                <p><?php esc_html_e('Stored Reviews:', 'review-wall'); ?></p>
                                <span class="rw-google-status__reviews">
                                    <?php
                                    if (!empty($reviews)) {
                                        echo esc_html(count($reviews));
                                    } else {
                                        echo esc_html__('Not available', 'review-wall');
                                    }
                                    ?>
                                </span>
                            </div>
                        </div>
                    <?php else :
                        $status_messages = [
                            'no_api_key'       => __('Could not retrieve Google API key. Please contact support.', 'review-wall'),
                            'api_error'        => __('Error connecting to Google API. Please try again.', 'review-wall'),
                            'invalid_response' => __('Invalid response from Google API. Please check your URL.', 'review-wall'),
                            'not_connected'    => __('Not connected. Please save settings to connect.', 'review-wall'),
                            'invalid_place_id' => __('Invalid Place ID in URL.', 'review-wall'),
                        ];

                        $status         = $api_stats['connection_status'];
                        $status_message = isset($status_messages[$status]) ? $status_messages[$status] : __('Connection status unknown.', 'review-wall');
                    ?>
                        <p class="description">
                            <?php echo esc_html($status_message); ?>
                        </p>
                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>

        <div class="review-wall-section rw-google-settings">
            <h2><?php esc_html_e('Google Reviews Setup', 'review-wall'); ?></h2>
            <p class="description"><?php esc_html_e('Enter the link to your Google Reviews page to connect your account. The system will automatically fetch rating and reviews.', 'review-wall'); ?></p>

            <form class="review-wall-form" method="post" action="">
                <?php wp_nonce_field('review_wall_action', 'review_wall_nonce'); ?>
                <input type="hidden" name="review_wall_submit_action" value="save_google_reviews_settings">

                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="review_wall_google_reviews_link"><?php esc_html_e('Google Reviews URL', 'review-wall'); ?></label>
                        </th>
                        <td>
                            <input type="url" name="review_wall_google_reviews_link" id="review_wall_google_reviews_link"
                                value="<?php echo esc_url($google_reviews_link); ?>" class="regular-text"
                                placeholder="https://www.google.com/maps/place/...?placeid=..." />

                            <p class="description">
                                <?php esc_html_e('Paste the link to your business\'s Google reviews page. Make sure it includes the "placeid" parameter.', 'review-wall'); ?>
                            </p>

                            <?php if (!empty($google_reviews_link)) : ?>
                                <p class="rw-google-settings__validation description">
                                    <span class="dashicons dashicons-info"></span>
                                    <span>
                                        <?php
                                        $place_id = Review_Wall_Google_Api::extract_place_id($google_reviews_link);

                                        if ($place_id) {
                                            // translators: %s is the detected Place ID.
                                            printf(esc_html__('Place ID detected: %s', 'review-wall'), '<code>' . esc_html($place_id) . '</code>');
                                        } else {
                                            echo '<span style="color: #d63638;">' . esc_html__('Warning: Could not extract Place ID from this URL.', 'review-wall') . '</span>';
                                        }
                                        ?>
                                    </span>
                                </p>
                            <?php endif; ?>

                            <p class="rw-google-settings__warning description" style="display: none;">
                                <span class="dashicons dashicons-warning"></span>
                                <?php esc_html_e('Warning: This URL does not appear to contain a Place ID parameter.', 'review-wall'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <div class="rw-google-settings__submit-item">
                    <?php submit_button(__('Save', 'review-wall'), 'primary', 'submit', false); ?>

                    <?php if (!empty($google_reviews_link) && $api_stats['is_connected']) : ?>
                        <button class="button button-secondary" type="submit" name="review_wall_submit_action" value="manual_sync_google_reviews">
                            <span class="dashicons dashicons-update"></span>
                            <?php esc_html_e('Sync Reviews Now', 'review-wall'); ?>
                        </button>
                    <?php endif; ?>
                </div>
            </form>
        </div>

        <?php /*<div class="review-wall-section rw-google-debug">
            <h2><?php esc_html_e('Debug & Testing', 'review-wall'); ?></h2>
            <p><?php esc_html_e('Use these tools to diagnose and troubleshoot Google Reviews API integration issues.', 'review-wall'); ?></p>

            <div class="rw-google-debug__forms">
                <form method="post">
                    <?php wp_nonce_field('review_wall_action', 'review_wall_nonce'); ?>
                    <button type="submit" name="review_wall_submit_action" value="debug_api_key" class="button">
                        <span class="dashicons dashicons-admin-network"></span>
                        <?php esc_html_e('Test API Key', 'review-wall'); ?>
                    </button>
                </form>

                <form class="review-wall-form" method="post" action="">
                    <?php wp_nonce_field('review_wall_action', 'review_wall_nonce'); ?>

                    <button type="submit" name="review_wall_submit_action" value="debug_place_id" class="button" <?php echo empty($google_reviews_link) ? 'disabled' : ''; ?>>
                        <span class="dashicons dashicons-location"></span>
                        <?php esc_html_e('Test Place ID', 'review-wall'); ?>
                    </button>
                </form>

                <form class="review-wall-form" method="post" action="">
                    <?php wp_nonce_field('review_wall_action', 'review_wall_nonce'); ?>

                    <button type="submit" name="review_wall_submit_action" value="debug_google_api" class="button" <?php echo empty($google_reviews_link) ? 'disabled' : ''; ?>>
                        <span class="dashicons dashicons-cloud"></span>
                        <?php esc_html_e('Test Google API', 'review-wall'); ?>
                    </button>
                </form>
            </div>

            <?php if ($debug_results) : ?>
                <div class="rw-google-debug__results">
                    <h4>
                        <span class="dashicons dashicons-info"></span>
                        <?php esc_html_e('Debug Results:', 'review-wall'); ?>
                    </h4>

                    <div class="rw-google-debug__results-content">
                        <?php echo esc_html($debug_results); ?>
                    </div>
                </div>

                <?php delete_transient('review_wall_debug_results'); ?>
            <?php endif; ?>
        </div>*/ ?>

        <div class="review-wall-section">
            <h2><?php esc_html_e('Add Manual Google Review', 'review-wall'); ?></h2>

            <form class="review-wall-form" method="post" action="">
                <?php wp_nonce_field('review_wall_action', 'review_wall_nonce'); ?>
                <input type="hidden" name="review_wall_submit_action" value="add_manual_google_review">

                <table class="form-table">
                    <tr>
                        <th><label for="review_author"><?php esc_html_e('Author Name', 'review-wall'); ?></label></th>
                        <td>
                            <input type="text" name="review_author" id="review_author" class="regular-text" required />
                            <p class="description">
                                <?php esc_html_e('Full name of the reviewer.', 'review-wall'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="review_author_photo"><?php esc_html_e('User Photo', 'review-wall'); ?></label></th>
                        <td>
                            <input type="hidden" id="review_author_photo" name="review_author_photo" value="">
                            <button type="button" class="button" id="upload_author_photo_button"><?php esc_html_e('Upload Photo', 'review-wall'); ?></button>
                            <button type="button" class="button" id="remove_author_photo_button" style="display:none;"><?php esc_html_e('Remove Photo', 'review-wall'); ?></button>
                            <div id="author_photo_preview" style="margin-top:10px;"></div>
                            <p class="description">
                                <?php esc_html_e('Optional photo of the reviewer.', 'review-wall'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="review_rating"><?php esc_html_e('Rating', 'review-wall'); ?></label></th>
                        <td>
                            <input type="number" name="review_rating" id="review_rating" min="1" max="5" step="1" required class="small-text" />
                            <p class="description">
                                <?php esc_html_e('Rating from 1 to 5 stars.', 'review-wall'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="review_content"><?php esc_html_e('Review Text', 'review-wall'); ?></label></th>
                        <td>
                            <textarea name="review_content" id="review_content" rows="4" class="large-text" required></textarea>
                            <p class="description">
                                <?php esc_html_e('Full content of the user review.', 'review-wall'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <?php submit_button(__('Add Review', 'review-wall')); ?>
            </form>
        </div>

        <?php if (!empty($reviews)) : ?>
            <div class="review-wall-section rw-google-shortcode">
                <h2><?php esc_html_e('Display Reviews', 'review-wall'); ?></h2>

                <p><?php esc_html_e('Use the shortcode below to display your Google reviews anywhere on your website.', 'review-wall'); ?></p>

                <div class="rw-google-shortcode__item">
                    <input class="regular-text" type="text" readonly value="[review_wall_google_reviews]" id="review_shortcode" />

                    <button type="button" class="button" id="copy_shortcode_button">
                        <span class="dashicons dashicons-admin-page"></span>
                        <?php esc_html_e('Copy', 'review-wall'); ?>
                    </button>
                </div>
            </div>

            <div class="review-wall-section rw-google-widget-settings">
                <h2><?php esc_html_e('Widget Settings', 'review-wall'); ?></h2>

                <p class="description">
                    <?php esc_html_e('Customize the appearance of your Google Reviews widget. You can change background colors and button styles (including hover effects).', 'review-wall'); ?>
                </p>

                <form class="review-wall-form" method="post" action="">
                    <?php wp_nonce_field('review_wall_action', 'review_wall_nonce'); ?>
                    <input type="hidden" name="review_wall_submit_action" value="save_widget_settings">

                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="review_wall_widget_title"><?php esc_html_e('Title', 'review-wall'); ?></label>
                            </th>
                            <td>
                                <input type="text" name="review_wall_widget_title" id="review_wall_widget_title"
                                    value="<?php echo esc_attr($review_wall_widget_title); ?>"
                                    class="regular-text">
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="review_wall_widget_btn_text"><?php esc_html_e('Button Text', 'review-wall'); ?></label>
                            </th>
                            <td>
                                <input type="text" name="review_wall_widget_btn_text" id="review_wall_widget_btn_text"
                                    value="<?php echo esc_attr($review_wall_widget_btn_text); ?>"
                                    class="regular-text">
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="review_wall_widget_btn_link"><?php esc_html_e('Button URL', 'review-wall'); ?></label>
                            </th>
                            <td>
                                <input type="url" name="review_wall_widget_btn_link" id="review_wall_widget_btn_link"
                                    value="<?php echo esc_attr($review_wall_widget_btn_link); ?>"
                                    class="regular-text">
                                <p class="description"><?php esc_html_e('URL that the widget button will link to (review form or Google review page).', 'review-wall'); ?></p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="review_wall_widget_read_more_text"><?php esc_html_e('Read More Text', 'review-wall'); ?></label>
                            </th>
                            <td>
                                <input type="text" name="review_wall_widget_read_more_text" id="review_wall_widget_read_more_text"
                                    value="<?php echo esc_attr($review_wall_widget_read_more_text); ?>"
                                    class="regular-text">
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="review_wall_widget_bg_color"><?php esc_html_e('Background Color', 'review-wall'); ?></label>
                            </th>
                            <td>
                                <input type="text" name="review_wall_widget_bg_color" id="review_wall_widget_bg_color"
                                    value="<?php echo esc_attr($review_wall_widget_bg_color); ?>"
                                    class="rw-color-field">
                                <p class="description"><?php esc_html_e('Background color of the entire widget.', 'review-wall'); ?></p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="review_wall_widget_btn_text_color"><?php esc_html_e('Button Text Color', 'review-wall'); ?></label>
                            </th>
                            <td>
                                <input type="text" name="review_wall_widget_btn_text_color" id="review_wall_widget_btn_text_color"
                                    value="<?php echo esc_attr($review_wall_widget_btn_text_color); ?>"
                                    class="rw-color-field">
                                <p class="description"><?php esc_html_e('Text color inside buttons.', 'review-wall'); ?></p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="review_wall_widget_btn_bg_color"><?php esc_html_e('Button Background Color', 'review-wall'); ?></label>
                            </th>
                            <td>
                                <input type="text" name="review_wall_widget_btn_bg_color" id="review_wall_widget_btn_bg_color"
                                    value="<?php echo esc_attr($review_wall_widget_btn_bg_color); ?>"
                                    class="rw-color-field">
                                <p class="description"><?php esc_html_e('Background color of the buttons.', 'review-wall'); ?></p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="review_wall_widget_btn_hover_text_color"><?php esc_html_e('Button Hover Text Color', 'review-wall'); ?></label>
                            </th>
                            <td>
                                <input type="text" name="review_wall_widget_btn_hover_text_color" id="review_wall_widget_btn_hover_text_color"
                                    value="<?php echo esc_attr($review_wall_widget_btn_hover_text_color); ?>"
                                    class="rw-color-field">
                                <p class="description"><?php esc_html_e('Text color when hovering over a button.', 'review-wall'); ?></p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="review_wall_widget_btn_hover_bg_color"><?php esc_html_e('Button Hover Background Color', 'review-wall'); ?></label>
                            </th>
                            <td>
                                <input type="text" name="review_wall_widget_btn_hover_bg_color" id="review_wall_widget_btn_hover_bg_color"
                                    value="<?php echo esc_attr($review_wall_widget_btn_hover_bg_color); ?>"
                                    class="rw-color-field">
                                <p class="description"><?php esc_html_e('Background color when hovering over a button.', 'review-wall'); ?></p>
                            </td>
                        </tr>
                    </table>

                    <?php submit_button(__('Save Settings', 'review-wall')); ?>
                </form>
            </div>
        <?php endif; ?>

        <div class="review-wall-section rw-google-reviews-section">
            <?php if (!empty($reviews)) : ?>
                <div class="rw-google-reviews">
                    <div class="rw-google-reviews__top">
                        <div class="rw-google-reviews__item">
                            <h2>
                                <?php echo esc_html__('Stored Reviews', 'review-wall'); ?>
                                <span class="rw-google-reviews__reviews"><?php echo esc_html('(' . count($reviews) . ')'); ?></span>
                            </h2>

                            <p>
                                <?php esc_html_e('Reviews are automatically synced daily. You can also trigger a manual sync above.', 'review-wall'); ?>
                            </p>
                        </div>

                        <div class="rw-google-reviews__arrows">
                            <div class="rw-google-reviews__arrow-prev swiper-button-prev">
                                <svg width="10" height="16" viewBox="0 0 10 16" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M8.5 15L1.5 8L8.5 1" stroke="black" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" />
                                </svg>
                            </div>
                            <div class="rw-google-reviews__arrow-next swiper-button-next">
                                <svg width="10" height="16" viewBox="0 0 10 16" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M8.5 15L1.5 8L8.5 1" stroke="black" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" />
                                </svg>
                            </div>
                        </div>
                    </div>

                    <div class="rw-google-reviews__swiper swiper">
                        <div class="swiper-wrapper">
                            <?php foreach ($reviews as $review) : ?>
                                <div class="swiper-slide">
                                    <div class="rw-google-review-card">
                                        <button class="rw-google-review-card__delete-review button" data-review-id="<?php echo esc_attr($review->id); ?>">
                                            <?php esc_html_e('Delete', 'review-wall'); ?>
                                        </button>

                                        <div class="rw-google-review-card__top">
                                            <?php if (!empty($review->review_author_photo)) :
                                                $photo_url = Review_Wall_Helper::get_photo_url($review->review_author_photo);
                                                if ($photo_url) : ?>
                                                    <div class="rw-google-review-card__photo">
                                                        <img src="<?php echo esc_url($photo_url); ?>" alt="<?php echo esc_attr($review->review_author); ?>">
                                                    </div>
                                                <?php endif; ?>
                                            <?php endif; ?>

                                            <p class="rw-google-review-card__author">
                                                <?php echo esc_html($review->review_author); ?>
                                            </p>
                                        </div>

                                        <div class="rw-google-review-card__item">
                                            <?php if ($review->review_rating > 0) : ?>
                                                <div class="rw-google-review-card__rating">
                                                    <?php for ($i = 0; $i < 5; $i++) : ?>
                                                        <?php if ($i < $review->review_rating) : ?>
                                                            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                <path fill-rule="evenodd" clip-rule="evenodd" d="M19.959 8.57966L15.8899 7.98865C15.6596 7.95965 15.4592 7.81464 15.353 7.60063L13.5387 3.94053C13.5357 3.93653 13.5337 3.93253 13.5318 3.92753C13.365 3.61652 13.106 3.35551 12.7934 3.18851C11.9527 2.7575 10.9096 3.08951 10.466 3.94053L8.6498 7.59964C8.54262 7.81364 8.34015 7.95965 8.10096 7.98965L4.04074 8.57966C3.64772 8.63866 3.30333 8.81467 3.04727 9.08666C2.73166 9.41567 2.56294 9.84968 2.57187 10.3077C2.5808 10.7657 2.7664 11.1907 3.09193 11.5047L6.03661 14.3608C6.20037 14.5148 6.27679 14.7498 6.23709 14.9788L5.54136 18.9969C5.48182 19.372 5.54533 19.759 5.71802 20.079C5.93439 20.489 6.29465 20.788 6.73333 20.924C6.89907 20.975 7.06879 21 7.23652 21C7.51242 21 7.78536 20.932 8.03248 20.799L11.6649 18.9069C11.8753 18.7939 12.1284 18.7939 12.3438 18.9089L15.9584 20.793C16.2829 20.973 16.6531 21.038 17.0431 20.978C17.9751 20.824 18.6132 19.934 18.4614 18.9889L17.7667 14.9788C17.726 14.7428 17.8014 14.5118 17.9731 14.3458L20.9088 11.5047C21.1838 11.2377 21.3644 10.8807 21.414 10.4997C21.5361 9.56767 20.8821 8.70567 19.959 8.57966Z" fill="#FDBF01" />
                                                            </svg>
                                                        <?php else : ?>
                                                            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                <path fill-rule="evenodd" clip-rule="evenodd" d="M19.959 8.57966L15.8899 7.98865C15.6596 7.95965 15.4592 7.81464 15.353 7.60063L13.5387 3.94053C13.5357 3.93653 13.5337 3.93253 13.5318 3.92753C13.365 3.61652 13.106 3.35551 12.7934 3.18851C11.9527 2.7575 10.9096 3.08951 10.466 3.94053L8.6498 7.59964C8.54262 7.81364 8.34015 7.95965 8.10096 7.98965L4.04074 8.57966C3.64772 8.63866 3.30333 8.81467 3.04727 9.08666C2.73166 9.41567 2.56294 9.84968 2.57187 10.3077C2.5808 10.7657 2.7664 11.1907 3.09193 11.5047L6.03661 14.3608C6.20037 14.5148 6.27679 14.7498 6.23709 14.9788L5.54136 18.9969C5.48182 19.372 5.54533 19.759 5.71802 20.079C5.93439 20.489 6.29465 20.788 6.73333 20.924C6.89907 20.975 7.06879 21 7.23652 21C7.51242 21 7.78536 20.932 8.03248 20.799L11.6649 18.9069C11.8753 18.7939 12.1284 18.7939 12.3438 18.9089L15.9584 20.793C16.2829 20.973 16.6531 21.038 17.0431 20.978C17.9751 20.824 18.6132 19.934 18.4614 18.9889L17.7667 14.9788C17.726 14.7428 17.8014 14.5118 17.9731 14.3458L20.9088 11.5047C21.1838 11.2377 21.3644 10.8807 21.414 10.4997C21.5361 9.56767 20.8821 8.70567 19.959 8.57966Z" fill="#C3C3C3" />
                                                            </svg>
                                                        <?php endif; ?>
                                                    <?php endfor; ?>
                                                </div>
                                            <?php endif; ?>

                                            <?php if ($review->review_content) : ?>
                                                <p class="rw-google-review-card__content">
                                                    <?php echo esc_html($review->review_content); ?>
                                                </p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            <?php else : ?>
                <?php require_once REVIEWS_WALL_PATH . 'includes/views/empty-reviews.php'; ?>
            <?php endif; ?>
        </div>
    </div>
</div>