<?php

defined('ABSPATH') || exit;

/**
 * Class Review_Wall_Post_Type
 *
 * Registers a custom post type for reviews and adds additional settings.
 */
class Review_Wall_Post_Type
{
    /**
     * Initializes hooks for registering the custom post type and managing columns in the admin area.
     */
    public static function init()
    {
        add_action('init', [self::class, 'register_review_post_type']);
        add_filter('manage_review_wall_posts_columns', [self::class, 'add_reviews_columns']);
        add_action('manage_review_wall_posts_custom_column', [self::class, 'custom_column_content'], 10, 2);
        add_action('add_meta_boxes', [self::class, 'add_review_meta_boxes']);
        add_action('do_meta_boxes', [self::class, 'remove_review_rating_meta_box']);
    }

    /**
     * Registers the custom post type for reviews.
     */
    public static function register_review_post_type()
    {
        $args = array(
            'public' => false,
            'label'  => 'Reviews',
            'menu_icon' => 'dashicons-star-filled',
            'supports' => array('title', 'editor', 'custom-fields'),
            'has_archive' => false,
            'rewrite' => false,
            'publicly_queryable' => false,
            'show_ui' => true,
            'show_in_menu' => false,
            'show_in_admin_bar' => false,
            'show_in_nav_menus' => false,
            'capabilities' => array(
                'create_posts' => 'do_not_allow',
            ),
            'map_meta_cap' => true,
        );

        register_post_type('review_wall', $args);
    }

    /**
     * Adds new columns in the admin area for reviews.
     */
    public static function add_reviews_columns($columns)
    {
        $new_columns = array();
        foreach ($columns as $key => $title) {
            if ($key === 'date') {
                $new_columns['review_wall_rating'] = 'Rating';
                $new_columns['review_wall_email'] = 'Email';
                $new_columns['review_wall_comment'] = 'Comment';
            }
            $new_columns[$key] = $title;
        }
        return $new_columns;
    }

    /**
     * Adds content to the new columns.
     */
    public static function custom_column_content($column_name, $post_id)
    {
        if ($column_name === 'review_wall_rating') {
            $rating = get_post_meta($post_id, 'review_wall_rating', true);
            echo esc_html($rating);
        } elseif ($column_name === 'review_wall_email') {
            $email = get_post_meta($post_id, 'review_wall_email', true);
            echo esc_html($email);
        } elseif ($column_name === 'review_wall_comment') {
            $comment = get_post_field('post_content', $post_id);
            echo esc_html(substr($comment, 0, 50)) . (strlen($comment) > 50 ? '...' : ''); // Display part of the comment
        }
    }

    /**
     * Adds meta boxes for review rating and email.
     */
    public static function add_review_meta_boxes()
    {
        add_meta_box(
            'review_wall_rating_view_meta_box',
            'Rating',
            [self::class, 'review_rating_view_meta_box_html'],
            'review_wall',
            'side',
            'high'
        );

        add_meta_box(
            'review_wall_email_view_meta_box',
            'Email',
            [self::class, 'review_email_view_meta_box_html'],
            'review_wall',
            'side',
            'high'
        );
    }

    /**
     * HTML for the review rating view meta box.
     */
    public static function review_rating_view_meta_box_html($post)
    {
        $rating = get_post_meta($post->ID, 'review_wall_rating', true);
        echo '<p><strong>Rating:</strong> ' . esc_html($rating) . '</p>';
    }

    /**
     * HTML for the email view meta box.
     */
    public static function review_email_view_meta_box_html($post)
    {
        $email = get_post_meta($post->ID, 'review_wall_email', true);
        echo '<p><strong>Email:</strong> ' . esc_html($email) . '</p>';
    }

    /**
     * Removes the default review rating meta box.
     */
    public static function remove_review_rating_meta_box()
    {
        remove_meta_box('review_rating_meta_box', 'review_wall', 'normal');
    }
}
