<?php

defined('ABSPATH') || exit;

/**
 * Class Review_Wall_Page_Manager
 *
 * Handles the creation, activation, deactivation, and deletion of the review wall page.
 */
class Review_Wall_Page_Manager
{
    /**
     * Slug for the review wall page.
     */
    const PAGE_SLUG = 'feedback';

    /**
     * Meta key to identify the review wall page.
     */
    const PAGE_META_KEY = '_review_wall_page';

    /**
     * Initializes hooks.
     */
    public static function init()
    {
        add_filter('template_include', [self::class, 'include_review_wall_template']);

        // 🔑 Handle review page activation/deactivation based on license
        // add_action('init', [self::class, 'toggle_page_status']);
    }

    /**
     * Creates or activates the review wall page.
     */
    public static function create_or_activate_page()
    {
        // Try to find an existing review wall page
        $existing_page = self::find_page();

        if ($existing_page) {
            // Page exists, make sure it's published
            if ($existing_page->post_status !== 'publish') {
                // Update the status to published
                wp_update_post([
                    'ID' => $existing_page->ID,
                    'post_status' => 'publish'
                ]);
            }
            return $existing_page->ID;
        } else {
            // Create a new page
            return self::create_page();
        }
    }

    /**
     * Deactivates the review wall page by setting its status to draft.
     */
    public static function deactivate_page()
    {
        $page = self::find_page();

        if ($page && $page->post_status === 'publish') {
            // Set the page to draft
            wp_update_post([
                'ID' => $page->ID,
                'post_status' => 'draft'
            ]);
        }
    }

    /**
     * Deletes the review wall page permanently.
     */
    public static function delete_page()
    {
        $page = self::find_page();

        if ($page) {
            // Delete the page permanently
            wp_delete_post($page->ID, true);
        }
    }

    /**
     * Finds the review wall page.
     *
     * @return WP_Post|null The page object if found, null otherwise.
     */
    private static function find_page()
    {
        // First try to find by meta key (most reliable)
        $pages = get_posts([
            'post_type' => 'page',
            'meta_key' => self::PAGE_META_KEY, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
            'meta_value' => 1, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
            'post_status' => ['publish', 'draft', 'trash', 'pending', 'future', 'private'],
            'numberposts' => 1
        ]);

        if (!empty($pages)) {
            return $pages[0];
        }

        // Try to find by slug as fallback
        $pages = get_posts([
            'name' => self::PAGE_SLUG,
            'post_type' => 'page',
            'post_status' => ['publish', 'draft', 'trash', 'pending', 'future', 'private'],
            'numberposts' => 1
        ]);

        if (!empty($pages)) {
            // Found by slug, add our meta key for future reference
            update_post_meta($pages[0]->ID, self::PAGE_META_KEY, 1);
            return $pages[0];
        }

        return null;
    }

    /**
     * Creates a new review wall page.
     *
     * @return int Page ID.
     */
    private static function create_page()
    {
        $page_data = [
            'post_title' => __('Feedback', 'review-wall'),
            'post_name' => self::PAGE_SLUG,
            'post_content' => '',
            'post_status' => 'publish',
            'post_type' => 'page',
            'comment_status' => 'closed'
        ];

        // Insert the page
        $page_id = wp_insert_post($page_data);

        if (!is_wp_error($page_id)) {
            // Add meta key to identify this as our page
            update_post_meta($page_id, self::PAGE_META_KEY, 1);
        }

        return $page_id;
    }

    /**
     * Checks if the current page is the review wall page.
     *
     * @return bool True if it's the review wall page, false otherwise.
     */
    public static function is_review_wall_page()
    {
        global $post;

        if (!is_object($post)) {
            return false;
        }

        // Check by meta first (most reliable)
        if (get_post_meta($post->ID, self::PAGE_META_KEY, true)) {
            return true;
        }

        // Check by slug as fallback
        return $post->post_type === 'page' && $post->post_name === self::PAGE_SLUG;
    }

    /**
     * Returns the permalink of the review wall page.
     *
     * @return string|false URL if found, false otherwise.
     */
    public static function get_page_url()
    {
        $page = self::find_page();

        if ($page) {
            return get_permalink($page->ID);
        }

        return false;
    }

    /**
     * Activates or deactivates the review wall page based on license status.
     */
    public static function toggle_page_status()
    {
        if (!Review_Wall_License::is_valid()) {
            // Flush rewrite rules to remove the custom post type URL structure
            flush_rewrite_rules();

            self::deactivate_page();
        }
    }

    /**
     * Custom template loader for the review wall page.
     *
     * @param string $template The path to the template.
     * @return string The path to the new template if conditions are met.
     */
    public static function include_review_wall_template($template)
    {
        if (self::is_review_wall_page()) {
            $new_template = REVIEWS_WALL_PATH . 'templates/review-page-template.php';
            if (file_exists($new_template)) {
                return $new_template;
            }
        }
        return $template;
    }
}
