<?php

defined('ABSPATH') || exit;

/**
 * Class Review_Wall_Ajax_Frontend
 *
 * Handles AJAX requests for the front-end.
 */
class Review_Wall_Ajax_Frontend
{
    /**
     * Initializes the AJAX handler.
     */
    public static function init()
    {
        // Register AJAX action handlers for logged-in and non-logged-in users
        add_action('wp_ajax_submit_review_wall_form', [__CLASS__, 'submit_review_wall_form']);
        add_action('wp_ajax_nopriv_submit_review_wall_form', [__CLASS__, 'submit_review_wall_form']);
    }

    /**
     * AJAX handler for submitting review wall form.
     */
    public static function submit_review_wall_form()
    {
        // Verify nonce for security
        check_ajax_referer('review_wall_nonce', 'nonce');

        // Retrieve and sanitize data from the POST request
        $data = isset($_POST['data']) ? array_map('sanitize_text_field', wp_unslash($_POST['data'])) : [];

        // Process data and store it if valid
        $validation_result = self::validate_data($data);
        if ($validation_result !== true) {
            wp_send_json_error(['message' => $validation_result]);
        }

        $result = self::store_review_data($data);

        // Send success response if data is stored successfully
        if ($result) {
            wp_send_json_success(['message' => 'Thank you for your feedback!']);
        } else {
            // Send error response if data could not be stored
            wp_send_json_error(['message' => 'Unable to submit your review at this time.']);
        }
    }

    /**
     * Validates the review data.
     * 
     * @param array $data The review data.
     * 
     * @return bool|string True if data is valid, error message otherwise.
     */
    private static function validate_data($data)
    {
        if (empty($data['rating'])) {
            return 'Rating is required.';
        }

        if (isset($data['comment']) && empty($data['comment'])) {
            return 'Comment cannot be empty.';
        }

        if (isset($data['email']) && !is_email($data['email'])) {
            return 'Invalid email address.';
        }

        return true;
    }

    /**
     * Stores the review data in the 'review_wall' custom post type.
     * 
     * @param array $data The review data.
     * 
     * @return bool|int The post ID on success, false on failure.
     */
    private static function store_review_data($data)
    {
        // Set the title and content for the review post
        $title = isset($data['title']) ? $data['title'] : 'Review';
        $content = isset($data['comment']) ? $data['comment'] : '';

        // Prepare post data for insertion
        $post_data = [
            'post_title'   => $title,
            'post_content' => $content,
            'post_status'  => 'publish',
            'post_type'    => 'review_wall',
        ];

        // Insert the post into the database
        $post_id = wp_insert_post($post_data);

        // Check if the post was inserted successfully
        if (!is_wp_error($post_id)) {
            // Save the rating in post meta if provided
            if (isset($data['rating'])) {
                update_post_meta($post_id, 'review_wall_rating', intval($data['rating']));
            }

            // Save the email in post meta if provided
            if (isset($data['email'])) {
                update_post_meta($post_id, 'review_wall_email', sanitize_email($data['email']));
            }
            return $post_id;
        }
        return false;
    }
}
