<?php

defined('ABSPATH') || exit;

/**
 * Class Review_Wall_Admin
 *
 * Handles the administrative interface for the Review Wall plugin.
 */
class Review_Wall_Admin
{
    /**
     * Initializes the admin interface by setting up hooks and instances.
     */
    public static function init()
    {
        add_action('admin_menu', [self::class, 'add_admin_menu']);
        add_action('admin_init', [self::class, 'register_settings']);
        add_action('current_screen', [self::class, 'check_reviews_access']);
    }

    /**
     * Adds the main menu and sub-menu items to the WordPress admin interface.
     */
    public static function add_admin_menu()
    {
        add_menu_page(
            __('Review Wall', 'review-wall'),
            __('Review Wall', 'review-wall'),
            'manage_options',
            'review-wall',
            [self::class, 'render_main_page'],
            'dashicons-star-filled',
            6
        );

        add_submenu_page(
            'review-wall',
            __('Dashboard', 'review-wall'),
            __('Dashboard', 'review-wall'),
            'manage_options',
            'review-wall',
            [self::class, 'render_main_page']
        );

        add_submenu_page(
            'review-wall',
            __('Settings', 'review-wall'),
            __('Settings', 'review-wall'),
            'manage_options',
            'review-wall-settings',
            [self::class, 'render_settings_page']
        );

        add_submenu_page(
            'review-wall',
            __('Google Reviews', 'review-wall'),
            __('Google Reviews', 'review-wall'),
            'manage_options',
            'review-wall-google-reviews',
            [self::class, 'render_google_reviews_page']
        );

        add_submenu_page(
            'review-wall',
            __('Reviews', 'review-wall'),
            __('Reviews', 'review-wall'),
            'manage_options',
            'edit.php?post_type=review_wall'
        );
    }

    /**
     * Check access to Reviews page and show license notice if needed
     */
    public static function check_reviews_access()
    {
        $screen = get_current_screen();

        if ($screen && $screen->post_type === 'review_wall' && !Review_Wall_License::is_valid()) {
            add_action('admin_notices', [self::class, 'render_license_notice']);

            add_action('admin_head', function () {
                echo '<style>
                .subsubsub, #posts-filter {
                    display: none !important;
                }
            </style>';
            });
        }
    }

    /**
     * Register plugin settings.
     */
    public static function register_settings()
    {
        require_once REVIEWS_WALL_PATH . 'includes/classes/class-review-wall-settings.php';

        foreach (Review_Wall_Settings::get_options() as $option_name => $default_value) {
            register_setting(
                'review_wall_options',
                $option_name,
                ['sanitize_callback' => 'sanitize_text_field']
            );
        }
    }

    /**
     * Renders the license required notice.
     */
    public static function render_license_notice()
    {
?>
        <div class="notice notice-warning" style="padding: 20px; border-left: 4px solid #dc3232; background: #fef7f7; margin-bottom: 20px;">
            <h2 style="margin-top:0;"><?php esc_html_e('License Required', 'review-wall'); ?></h2>
            <p><?php esc_html_e('To access this page, you need a valid license key. Please enter your API key on the main Review Wall page to activate premium features.', 'review-wall'); ?></p>
            <p>
                <a href="<?php echo esc_url(admin_url('admin.php?page=review-wall')); ?>" class="button button-primary">
                    <?php esc_html_e('Go to Review Wall Main Page', 'review-wall'); ?>
                </a>
            </p>
        </div>
<?php
    }

    /**
     * Renders the main page for the plugin.
     */
    public static function render_main_page()
    {
        require_once REVIEWS_WALL_PATH . 'includes/views/dashboard.php';
    }

    /**
     * Renders the settings page for the plugin.
     */
    public static function render_settings_page()
    {
        if (!Review_Wall_License::is_valid()) {
            self::render_license_notice();
            return;
        }

        require_once REVIEWS_WALL_PATH . 'includes/views/settings.php';
    }

    /**
     * Renders the Google Reviews page.
     */
    public static function render_google_reviews_page()
    {
        if (!Review_Wall_License::is_valid()) {
            self::render_license_notice();
            return;
        }

        require_once REVIEWS_WALL_PATH . 'includes/views/google-reviews.php';
    }
}
