<?php

defined('ABSPATH') || exit;

/**
 * Class Review_Wall_Activation
 * 
 * Fired during plugin activation.
 */
class Review_Wall_Activation
{
    /**
     * Activate the plugin.
     *
     * Creates the database table, registers the post type, creates default review page,
     * and sets default options.
     */
    public static function activate()
    {
        // Create database table
        self::create_database_table();

        // Register post type for reviews
        self::register_post_type();

        // Create or publish the review wall page
        self::create_review_page();

        // Set default options
        self::set_default_options();

        // Setup Google Reviews cron
        self::setup_google_cron();

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Create the database table for redirects.
     */
    private static function create_database_table()
    {
        require_once REVIEWS_WALL_PATH . 'includes/classes/class-review-wall-db.php';
        $db = new Review_Wall_DB();
        $db->create_table();
    }

    /**
     * Register the custom post type.
     */
    private static function register_post_type()
    {
        require_once REVIEWS_WALL_PATH . 'includes/classes/class-review-wall-post-type.php';
        Review_Wall_Post_Type::register_review_post_type();
    }

    /**
     * Create or activate the review wall page.
     */
    private static function create_review_page()
    {
        require_once REVIEWS_WALL_PATH . 'includes/classes/class-review-wall-page-manager.php';
        Review_Wall_Page_Manager::create_or_activate_page();
    }

    /**
     * Setup Google Reviews cron job.
     */
    private static function setup_google_cron()
    {
        require_once REVIEWS_WALL_PATH . 'includes/classes/class-review-wall-google-api.php';
        Review_Wall_Google_Api::setup_cron();
    }

    /**
     * Set default options.
     */
    private static function set_default_options()
    {
        require_once REVIEWS_WALL_PATH . 'includes/classes/class-review-wall-settings.php';

        foreach (Review_Wall_Settings::get_options() as $option_name => $default_value) {
            if (get_option($option_name) === false) {
                add_option($option_name, $default_value);
            }
        }
    }
}
