(function ($) {
    'use strict';

    // Variable to store the currently selected rating
    let selectedRating = 0;

    $(document).ready(function () {
        reviewWallShowStep(1); // Show the first step initially
        reviewWallHandleRating();
        reviewWallHandleContinue();
    });

    /**
     * Adds the 'loading-spin' class to the form to indicate a loading state.
     */
    function reviewWallShowLoading() {
        $('.review-wall-form').addClass('loading-spin');
    }

    /**
     * Removes the 'loading-spin' class from the form to indicate loading finished.
     */
    function reviewWallHideLoading() {
        $('.review-wall-form').removeClass('loading-spin');
    }

    /**
     * Shows the specified step of the form and hides the others.
     * @param {number} step - Step number to show (1, 2, or 3).
     */
    function reviewWallShowStep(step) {
        $('.review-wall-form__step').hide();
        $(`.review-wall-form__step.step-${step}`).show();
    }

    /**
     * Submits the review form data via AJAX.
     * @param {Object} data - Data object to send.
     * @param {Function} [callback] - Optional callback executed on success.
     */
    function reviewWallSubmitForm(data, callback) {
        reviewWallShowLoading();

        $.ajax({
            url: reviewWallParams.ajax_url,
            type: 'POST',
            data: {
                action: 'submit_review_wall_form',
                nonce: reviewWallParams.nonce,
                data: data
            },
            success: function (response) {
                if (response.success) {
                    if (callback) callback();
                } else {
                    alert(response.data.message || 'An error occurred. Please try again.');
                }
                reviewWallHideLoading();
            },
            error: function () {
                alert('An error occurred while processing your request.');
                reviewWallHideLoading();
            }
        });
    }

    /**
     * Handles star rating hover and click events.
     * Updates selectedRating and star visuals accordingly.
     */
    function reviewWallHandleRating() {
        const stars = $('.review-wall-star');

        stars.on('mouseenter', function () {
            const index = $(this).index();
            updateStarClasses(stars, index + 1, 'active');
        });

        stars.on('mouseleave', function () {
            resetStarClasses(stars, selectedRating);
        });

        stars.on('click', function () {
            selectedRating = parseInt($(this).data('rating'), 10);
            updateStarClasses(stars, selectedRating, 'selected');
        });

        $(stars).parent().on('mouseleave', function () {
            resetStarClasses(stars, selectedRating);
        });
    }

    /**
     * Adds or removes the given class to stars up to the count.
     * @param {jQuery} stars - The star elements.
     * @param {number} count - Number of stars to activate.
     * @param {string} className - CSS class to toggle.
     */
    function updateStarClasses(stars, count, className) {
        stars.each(function (i) {
            $(this).toggleClass(className, i < count);
        });
    }

    /**
     * Resets stars to reflect the currently selected rating.
     * @param {jQuery} stars - The star elements.
     * @param {number} selectedRating - The currently selected rating value.
     */
    function resetStarClasses(stars, selectedRating) {
        stars.removeClass('active selected').each(function (i) {
            if (i < selectedRating) {
                $(this).addClass('selected');
            }
        });
    }

    /**
     * Handles the continue button clicks for each step.
     * Reads dynamic constants locally where needed.
     */
    function reviewWallHandleContinue() {
        const ratingThreshold = parseInt($('.review-wall-form').data('rating-threshold'), 10);
        const url = $('.review-wall-form').data('url');
        const urlName = $('.review-wall-form').data('url-name');

        $('.review-wall-form__step.step-1 .review-wall-form__btn').on('click', function () {
            if (selectedRating === 0) {
                alert('Please select a rating.');
                return;
            }

            if (selectedRating <= ratingThreshold) {
                reviewWallShowLoading();
                setTimeout(function () {
                    reviewWallShowStep(2);
                    reviewWallHideLoading();
                }, 200); // Small delay for loading effect
            } else {
                reviewWallSubmitForm({
                    rating: selectedRating,
                    title: urlName
                }, function () {
                    window.location.href = url;
                });
            }
        });

        $('.review-wall-form__step.step-2 .review-wall-form__btn').on('click', function () {
            const comment = $('textarea[name="review_wall_comment"]').val();
            const email = $('input[name="review_wall_email"]').val();

            reviewWallSubmitForm({
                rating: selectedRating,
                title: urlName,
                comment: comment,
                email: email
            }, function () {
                reviewWallShowStep(3);
            });
        });
    }
})(jQuery);