jQuery(document).ready(function ($) {
    // Constants
    const ratingThreshold = parseInt($('.review-wall-form').data('rating-threshold'), 10);
    const url = $('.review-wall-form').data('url');
    const urlName = $('.review-wall-form').data('url-name');
    let selectedRating = 0;

    /**
     * reviewWallShowLoading
     * Adds the 'loading-spin' class to the form to indicate a loading state.
     */
    function reviewWallShowLoading() {
        $('.review-wall-form').addClass('loading-spin');
    }

    /**
     * reviewWallHideLoading
     * Removes the 'loading-spin' class the form to indicate the end of loading state.
     */
    function reviewWallHideLoading() {
        $('.review-wall-form').removeClass('loading-spin');
    }

    /**
     * reviewWallShowStep
     * Shows the specified step and hides others.
     * @param {Number} step - Step number to be displayed (1, 2, or 3).
     */
    function reviewWallShowStep(step) {
        $('.review-wall-form__step').hide();
        $(`.review-wall-form__step.step-${step}`).show();
    }

    /**
     * reviewWallSubmitForm
     * Submits the form data via AJAX.
     * @param {Object} data - The data to be submitted.
     * @param {Function} [callback] - Optional callback function to be called on success.
     */
    function reviewWallSubmitForm(data, callback) {
        // Show loading spinner
        reviewWallShowLoading();

        $.ajax({
            url: reviewWallParams.ajax_url,
            type: 'POST',
            data: {
                action: 'submit_review_wall_form',
                nonce: reviewWallParams.nonce,
                data: data
            },
            success: function (response) {
                if (response.success) {
                    if (callback) callback();
                } else {
                    alert(response.data.message || 'An error occurred. Please try again.');
                }

                // Hide loading spinner
                reviewWallHideLoading();
            },
            error: function () {
                alert('An error occurred while processing your request.');
                reviewWallHideLoading();
            }
        });
    }

    /**
     * reviewWallHandleRating
     * Handles the click and hover event on star ratings.
     */
    function reviewWallHandleRating() {
        const stars = $('.review-wall-star');

        stars.on('mouseenter', function () {
            const index = $(this).index();
            updateStarClasses(stars, index + 1, 'active');
        });

        stars.on('mouseleave', function () {
            resetStarClasses(stars, selectedRating);
        });

        stars.on('click', function () {
            selectedRating = parseInt($(this).data('rating'), 10);
            updateStarClasses(stars, selectedRating, 'selected');
        });

        $(stars).parent().on('mouseleave', function () {
            resetStarClasses(stars, selectedRating);
        });
    }

    /**
     * updateStarClasses
     * Function to update the classes of star rating labels.
     * @param {jQuery} stars - The jQuery object containing star rating elements.
     * @param {number} count - The number of stars to be given the class.
     * @param {string} className - The class to add to the labels.
     */
    function updateStarClasses(stars, count, className) {
        stars.each(function (i) {
            $(this).toggleClass(className, i < count);
        });
    }

    /**
     * resetStarClasses
     * Function to reset the classes of star rating labels to match the selected rating.
     * @param {jQuery} stars - The jQuery object containing star rating elements.
     * @param {number} selectedRating - The currently selected rating.
     */
    function resetStarClasses(stars, selectedRating) {
        stars.removeClass('active selected').each(function (i) {
            if (i < selectedRating) {
                $(this).addClass('selected');
            }
        });
    }

    /**
     * reviewWallHandleContinue
     * Handles the click event on the continue button.
     */
    function reviewWallHandleContinue() {
        $('.review-wall-form__step.step-1 .review-wall-form__btn').on('click', function () {
            if (selectedRating === 0) {
                alert('Please select a rating.');
                return;
            }

            if (selectedRating <= ratingThreshold) {
                // Show loading spinner, then display step 2 after a small delay
                reviewWallShowLoading();
                setTimeout(function () {
                    reviewWallShowStep(2);
                    reviewWallHideLoading();
                }, 200); // 200 millisecond delay for loading simulation
            } else {
                // Submit the form immediately if the rating is above the threshold
                reviewWallSubmitForm({
                    rating: selectedRating,
                    title: urlName
                }, function () {
                    window.location.href = url;
                });
            }
        });

        $('.review-wall-form__step.step-2 .review-wall-form__btn').on('click', function () {
            const comment = $('textarea[name="review_wall_comment"]').val();
            const email = $('input[name="review_wall_email"]').val();

            reviewWallSubmitForm({
                rating: selectedRating,
                title: urlName,
                comment: comment,
                email: email // Submit email
            }, function () {
                reviewWallShowStep(3);
            });
        });
    }

    /**
     * Initializes the review wall form interactions.
     */
    function initReviewWall() {
        reviewWallShowStep(1); // Show the first step initially
        reviewWallHandleRating();
        reviewWallHandleContinue();
    }

    // Initialize when the document is ready
    initReviewWall();
});
