<?php
/**
 * API handler for fetching plugin and theme versions.
 *
 * @package Retro_Version_Manager
 * @version 1.0.4 Updated for theme support.
 */

namespace Retrvema;

if (!defined('ABSPATH')) {
    exit;
}

class Retrvema_API {
    public function register() {
        // No hooks needed for API class yet.
    }

    public function get_versions($slug, $type = 'plugin') {
        if (empty($slug) || !preg_match('/^[a-z0-9-]+$/i', $slug)) {
            return new \WP_Error('invalid_slug', __('Invalid slug.', 'retro-version-manager'));
        }

        // Check cache first
        $cache_key = 'retrvema_' . $type . '_versions_' . $slug;
        $cached_data = get_transient($cache_key);
        if ($cached_data !== false) {
            return $cached_data;
        }

        $url = $type === 'theme' ?
            "https://api.wordpress.org/themes/info/1.2/?action=theme_information&request[slug]=" . urlencode($slug) . "&request[fields][versions]=1" :
            "https://api.wordpress.org/plugins/info/1.2/?action=plugin_information&request[slug]=" . urlencode($slug) . "&request[fields][versions]=1";

        // Retry mechanism for API request
        $attempts = 3;
        $response = false;
        for ($i = 0; $i < $attempts; $i++) {
            $response = wp_remote_get($url, ['timeout' => 30]);
            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                break;
            }
            sleep(1); // Wait before retrying
        }

        if (is_wp_error($response)) {
            return new \WP_Error('api_error', __('Failed to connect to WordPress.org API: ', 'retro-version-manager') . $response->get_error_message());
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);

        if (empty($data)) {
            return new \WP_Error('no_data', __('No data returned from WordPress.org API.', 'retro-version-manager'));
        }

        // Handle theme-specific cases where versions may not be listed
        $versions = [];
        if ($type === 'theme' && isset($data->version) && !isset($data->versions)) {
            // For themes with only a single version (e.g., bundled themes)
            $versions = [$data->version];
        } elseif (isset($data->versions)) {
            $versions_data = is_object($data->versions) ? (array) $data->versions : $data->versions;
            $versions = array_keys(array_filter($versions_data, function($key) {
                return $key !== 'trunk';
            }, ARRAY_FILTER_USE_KEY));
            rsort($versions);
        }

        // Optimize download URL validation: Limit to 10 versions and reduce timeout
        $valid_versions = [];
        $max_versions = 10; // Limit to prevent excessive requests
        foreach (array_slice($versions, 0, $max_versions) as $version) {
            $download_url = sprintf('https://downloads.wordpress.org/%s/%s.%s.zip', $type, $slug, $version);
            $response = wp_remote_head($download_url, ['timeout' => 5, 'sslverify' => false]);
            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                $valid_versions[] = $version;
            }
        }
        $versions = $valid_versions;

        if (empty($versions)) {
           
            return new \WP_Error('no_versions', sprintf(
				 // translators: %s: Type (theme or plugin)
                __('No versions found for this %s. It may be a bundled theme (e.g., twentytwenty) with limited version history. Try downloading from the WordPress core repository.', 'retro-version-manager'),
                $type
            ));
        }

        // Cache the result for 1 hour
        $result = [
            'versions' => $versions,
            'data' => $data,
        ];
        set_transient($cache_key, $result, HOUR_IN_SECONDS);

        return $result;
    }
}
?>