<?php
/**
 * Plugin Name: Restock Notifier For WooCommerce
 * Plugin URI: https://wordpress.org/plugins/restock-notifier-for-woocommerce/
 * Description: Allows customers to subscribe to out-of-stock WooCommerce products and get notified when the product is back in stock.
 * Version: 1.1.0
 * Author: PCPrajapat
 * Author URI: https://devpcprajapat.com/
 * License: GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: restock-notifier-for-woocommerce
 * Domain Path: /languages
 * Requires Plugins: woocommerce
 *
 * @package Restock_Notifier_For_Woocommerce
 */


defined( 'ABSPATH' ) || exit;

// Plugin constants
define( 'RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_VERSION', '1.1.0' );
define( 'RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_FILE', __FILE__ );
define( 'RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// On activation, just set a flag — don't redirect or deactivate directly
register_activation_hook( __FILE__, 'restocknotifierpcprajapat_check_woocommerce_on_activation' );
function restocknotifierpcprajapat_check_woocommerce_on_activation() {
	update_option( 'restocknotifierpcprajapat_activation_check', true );
}

register_activation_hook( __FILE__, 'restocknotifierpcprajapat_plugin_activate' );

function restocknotifierpcprajapat_plugin_activate() {
	if ( ! class_exists( 'WooCommerce' ) ) {
		deactivate_plugins( plugin_basename( __FILE__ ) );
		set_transient( 'restocknotifierpcprajapat_wc_missing_notice', true, 5 );
		wp_redirect( admin_url( 'plugins.php' ) );
		exit;
	}

	require_once plugin_dir_path( __FILE__ ) . 'includes/class-restock-notifier-for-woocommerce-db.php';
	Restock_Notifier_For_Woocommerce_DB::create_table();
}

// On activation, set default settings
register_activation_hook(
	__FILE__,
	function () {
		if ( get_option( 'restocknotifierpcprajapat_settings' ) === false ) {
			// Set a flag to create default settings after text domain is loaded
			update_option( 'restocknotifierpcprajapat_setup_defaults', true );
		}
	}
);

// 👇 Proper WooCommerce check + deactivation + notice (executed on admin_init)
add_action( 'admin_init', 'restocknotifierpcprajapat_maybe_deactivate_if_no_wc' );
function restocknotifierpcprajapat_maybe_deactivate_if_no_wc() {
	if ( get_option( 'restocknotifierpcprajapat_activation_check' ) ) {
		delete_option( 'restocknotifierpcprajapat_activation_check' );

		if ( ! class_exists( 'WooCommerce' ) ) {
			deactivate_plugins( plugin_basename( __FILE__ ) );
			add_action(
				'admin_notices',
				function () {
					echo '<div class="notice notice-error is-dismissible"><p>';
					echo 'Restock Notifier For WooCommerce requires WooCommerce to be installed and active. The plugin has been deactivated.';
					echo '</p></div>';
				}
			);
		}
	}
}

// Runtime: load plugin only if WooCommerce active
add_action( 'plugins_loaded', 'restocknotifierpcprajapat_init_plugin', 11 );
function restocknotifierpcprajapat_init_plugin() {
	if ( ! class_exists( 'WooCommerce' ) ) {
		return;
	}

	require_once RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'includes/class-restock-notifier-for-woocommerce-loader.php';
	new Restock_Notifier_For_Woocommerce_Loader();
}

// Set up default settings after text domain is loaded
add_action( 'init', 'restocknotifierpcprajapat_setup_default_settings', 20 );
function restocknotifierpcprajapat_setup_default_settings() {
	if ( get_option( 'restocknotifierpcprajapat_setup_defaults' ) ) {
		$templates = restocknotifierpcprajapat_get_default_email_templates();

		add_option(
			'restocknotifierpcprajapat_settings',
			array(
				'restocknotifierpcprajapat_enable_plugin'  => 1,
				'restocknotifierpcprajapat_display_form_after' => 'after_price',
				'restocknotifierpcprajapat_success_message' => esc_html__( "You'll be notified when this product is back in stock.", 'restock-notifier-for-woocommerce' ),
				'restocknotifierpcprajapat_form_bg_color_text' => '#ffffff',
				'restocknotifierpcprajapat_form_text_color_text' => '#333333',
				'restocknotifierpcprajapat_form_width'     => '100%',
				'restocknotifierpcprajapat_form_border_size' => '1',
				'restocknotifierpcprajapat_form_border_color_text' => '#dddddd',
				'restocknotifierpcprajapat_form_border_radius' => '4',
				'restocknotifierpcprajapat_margin_top'     => '20',
				'restocknotifierpcprajapat_margin_right'   => '0',
				'restocknotifierpcprajapat_margin_bottom'  => '20',
				'restocknotifierpcprajapat_margin_left'    => '0',
				'restocknotifierpcprajapat_padding_top'    => '20',
				'restocknotifierpcprajapat_padding_right'  => '20',
				'restocknotifierpcprajapat_padding_bottom' => '20',
				'restocknotifierpcprajapat_padding_left'   => '20',
				'restocknotifierpcprajapat_email_field_border_size' => '1',
				'restocknotifierpcprajapat_button_text'    => esc_html__( 'Notify Me', 'restock-notifier-for-woocommerce' ),
				'restocknotifierpcprajapat_button_bg_color_text' => '#2271b1',
				'restocknotifierpcprajapat_button_text_color_text' => '#ffffff',
				'restocknotifierpcprajapat_button_border_size' => '1',
				'restocknotifierpcprajapat_button_border_color_text' => '#2271b1',
				'restocknotifierpcprajapat_button_border_radius_top' => '0',
				'restocknotifierpcprajapat_button_border_radius_right' => '6',
				'restocknotifierpcprajapat_button_border_radius_bottom' => '6',
				'restocknotifierpcprajapat_button_border_radius_left' => '0',
				'restocknotifierpcprajapat_button_hover_bg_color_text' => '#135e96',
				'restocknotifierpcprajapat_button_hover_text_color_text' => '#ffffff',
				'restocknotifierpcprajapat_email_subject'  => esc_html__( '%product_name% is back in stock.', 'restock-notifier-for-woocommerce' ),
				'restocknotifierpcprajapat_email_body'     => $templates['subscription'],
				'restocknotifierpcprajapat_email_body_stock_in' => $templates['stock_in'],
				'restocknotifierpcprajapat_from_name'      => get_bloginfo( 'name' ),
				'restocknotifierpcprajapat_from_email'     => get_option( 'admin_email' ),
				'restocknotifierpcprajapat_test_email'     => get_option( 'admin_email' ),
				'restocknotifierpcprajapat_stock_threshold' => 1,
				'restocknotifierpcprajapat_notification_limit' => 50,
				'restocknotifierpcprajapat_enable_debug'   => 0,
				'restocknotifierpcprajapat_data_cleanup'   => 'never',
			)
		);

		delete_option( 'restocknotifierpcprajapat_setup_defaults' );
	}
}

/**
 * Get default email templates with proper translation support
 */
function restocknotifierpcprajapat_get_default_email_templates() {
	$default_email_body_stock_in = wpautop(
		sprintf(
			'<table width="100%%" cellpadding="0" cellspacing="0" style="background-color: #f0f4f8; padding: 40px 0;">
				<tr><td align="center">
					<table width="600" cellpadding="0" cellspacing="0" style="background-color: #ffffff; border-radius: 12px; box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12); overflow: hidden;">
						<tr>
							<td style="background: linear-gradient(135deg, #3b82f6, #2563eb); padding: 35px 40px; color: #fff;">
								<h1 style="margin: 0; font-size: 28px; font-weight: 700; color:#ffffff;">%s</h1>
							</td>
						</tr>
						<tr>
							<td align="center" style="padding: 40px 40px 20px;">
								<h2 style="margin: 0; font-size: 24px; font-weight: 600; color: #1a202c;">%s <span style="color: #3b82f6;">%%product_name%%</span></h2>
							</td>
						</tr>
						<tr>
							<td style="padding: 0 40px;">
								<p style="font-size: 17px; line-height: 28px; color: #4a5568;">%s</p>
							</td>
						</tr>
						<tr>
							<td style="padding: 30px 40px;">
								<table width="100%%" cellpadding="0" cellspacing="0" style="background-color: #fff; border: 1px solid #e2e8f0; border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.08);">
									<tr>
										<td style="padding: 25px;">
											<table width="100%%" cellpadding="0" cellspacing="0">
												<tr>
													<td width="160" style="vertical-align: top;">
														%%product_image%%
													</td>
													<td style="vertical-align: top; padding-left: 20px;">
														<h3 style="margin: 0 0 10px; font-size: 20px; color: #1a202c;">%%product_name%%</h3>
														<p style="margin: 0 0 20px; font-size: 15px; line-height: 24px; color: #4a5568;">%%product_description%%</p>
														<a href="%%product_link%%" style="display: inline-block; background: linear-gradient(to right, #3b82f6, #2563eb); color: #fff; font-size: 15px; font-weight: 600; text-decoration: none; padding: 12px 24px; border-radius: 6px;">%s</a>
													</td>
												</tr>
											</table>
										</td>
									</tr>
								</table>
							</td>
						</tr>
						<tr>
							<td style="padding: 30px 40px; background-color: #f8fafc; border-top: 1px solid #e2e8f0; text-align: center;">
								<p style="font-size: 15px; color: #4a5568;">%s<br><span style="color: #3b82f6; font-weight: 600;">%s</span></p>
							</td>
						</tr>
					</table>
				</td></tr>
			</table>',
			esc_html__( 'Restock Notifier For WooCommerce', 'restock-notifier-for-woocommerce' ),
			esc_html__( 'Good news! Back in stock:', 'restock-notifier-for-woocommerce' ),
			esc_html__( 'The product you subscribed for is now available. Grab it before it goes out of stock again!', 'restock-notifier-for-woocommerce' ),
			esc_html__( 'View Product', 'restock-notifier-for-woocommerce' ),
			esc_html__( 'Thank you for your patience.', 'restock-notifier-for-woocommerce' ),
			esc_html__( 'The Team', 'restock-notifier-for-woocommerce' )
		)
	);

	$default_email_body = wpautop(
		sprintf(
			'<table width="100%%" cellpadding="0" cellspacing="0" style="background-color: #f0f4f8; padding: 40px 0;">
				<tr><td align="center">
					<table width="600" cellpadding="0" cellspacing="0" style="background-color: #ffffff; border-radius: 12px; box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12); overflow: hidden;">
						<tr>
							<td style="background: linear-gradient(135deg, #3b82f6, #2563eb); padding: 35px 40px; color: #fff;">
								<h1 style="margin: 0; font-size: 28px; font-weight: 700; color:#ffffff;">%s</h1>
							</td>
						</tr>
						<tr>
							<td align="center" style="padding: 40px 40px 20px;">
								<h2 style="margin: 0; font-size: 24px; font-weight: 600; color: #1a202c;">%s <span style="color: #3b82f6;">%%product_name%%</span></h2>
							</td>
						</tr>
						<tr>
							<td style="padding: 0 40px;">
								<p style="font-size: 17px; line-height: 28px; color: #4a5568;">%s</p>
							</td>
						</tr>
						<tr>
							<td style="padding: 30px 40px;">
								<table width="100%%" cellpadding="0" cellspacing="0" style="background-color: #fff; border: 1px solid #e2e8f0; border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.08);">
									<tr>
										<td style="padding: 25px;">
											<table width="100%%" cellpadding="0" cellspacing="0">
												<tr>
													<td width="160" style="vertical-align: top;">
														%%product_image%%
													</td>
													<td style="vertical-align: top; padding-left: 20px;">
														<h3 style="margin: 0 0 10px; font-size: 20px; color: #1a202c;">%%product_name%%</h3>
														<p style="margin: 0 0 20px; font-size: 15px; line-height: 24px; color: #4a5568;">%%product_description%%</p>
														<a href="%%product_link%%" style="display: inline-block; background: linear-gradient(to right, #3b82f6, #2563eb); color: #fff; font-size: 15px; font-weight: 600; text-decoration: none; padding: 12px 24px; border-radius: 6px;">%s</a>
													</td>
												</tr>
											</table>
										</td>
									</tr>
								</table>
							</td>
						</tr>
						<tr>
							<td style="padding: 30px 40px; background-color: #f8fafc; border-top: 1px solid #e2e8f0; text-align: center;">
								<p style="font-size: 15px; color: #4a5568;">%s<br><span style="color: #3b82f6; font-weight: 600;">%s</span></p>
							</td>
						</tr>
					</table>
				</td></tr>
			</table>',
			esc_html__( 'Restock Notifier For WooCommerce', 'restock-notifier-for-woocommerce' ),
			esc_html__( 'Subscribed for:', 'restock-notifier-for-woocommerce' ),
			esc_html__( 'Thank you for subscribing to stock notifications. We\'ll notify you as soon as the product is back in stock.', 'restock-notifier-for-woocommerce' ),
			esc_html__( 'View Product', 'restock-notifier-for-woocommerce' ),
			esc_html__( 'Best regards,', 'restock-notifier-for-woocommerce' ),
			esc_html__( 'The Team', 'restock-notifier-for-woocommerce' )
		)
	);

	return array(
		'stock_in'     => $default_email_body_stock_in,
		'subscription' => $default_email_body,
	);
}

// Add settings link to plugin page
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'restocknotifierpcprajapat_add_settings_link' );
function restocknotifierpcprajapat_add_settings_link( $links ) {
	$settings_link = '<a href="' . admin_url( 'admin.php?page=restock-notifier-for-woocommerce' ) . '">Settings</a>';
	array_unshift( $links, $settings_link );
	return $links;
}
