<?php
/**
 * Loader Class for Restock Notifier For WooCommerce
 *
 * Initializes core plugin classes and registers necessary WordPress hooks and actions.
 * Acts as the main controller to wire together all plugin functionalities.
 *
 * @package Restock_Notifier_For_Woocommerce
 * @subpackage Core
 * @version 1.0.0
 */

defined( 'ABSPATH' ) || exit;

class Restock_Notifier_For_Woocommerce_Loader {

	public function __construct() {
		$this->load_dependencies();

		register_activation_hook( RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_FILE, array( 'Restock_Notifier_For_Woocommerce_DB', 'create_table' ) );

		$settings = get_option( 'restocknotifierpcprajapat_settings', array() );

		if ( isset( $settings['restocknotifierpcprajapat_enable_plugin'] ) && $settings['restocknotifierpcprajapat_enable_plugin'] ) {
			add_action( 'wp', array( $this, 'add_dynamic_form_hook' ) );
			add_action( 'init', array( 'Restock_Notifier_For_Woocommerce_Form', 'register_shortcode' ) );
			add_action( 'init', array( 'Restock_Notifier_For_Woocommerce_Form', 'handle_form_submission' ) );
			add_action( 'admin_init', array( $this, 'handle_clear_logs' ) );
		}

		add_action( 'save_post', array( 'Restock_Notifier_For_Woocommerce_Email', 'check_product_stock_on_save' ), 10, 3 );
		add_action( 'woocommerce_product_set_stock', array( 'Restock_Notifier_For_Woocommerce_Email', 'check_variation_stock_on_set' ), 20, 1 );

		// 🔁 Setup cron jobs at init hook to avoid early translation loading
		add_action( 'init', array( $this, 'setup_cron_jobs' ) );
		add_action( 'restocknotifierpcprajapat_cron_send_batch', array( $this, 'send_cron_batch_handler' ) );

		// WooCommerce notice show for guests via transient (no refresh needed)
		add_action(
			'wp',
			function () {
				if ( ! is_product() ) {
					return;
				}

				global $post;

				if ( ! isset( $post->ID ) ) {
					return;
				}

				$product_id    = $post->ID;
				$remote_ip     = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
				$transient_key = 'restocknotifierpcprajapat_subscribed_' . md5( $remote_ip . $product_id );

				if ( get_transient( $transient_key ) ) {
					wc_add_notice( __( 'You have been subscribed for stock alert.', 'restock-notifier-for-woocommerce' ), 'success' );
					delete_transient( $transient_key );
				}
			},
			5
		);
	}

	public function add_dynamic_form_hook() {
		if ( ! is_product() ) {
			return;
		}

		$settings = get_option( 'restocknotifierpcprajapat_settings', array() );
		$location = $settings['restocknotifierpcprajapat_display_form_after'] ?? 'after_price';

		$priority = 10;

		$map = array(
			'after_title'       => array( 'woocommerce_single_product_summary', 6 ),
			'after_price'       => array( 'woocommerce_single_product_summary', 11 ),
			'after_description' => array( 'woocommerce_single_product_summary', 23 ),
		);

		if ( isset( $map[ $location ] ) ) {
			[ $hook, $priority ] = $map[ $location ];
			add_action( $hook, array( 'Restock_Notifier_For_Woocommerce_Form', 'display_form' ), $priority );
		} elseif ( 'shortcode' === $location ) {
			add_shortcode( 'restock_notifier_for_woocommerce_form', array( 'Restock_Notifier_For_Woocommerce_Form', 'display_form' ) );
		}

		if ( ! empty( $settings['restocknotifierpcprajapat_enable_debug'] ) ) {
			Restock_Notifier_For_Woocommerce_Logger::log( "[WSN Debug] Form hook attached at location: {$location}" );
		}
	}

	private function load_dependencies() {
		require_once RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'includes/admin/class-restock-notifier-for-woocommerce-logger.php';
		new Restock_Notifier_For_Woocommerce_Logger();

		require_once RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'includes/class-restock-notifier-for-woocommerce-db.php';
		require_once RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'includes/class-restock-notifier-for-woocommerce-form.php';
		require_once RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'includes/class-restock-notifier-for-woocommerce-email.php';
		require_once RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'includes/admin/class-restock-notifier-for-woocommerce-admin.php';
		new Restock_Notifier_For_Woocommerce_Admin();
		require_once RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'includes/admin/class-restock-notifier-for-woocommerce-ajax-send-test-email.php';
		Restock_Notifier_For_Woocommerce_AJAX_Send_Test_Email::init();
		require_once RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'includes/admin/class-restock-notifier-for-woocommerce-subscriber-actions.php';
		require_once RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'includes/admin/class-restock-notifier-for-woocommerce-subscriber-helper.php';
		require_once RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'includes/admin/class-restock-notifier-for-woocommerce-subscriber-table.php';
		new Restock_Notifier_For_Woocommerce_Subscriber_Table();
		new Restock_Notifier_For_Woocommerce_Subscriber_Actions();
	}

	public function handle_clear_logs() {
		if ( isset( $_POST['restocknotifierpcprajapat_clear_logs'] ) && current_user_can( 'manage_options' ) ) {
			check_admin_referer( 'restocknotifierpcprajapat_clear_logs_action', 'restocknotifierpcprajapat_clear_logs_nonce' );
			$upload_dir = wp_upload_dir();
			$log_dir    = $upload_dir['basedir'] . '/restock-notifier-for-woocommerce/';
			$log_file   = $log_dir . 'restock-notifier-for-woocommerce-debug.log';
			if ( ! file_exists( $log_dir ) ) {
				wp_mkdir_p( $log_dir );
			}
			if ( file_exists( $log_file ) ) {
				file_put_contents( $log_file, '' );
				// Add a success message
				add_action(
					'admin_notices',
					function () {
						echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Debug logs cleared successfully.', 'restock-notifier-for-woocommerce' ) . '</p></div>';
					}
				);
			} else {
				// Add a notice if file doesn't exist
				add_action(
					'admin_notices',
					function () {
						echo '<div class="notice notice-warning is-dismissible"><p>' . esc_html__( 'No log file found to clear.', 'restock-notifier-for-woocommerce' ) . '</p></div>';
					}
				);
			}
		}
	}

	// 🔁 Setup cron jobs at init hook to avoid early translation loading
	public function setup_cron_jobs() {
		$settings = get_option( 'restocknotifierpcprajapat_settings', array() );

		// Only setup cron if enabled in settings
		if ( isset( $settings['restocknotifierpcprajapat_enable_cron'] ) && $settings['restocknotifierpcprajapat_enable_cron'] ) {
			// Add custom cron schedule for 5 minutes
			add_filter( 'cron_schedules', array( $this, 'add_custom_cron_schedule' ) );

			// Schedule the batch cron if not already scheduled
			if ( ! wp_next_scheduled( 'restocknotifierpcprajapat_cron_send_batch' ) ) {
				wp_schedule_event( time(), 'restock_notifier_for_woocommerce_every_5_minutes', 'restocknotifierpcprajapat_cron_send_batch' );
			}
		}
	}

	// 🔁 Add custom cron schedule for 5 minutes
	public function add_custom_cron_schedule( $schedules ) {
		$schedules['restock_notifier_for_woocommerce_every_5_minutes'] = array(
			'interval' => 60,
			'display'  => __( 'Every 5 Minutes', 'restock-notifier-for-woocommerce' ),
		);
		return $schedules;
	}

	// 🔄 Send the batch cron if not already scheduled
	public function send_cron_batch_handler() {
		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Selecting all rows, no cache needed.
		$product_ids = $wpdb->get_col(
			'SELECT DISTINCT product_id FROM `' . esc_sql( $wpdb->prefix . 'restocknotifierpcprajapat_subscribers' ) . '` WHERE notified = 0'
		);

		if ( empty( $product_ids ) ) {
			return;
		}

		foreach ( $product_ids as $product_id ) {
			$product = wc_get_product( $product_id );
			if ( ! $product || ! $product->managing_stock() || ! $product->is_in_stock() ) {
				continue;
			}

			if ( method_exists( 'Restock_Notifier_For_Woocommerce_Email', 'send_batch_for_product' ) ) {
				Restock_Notifier_For_Woocommerce_Email::send_batch_for_product( $product_id, $product );
			}
		}
	}
}
