<?php
/**
 * Frontend Form Handler for Restock Notifier For WooCommerce
 *
 * Renders the out-of-stock email subscription form and processes the form submission.
 *
 * @package Restock_Notifier_For_Woocommerce
 * @subpackage Form
 * @version 1.0.0
 */

defined( 'ABSPATH' ) || exit;

class Restock_Notifier_For_Woocommerce_Form {

	/**
	 * Display the subscription form.
	 */
	public static function display_form() {
		global $product;

		if ( ! $product ) {
			return;
		}
		$settings = get_option( 'restocknotifierpcprajapat_settings', array() );
		if ( ! empty( $settings['restocknotifierpcprajapat_enable_debug'] ) ) {
			Restock_Notifier_For_Woocommerce_Logger::log( '[WSN Debug] Checking product type for display_form: ' . $product->get_type() );
		}

		if ( $product->is_type( 'variable' ) ) {
			echo '<div id="restocknotifierpcprajapat-notify-container" style="display: none;">';
			wc_get_template(
				'form-out-of-stock.php',
				array(
					'product_id' => $product->get_id(), // if needed
				),
				'',
				RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'templates/'
			);
			echo '</div>';

			if ( ! empty( $settings['restocknotifierpcprajapat_enable_debug'] ) ) {
				Restock_Notifier_For_Woocommerce_Logger::log( '[WSN Debug] Rendered form for variable product ID: ' . $product->get_id() );
			}

			return;
		}

		if ( ! $product->is_in_stock() ) {
			wc_get_template( 'form-out-of-stock.php', array(), '', RESTOCKNOTIFIERPCPRAJAPAT_PLUGIN_DIR . 'templates/' );
			if ( ! empty( $settings['restocknotifierpcprajapat_enable_debug'] ) ) {
				Restock_Notifier_For_Woocommerce_Logger::log( '[WSN Debug] Rendered form for out-of-stock product ID: ' . $product->get_id() );
			}
		}
	}



	/**
	 * Handle form submission, save to DB + send confirmation email.
	 */
	public static function handle_form_submission() {

		if (
			! isset( $_POST['restocknotifierpcprajapat_nonce'] ) ||
			! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['restocknotifierpcprajapat_nonce'] ) ), 'restocknotifierpcprajapat_subscribe_action' )
		) {
			return;
		}

		if (
			! isset( $_POST['restocknotifierpcprajapat_submit'] ) ||
			! isset( $_POST['restocknotifierpcprajapat_email'] ) ||
			! is_email( sanitize_email( wp_unslash( $_POST['restocknotifierpcprajapat_email'] ) ) ) ||
			! isset( $_POST['restocknotifierpcprajapat_product_id'] )
		) {
			return;
		}

		global $wpdb;

		$email      = sanitize_email( wp_unslash( $_POST['restocknotifierpcprajapat_email'] ) );
		$product_id = absint( wp_unslash( $_POST['restocknotifierpcprajapat_product_id'] ) );
		$table_name = $wpdb->prefix . 'restocknotifierpcprajapat_subscribers';

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return; // invalid product
		}

		$settings = get_option( 'restocknotifierpcprajapat_settings', array() );
		if ( ! empty( $settings['restocknotifierpcprajapat_enable_debug'] ) ) {
			Restock_Notifier_For_Woocommerce_Logger::log( "[WSN Debug] Subscription attempt for email: $email and product ID: $product_id" );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Selecting a single row, no cache needed.
		$exists = $wpdb->get_var(
			$wpdb->prepare(
				'SELECT COUNT(*) FROM ' . esc_sql( $table_name ) . ' WHERE email = %s AND product_id = %d AND notified = 0',
				$email,
				$product_id
			)
		);

		if ( ! $exists ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Inserting a single row directly, no cache needed.
			$wpdb->insert(
				$table_name,
				array(
					'email'      => $email,
					'product_id' => $product_id,
				)
			);

			if ( ! empty( $settings['restocknotifierpcprajapat_enable_debug'] ) ) {
				Restock_Notifier_For_Woocommerce_Logger::log( "[WSN Debug] Inserted new subscriber: $email for product ID: $product_id" );
			}

			// Send confirmation email
			// translators: %s is the product name.
			$subject = sprintf( __( 'Subscribed for: %s', 'restock-notifier-for-woocommerce' ), $product->get_name() );

			$settings         = get_option( 'restocknotifierpcprajapat_settings', array() );
			$body_template    = $settings['restocknotifierpcprajapat_email_body'] ?? '';
			$product_image_id = $product->get_image_id();

			$replacements = array(
				'%product_name%'        => $product->get_name(),
				'%product_link%'        => get_permalink( $product_id ),
				'%product_description%' => $product->get_short_description(),
				'%product_image%'       => $product_image_id ? wp_get_attachment_image(
					$product_image_id,
					'medium',
					false,
					array(
						'alt'   => $product->get_name(),
						'style' => 'max-width:100%; height:auto; border-radius:6px;',
					)
				) : '',
				'%store_name%'          => get_bloginfo( 'name' ),
			);

			$final_body   = str_replace( array_keys( $replacements ), array_values( $replacements ), $body_template );
			$wrapped_body = wp_kses_post( wpautop( $final_body ) );

			// Prepare headers
			$headers    = array( 'Content-Type: text/html; charset=UTF-8' );
			$from_name  = $settings['restocknotifierpcprajapat_from_name'] ?? get_bloginfo( 'name' );
			$from_email = $settings['restocknotifierpcprajapat_from_email'] ?? get_option( 'admin_email' );
			$headers[]  = 'From: ' . $from_name . ' <' . $from_email . '>';

			// Send mail
			wc_mail( $email, $subject, $wrapped_body, $headers );

			if ( ! empty( $settings['restocknotifierpcprajapat_enable_debug'] ) ) {
				Restock_Notifier_For_Woocommerce_Logger::log( "[WSN Debug] Confirmation email sent to $email for product ID: $product_id" );
			}
		}

		// Set transient to show success message after redirect
		$remote_ip     = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
		$transient_key = 'restocknotifierpcprajapat_subscribed_' . md5( $remote_ip . $product_id );

		set_transient( $transient_key, true, 60 );

		wp_safe_redirect( get_permalink( $product_id ) );
		exit;
	}

	/**
	 * Register shortcode [restock_notifier_for_woocommerce].
	 */
	public static function register_shortcode() {
		add_shortcode( 'restock-notifier-for-woocommerce', array( __CLASS__, 'shortcode_output' ) );
	}

	/**
	 * Shortcode handler.
	 */
	public static function shortcode_output( $atts = array(), $content = null ) {
		// Only show form if shortcode option is selected
		$settings = get_option( 'restocknotifierpcprajapat_settings', array() );

		if ( ( $settings['restocknotifierpcprajapat_display_form_after'] ?? 'after_price' ) !== 'shortcode' ) {
			return ''; // prevent double output
		}

		ob_start();
		self::display_form();
		return ob_get_clean();
	}
}
