<?php
/**
 * Email Notification Handler for Restock Notifier For WooCommerce
 *
 * @package Restock_Notifier_For_Woocommerce
 * @subpackage Email
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || exit;

class Restock_Notifier_For_Woocommerce_Email {

	public static function maybe_send_email( $product_id, $stock_status, $product ) {
		if ( 'instock' !== $stock_status ) {
			return;
		}

		$settings  = get_option( 'restocknotifierpcprajapat_settings', array() );
		$threshold = isset( $settings['restocknotifierpcprajapat_stock_threshold'] ) ? intval( $settings['restocknotifierpcprajapat_stock_threshold'] ) : 1;

		if ( $product->get_stock_quantity() < $threshold ) {
			return;
		}

		// Let cron handle the sending now
		if ( 1 === (int) $settings['restocknotifierpcprajapat_enable_debug'] && ! empty( $settings['restocknotifierpcprajapat_enable_debug'] ) ) {
			Restock_Notifier_For_Woocommerce_Logger::log( "[WSN Debug] Product ID $product_id is in stock and ready for cron-based notification." );
		}

		// ✅ Immediately send first batch
		self::send_batch_for_product( $product_id, $product );
	}

	public static function send_batch_for_product( $product_id, $product ) {
		global $wpdb;

		$settings  = get_option( 'restocknotifierpcprajapat_settings', array() );
		$threshold = isset( $settings['restocknotifierpcprajapat_stock_threshold'] ) ? intval( $settings['restocknotifierpcprajapat_stock_threshold'] ) : 1;
		$limit     = isset( $settings['restocknotifierpcprajapat_notification_limit'] ) ? intval( $settings['restocknotifierpcprajapat_notification_limit'] ) : 50;
		$limit     = $limit > 0 ? $limit : 50;

		if ( $product->get_stock_quantity() < $threshold ) {
			return;
		}

		$table   = $wpdb->prefix . 'restocknotifierpcprajapat_subscribers';
		$last_id = get_transient( "restocknotifierpcprajapat_batch_pointer_{$product_id}" );
		$last_id = $last_id ? intval( $last_id ) : 0;

		$table = esc_sql( $wpdb->prefix . 'restocknotifierpcprajapat_subscribers' );
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Selecting all rows, no cache needed.
		$subscribers = $wpdb->get_results(
			$wpdb->prepare(
				'SELECT * FROM `' . esc_sql( $wpdb->prefix . 'restocknotifierpcprajapat_subscribers' ) . '` WHERE product_id = %d AND notified = 0 AND id > %d ORDER BY id ASC LIMIT %d',
				$product_id,
				$last_id,
				$limit
			)
		);

		if ( empty( $subscribers ) ) {
			delete_transient( "restocknotifierpcprajapat_batch_pointer_{$product_id}" );
			return;
		}

		foreach ( $subscribers as $subscriber ) {
			$email = sanitize_email( $subscriber->email );

			$subject_template = $settings['restocknotifierpcprajapat_email_subject'] ?? __( 'Back In Stock: %product_name%', 'restock-notifier-for-woocommerce' );
			$subject          = str_replace( '%product_name%', $product->get_name(), $subject_template );

			$body_template    = $settings['restocknotifierpcprajapat_email_body_stock_in'] ?? '';
			$product_image_id = $product->get_image_id();

			$replacements = array(
				'%product_name%'        => $product->get_name(),
				'%product_link%'        => get_permalink( $product_id ),
				'%product_description%' => $product->get_short_description(),
				'%product_image%'       => $product_image_id ? wp_get_attachment_image(
					$product_image_id,
					'medium',
					false,
					array(
						'alt' => $product->get_name(),
					)
				) : '',
				'%store_name%'          => get_bloginfo( 'name' ),
			);

			$final_body = wpautop( str_replace( array_keys( $replacements ), array_values( $replacements ), $body_template ) );
			$headers    = array( 'Content-Type: text/html; charset=UTF-8' );
			$headers[]  = 'From: ' . ( $settings['restocknotifierpcprajapat_from_name'] ?? get_bloginfo( 'name' ) ) . ' <' . ( $settings['restocknotifierpcprajapat_from_email'] ?? get_option( 'admin_email' ) ) . '>';

			wc_mail( $email, $subject, $final_body, $headers );

			if ( 1 === (int) $settings['restocknotifierpcprajapat_enable_debug'] && ! empty( $settings['restocknotifierpcprajapat_enable_debug'] ) ) {
				Restock_Notifier_For_Woocommerce_Logger::log( "[WSN Debug] Stock alert sent to: $email for product: {$product->get_name()}" );
			}

			$table = esc_sql( $wpdb->prefix . 'restocknotifierpcprajapat_subscribers' );

			if ( isset( $settings['restocknotifierpcprajapat_data_cleanup'] ) && 'after_notification' === $settings['restocknotifierpcprajapat_data_cleanup'] ) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Deleting multiple rows directly, no cache needed.
				$wpdb->delete( $table, array( 'id' => $subscriber->id ) );
			} else {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Updating a single row directly, no cache needed.
				$wpdb->update(
					$table,
					array(
						'notified'    => 1,
						'notified_at' => current_time( 'mysql' ),
					),
					array( 'id' => $subscriber->id )
				);
			}

			$last_id = $subscriber->id;
		}

		set_transient( "restocknotifierpcprajapat_batch_pointer_{$product_id}", $last_id, 12 * HOUR_IN_SECONDS );
	}

	public static function check_product_stock_on_save( $post_id, $post, $update ) {
		if ( 'product' !== $post->post_type ) {
			return;
		}

		$product = wc_get_product( $post_id );
		if ( ! $product || ! $product->managing_stock() ) {
			return;
		}

		$settings  = get_option( 'restocknotifierpcprajapat_settings', array() );
		$threshold = isset( $settings['restocknotifierpcprajapat_stock_threshold'] ) ? intval( $settings['restocknotifierpcprajapat_stock_threshold'] ) : 1;

		if ( $product->is_type( 'simple' ) && $product->is_in_stock() && $product->get_stock_quantity() >= $threshold ) {
			self::maybe_send_email( $product->get_id(), 'instock', $product );
		}

		if ( $product->is_type( 'variable' ) ) {
			foreach ( $product->get_children() as $variation_id ) {
				$variation = wc_get_product( $variation_id );
				if ( $variation && $variation->is_in_stock() && $variation->get_stock_quantity() >= $threshold ) {
					self::maybe_send_email( $variation_id, 'instock', $variation );
				}
			}
		}

		if ( $product->is_type( 'grouped' ) ) {
			foreach ( $product->get_children() as $child_id ) {
				$child_product = wc_get_product( $child_id );
				if ( $child_product && $child_product->managing_stock() && $child_product->is_in_stock() && $child_product->get_stock_quantity() >= $threshold ) {
					self::maybe_send_email( $child_id, 'instock', $child_product );
				}
			}
		}
	}

	public static function check_variation_stock_on_set( $stock ) {
		if ( ! is_a( $stock, 'WC_Product_Variation' ) ) {
			return;
		}

		$product = wc_get_product( $stock->get_id() );
		if ( ! $product || ! $product->managing_stock() ) {
			return;
		}

		$settings  = get_option( 'restocknotifierpcprajapat_settings', array() );
		$threshold = isset( $settings['restocknotifierpcprajapat_stock_threshold'] ) ? intval( $settings['restocknotifierpcprajapat_stock_threshold'] ) : 1;

		if ( $product->is_in_stock() && $product->get_stock_quantity() >= $threshold ) {
			self::maybe_send_email( $product->get_id(), 'instock', $product );
		}
	}
}
