<?php
/**
 * Admin functionality for Restock Notifier For WooCommerce.
 *
 * @package Restock_Notifier_For_Woocommerce
 */

defined( 'ABSPATH' ) || exit;

class Restock_Notifier_For_Woocommerce_Admin {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_menu_page' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_public_assets' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	/**
	 * Add plugin settings page in WP Admin.
	 */
	public function add_menu_page() {
		add_menu_page(
			__( 'Restock Notifier For WooCommerce', 'restock-notifier-for-woocommerce' ),
			__( 'Stock Notifier', 'restock-notifier-for-woocommerce' ),
			'manage_woocommerce',
			'restock-notifier-for-woocommerce',
			array( $this, 'render_settings_page' ),
			'dashicons-email-alt',
			56
		);

		$settings = get_option( 'restocknotifierpcprajapat_settings', array() );
	}

	/**
	 * Render the plugin settings page.
	 */
	public function render_settings_page() {
		echo '<div class="restocknotifierpcprajapat-admin bg-gray-50 text-gray-800 font-sans">
			<div class="mx-auto px-4 py-8 pr-10">';
			// Tabs layout (header, tabs, footer)
			include plugin_dir_path( __FILE__ ) . 'views/layout-header.php';
			include plugin_dir_path( __FILE__ ) . 'views/tabs-list.php';
			echo '<div class="flex flex-col xl:flex-row w-full">
				<div class="lg:w-full xl:w-[65%] 2xl:w-[75%] xl:pr-5">';
				// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Checking WP core flag; nonce is handled by WordPress options API.
		if ( isset( $_GET['settings-updated'] ) && 'true' === $_GET['settings-updated'] ) {
			echo '<div id="settings-saved" class="notice notice-success is-dismissible bg-white border-l-4 border-green-500 shadow-md rounded-lg px-6 py-2 mb-6 transform transition-all duration-500 ease-in-out">
									<p class="text-sm font-medium text-green-800">';
					esc_html_e( 'Settings saved successfully!', 'restock-notifier-for-woocommerce' );
					echo '</p>
								</div>';

			$settings = get_option( 'restocknotifierpcprajapat_settings', array() );
			if ( ! empty( $settings['restocknotifierpcprajapat_enable_debug'] ) ) {
				Restock_Notifier_For_Woocommerce_Logger::log( '[WSN Debug] Settings updated and saved.' );
			}
		}
					include plugin_dir_path( __FILE__ ) . 'views/tab-general.php';
					include plugin_dir_path( __FILE__ ) . 'views/tab-email.php';
					include plugin_dir_path( __FILE__ ) . 'views/tab-subscribers.php';
					include plugin_dir_path( __FILE__ ) . 'views/tab-design.php';
					include plugin_dir_path( __FILE__ ) . 'views/tab-advanced.php';
					include plugin_dir_path( __FILE__ ) . 'views/tab-logs.php';
				echo '</div>
				<div class="lg:w-full xl:w-[35%] 2xl:w-[25%] mt-5 xl:mt-0 xl:pl-5 h-fit sticky top-10">';
		?>
						<!-- Preview Section -->
						<div class="bg-white rounded-lg shadow-card overflow-hidden">
							<div class="gradient-header border-b border-gray-200 px-3 py-2">
								<h3 class="text-lg font-semibold flex items-center text-gray-800">
									<span class="dashicons dashicons-visibility mr-2 text-primary-600"></span>
									<?php esc_html_e( 'Form Preview', 'restock-notifier-for-woocommerce' ); ?>
								</h3>
							</div>
							<div class="p-6">
								<div class="max-w-md mx-auto p-0">
									<div id="restocknotifierpcprajapat_front_subscriber_form">
										<div class="flex items-center mb-2">
											<svg class="w-5 h-5 text-yellow-500 mr-2" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
												<path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2h-1V9a1 1 0 00-1-1z" clip-rule="evenodd"></path>
											</svg>
											<span class="font-medium" id="form-heading"><?php esc_html_e( 'Out of Stock', 'restock-notifier-for-woocommerce' ); ?></span>
										</div>
										<p class="text-sm mb-3"><?php esc_html_e( 'Get notified when this product is back in stock:', 'restock-notifier-for-woocommerce' ); ?></p>
										<div class="flex">
											<input type="email" id="preview-email-field" placeholder="Your email address" class="flex-1 border preview_email_field border-gray-300 px-3 py-2">
											<button class="px-4 py-3 font-medium notify-btn" id="preview-button"><?php esc_html_e( 'Notify Me', 'restock-notifier-for-woocommerce' ); ?></button>
										</div>
									</div>
									<div class="text-center text-sm text-gray-500">
										<p><?php esc_html_e( 'This is how the notification form will appear on your product pages.', 'restock-notifier-for-woocommerce' ); ?></p>
									</div>
								</div>
							</div>
						</div>
					<?php
					echo '</div>
			</div>';
					include plugin_dir_path( __FILE__ ) . 'views/layout-footer.php';

					echo '</div></div>';
	}


	/**
	 * Load assets only on plugin page in admin.
	 */
	public function enqueue_admin_assets( $hook ) {
		if ( 'toplevel_page_restock-notifier-for-woocommerce' !== $hook ) {
			return;
		}

		wp_enqueue_style(
			'restocknotifierpcprajapat-admin-style',
			plugin_dir_url( __FILE__ ) . '../../assets/css/restock-notifier-for-woocommerce-admin.css',
			array(),
			'1.0'
		);

		// Load WordPress bundled CodeMirror scripts & styles
		wp_enqueue_code_editor(
			array(
				'type' => 'text/css',
			)
		);

		wp_enqueue_script( 'wp-codemirror' );
		wp_enqueue_style( 'wp-codemirror' );

		wp_enqueue_script(
			'restocknotifierpcprajapat-admin-js',
			plugin_dir_url( __FILE__ ) . '../../assets/js/restock-notifier-for-woocommerce-admin.js',
			array( 'wp-codemirror' ),
			'1.0.0',
			true // Load in footer
		);

		// ✅ Localize script with nonce
		wp_localize_script(
			'restocknotifierpcprajapat-admin-js',
			'restocknotifierpcprajapatAdmin',
			array(
				'nonce'              => wp_create_nonce( 'restocknotifierpcprajapat_admin_nonce' ),
				'ajax_url'           => admin_url( 'admin-ajax.php' ),
				'confirm_clear_logs' => __( 'Are you sure you want to clear all debug logs?', 'restock-notifier-for-woocommerce' ),
			)
		);
	}

	/**
	 * Load Assets only on fronted.
	 */
	public function enqueue_public_assets( $hook ) {
		wp_enqueue_style(
			'restocknotifierpcprajapat-public-style',
			plugin_dir_url( __FILE__ ) . '../../assets/css/restock-notifier-for-woocommerce-public.css',
			array(),
			'1.0'
		);

		wp_enqueue_script(
			'restocknotifierpcprajapat-public-js',
			plugin_dir_url( __FILE__ ) . '../../assets/js/restock-notifier-for-woocommerce-public.js',
			array( 'jquery' ),
			'1.0.0',
			true // Load in footer
		);

		wp_localize_script(
			'restocknotifierpcprajapat-public-js',
			'restocknotifierpcprajapat_ajax',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
			)
		);

		$settings = get_option( 'restocknotifierpcprajapat_settings', array() );

		/**
		 * Form CSS
		 */
		$form_css = sprintf(
			'.restocknotifierpcprajapat_front_subscriber_form { background-color: %s; color: %s; max-width: %s; border: %spx solid %s; border-radius: %spx; margin: %spx %spx %spx %spx; padding: %spx %spx %spx %spx; }
			.restocknotifierpcprajapat_email_field { border: %spx solid %s; border-radius: %spx %spx %spx %spx !important; padding: 10px; width:%s; border-right: %s; }
			.restocknotifierpcprajapat-stock-alert-form button { background-color: %s; color: %s; border: %spx solid %s; border-radius: %spx %spx %spx %spx !important; padding: 10px 15px; white-space: nowrap; }
			.restocknotifierpcprajapat-stock-alert-form button:hover { background-color: %s !important; color: %s !important; cursor: pointer; }',
			// Form CSS
			esc_attr( $settings['restocknotifierpcprajapat_form_bg_color_text'] ?? '#fff' ),
			esc_attr( $settings['restocknotifierpcprajapat_form_text_color_text'] ?? '#333' ),
			esc_attr( $settings['restocknotifierpcprajapat_form_width'] ?? '100%' ),
			intval( $settings['restocknotifierpcprajapat_form_border_size'] ?? 1 ),
			esc_attr( $settings['restocknotifierpcprajapat_form_border_color_text'] ?? '#ddd' ),
			intval( $settings['restocknotifierpcprajapat_form_border_radius'] ?? 4 ),
			intval( $settings['restocknotifierpcprajapat_margin_top'] ?? 20 ),
			intval( $settings['restocknotifierpcprajapat_margin_right'] ?? 0 ),
			intval( $settings['restocknotifierpcprajapat_margin_bottom'] ?? 20 ),
			intval( $settings['restocknotifierpcprajapat_margin_left'] ?? 0 ),
			intval( $settings['restocknotifierpcprajapat_padding_top'] ?? 20 ),
			intval( $settings['restocknotifierpcprajapat_padding_right'] ?? 20 ),
			intval( $settings['restocknotifierpcprajapat_padding_bottom'] ?? 20 ),
			intval( $settings['restocknotifierpcprajapat_padding_left'] ?? 20 ),
			// Email field CSS
			intval( $settings['restocknotifierpcprajapat_email_field_border_size'] ?? 1 ),
			esc_attr( $settings['restocknotifierpcprajapat_email_field_border_color_text'] ?? '#ddd' ),
			intval( $settings['restocknotifierpcprajapat_email_field_border_radius_top'] ?? 6 ),
			intval( $settings['restocknotifierpcprajapat_email_field_border_radius_right'] ?? 0 ),
			intval( $settings['restocknotifierpcprajapat_email_field_border_radius_bottom'] ?? 0 ),
			intval( $settings['restocknotifierpcprajapat_email_field_border_radius_left'] ?? 6 ),
			esc_attr( '100%' ),
			esc_attr( '0px' ),
			// Button CSS
			esc_attr( $settings['restocknotifierpcprajapat_button_bg_color_text'] ?? '#2271b1' ),
			esc_attr( $settings['restocknotifierpcprajapat_button_text_color_text'] ?? '#fff' ),
			intval( $settings['restocknotifierpcprajapat_button_border_size'] ?? 1 ),
			esc_attr( $settings['restocknotifierpcprajapat_button_border_color_text'] ?? '#2271b1' ),
			intval( $settings['restocknotifierpcprajapat_button_border_radius_top'] ?? 0 ),
			intval( $settings['restocknotifierpcprajapat_button_border_radius_right'] ?? 6 ),
			intval( $settings['restocknotifierpcprajapat_button_border_radius_bottom'] ?? 6 ),
			intval( $settings['restocknotifierpcprajapat_button_border_radius_left'] ?? 0 ),
			esc_attr( $settings['restocknotifierpcprajapat_button_hover_bg_color_text'] ?? '#135e96' ),
			esc_attr( $settings['restocknotifierpcprajapat_button_hover_text_color_text'] ?? '#fff' )
		);

		wp_add_inline_style( 'restocknotifierpcprajapat-public-style', $form_css );
	}

	/**
	 * Load settings for general tab
	 */
	public function register_settings() {
		// Register once for the 'general' tab
		register_setting(
			'restocknotifierpcprajapat_general',
			'restocknotifierpcprajapat_settings',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_array' ),
			)
		);

		// Register once for the 'design' tab
		register_setting(
			'restocknotifierpcprajapat_design',
			'restocknotifierpcprajapat_settings',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_array' ),
			)
		);

		// Register once for the 'email template' tab
		register_setting(
			'restocknotifierpcprajapat_email_template',
			'restocknotifierpcprajapat_settings',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_array' ),
			)
		);

		// Register once for the 'advanced' tab
		register_setting(
			'restocknotifierpcprajapat_advanced',
			'restocknotifierpcprajapat_settings',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_array' ),
			)
		);

		$settings = get_option( 'restocknotifierpcprajapat_settings', array() );
	}

	public function sanitize_array( $input ) {
		$existing = get_option( 'restocknotifierpcprajapat_settings', array() );

		if ( ! is_array( $existing ) ) {
			$existing = array();
		}

		// Merge existing settings with new input
		$merged = array_merge( $existing, $input );

		// Now sanitize only the incoming fields
		$output = array();

		foreach ( $merged as $key => $value ) {
			switch ( $key ) {
				case 'restocknotifierpcprajapat_enable_plugin':
					$output[ $key ] = ( 1 === (int) $value ) ? 1 : 0;
					break;
				case 'restocknotifierpcprajapat_form_bg_color_text':
				case 'restocknotifierpcprajapat_form_text_color_text':
				case 'restocknotifierpcprajapat_form_border_color_text':
				case 'restocknotifierpcprajapat_button_bg_color_text':
				case 'restocknotifierpcprajapat_button_text_color_text':
				case 'restocknotifierpcprajapat_button_border_color_text':
				case 'restocknotifierpcprajapat_button_hover_bg_color_text':
				case 'restocknotifierpcprajapat_button_hover_text_color_text':
					$output[ $key ] = sanitize_hex_color( $value );
					break;
				case 'restocknotifierpcprajapat_from_name':
					$output[ $key ] = sanitize_text_field( $value );
					break;
				case 'restocknotifierpcprajapat_from_email':
					$output[ $key ] = sanitize_email( $value );
					break;
				case 'restocknotifierpcprajapat_test_email':
					$output[ $key ] = sanitize_email( $value );
					break;
				case 'restocknotifierpcprajapat_success_message':
				case 'restocknotifierpcprajapat_email_subject':
					$output[ $key ] = sanitize_text_field( $value );
					break;
				case 'restocknotifierpcprajapat_email_body':
					$output[ $key ] = wp_kses_post( $value );
					break;
				case 'restocknotifierpcprajapat_email_body_stock_in':
					$output[ $key ] = wp_kses_post( $value );
					break;
				case 'restocknotifierpcprajapat_stock_threshold':
				case 'restocknotifierpcprajapat_notification_limit':
					$output[ $key ] = absint( $value );
					break;
				case 'restocknotifierpcprajapat_enable_debug':
				case 'restocknotifierpcprajapat_enable_cron':
					$output[ $key ] = ( 1 === (int) $value ) ? 1 : 0;
					break;
				case 'restocknotifierpcprajapat_data_cleanup':
					$valid_values   = array( 'never', 'after_notification', '30_days' );
					$output[ $key ] = in_array( $value, $valid_values, true ) ? $value : 'never';
					break;
				default:
					$output[ $key ] = sanitize_text_field( $value );
			}
		}

		return $output;
	}
}
