<?php
/**
 * Plugin Name:       RestArmor Security
 * Description:       Advanced security suite. Blocks REST API, disables XML-RPC, prevents user enumeration, and secures endpoints.
 * Version:           2.3
 * Author:            Md. Rakib Ullah
 * Author URI:        https://www.linkedin.com/in/rakib417/
 * Text Domain:       rest-armor-security
 * Domain Path:       /languages
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.8
 * Requires PHP:      7.4
 * Tested up to:      6.9
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Main Class for RestArmor Security
 */
class RestArmor_Security
{

	/**
	 * Initialize the plugin.
	 */
	public function __construct()
	{
		// REMOVED: load_plugin_textdomain is no longer needed for WP.org plugins.

		// Security Features.
		add_filter('xmlrpc_enabled', '__return_false'); // Disable XML-RPC.
		add_filter('xmlrpc_methods', array($this, 'disable_xmlrpc_pingback'));
		add_filter('rest_authentication_errors', array($this, 'restrict_rest_api'));
		add_action('template_redirect', array($this, 'block_user_enumeration'));
		add_filter('the_generator', '__return_empty_string'); // Hide WP Version.
		add_filter('rest_endpoints', array($this, 'disable_user_endpoints'));

		// Admin Bar Status Indicator.
		add_action('admin_bar_menu', array($this, 'add_admin_bar_indicator'), 999);
	}

	/**
	 * Add "RestArmor: Active" to the Admin Bar.
	 *
	 * @param WP_Admin_Bar $wp_admin_bar The admin bar object.
	 */
	public function add_admin_bar_indicator($wp_admin_bar)
	{
		// Only show to admins.
		if (!current_user_can('manage_options')) {
			return;
		}

		$wp_admin_bar->add_node(
			array(
				'id' => 'rest-armor-status',
				'title' => '<span class="ab-icon dashicons dashicons-shield"></span> ' . esc_html__('RestArmor: Active', 'rest-armor-security'),
				'href' => '#',
				'meta' => array(
					'title' => esc_html__('Your site is protected by RestArmor Security', 'rest-armor-security'),
					'class' => 'rest-armor-status-green',
				),
			)
		);
	}

	/**
	 * Disable XML-RPC Pingback specifically.
	 *
	 * @param array $methods List of XML-RPC methods.
	 * @return array
	 */
	public function disable_xmlrpc_pingback($methods)
	{
		unset($methods['pingback.ping']);
		return $methods;
	}

	/**
	 * Restrict REST API Access.
	 * Allow only logged-in users or localhost.
	 *
	 * @param mixed $result Current REST result.
	 * @return mixed|WP_Error
	 */
	public function restrict_rest_api($result)
	{
		if (true === $result || is_wp_error($result)) {
			return $result;
		}

		if (is_user_logged_in()) {
			return $result;
		}

		// Allow Whitelisted Routes (e.g., Contact Form 7, Jetpack).
		$route = isset($GLOBALS['wp']->query_vars['rest_route']) ? $GLOBALS['wp']->query_vars['rest_route'] : '';
		$whitelist = array('/contact-form-7/', '/jetpack/');

		foreach ($whitelist as $allowed) {
			if (strpos($route, $allowed) !== false) {
				return $result;
			}
		}

		return new WP_Error(
			'rest_forbidden',
			__('REST API restricted by RestArmor Security.', 'rest-armor-security'),
			array('status' => 401)
		);
	}

	/**
	 * Block User Enumeration (/?author=1).
	 *
	 * NOTE: This function checks $_GET['author'] to prevent bot enumeration.
	 * A nonce check is NOT applicable here because this is a public firewall rule.
	 */
	public function block_user_enumeration()
	{
		if (is_admin() || is_user_logged_in()) {
			return;
		}

		// Strict Check: Use $_GET instead of $_REQUEST.
		if (isset($_GET['author'])) {

			// Sanitize the input immediately.
			$author_val = sanitize_text_field(wp_unslash($_GET['author']));

			if (is_numeric($author_val)) {
				status_header(403);
				wp_die(
					esc_html__('User enumeration is forbidden by RestArmor Security.', 'rest-armor-security'),
					esc_html__('Forbidden', 'rest-armor-security'),
					array('response' => 403)
				);
			}
		}
	}

	/**
	 * Remove User Endpoints from REST API (/wp/v2/users).
	 *
	 * @param array $endpoints Registered REST endpoints.
	 * @return array
	 */
	public function disable_user_endpoints($endpoints)
	{
		if (!is_user_logged_in()) {
			if (isset($endpoints['/wp/v2/users'])) {
				unset($endpoints['/wp/v2/users']);
			}
			if (isset($endpoints['/wp/v2/users/(?P<id>[\d]+)'])) {
				unset($endpoints['/wp/v2/users/(?P<id>[\d]+)']);
			}
		}
		return $endpoints;
	}
}

// Initialize the plugin.
new RestArmor_Security();