<?php

/**
 * Modify product display for subscription products.
 */
add_action('woocommerce_before_single_product', 'rorder_check_subscription_and_modify_product_display', 1);

function rorder_check_subscription_and_modify_product_display() {
    global $post, $product;

    $product_type = $product->get_type();
    // Check if the product has subscription fields
    $is_subscription = get_post_meta($product->get_id(), '_subscription_fields', true);
    $enable_subscription = get_post_meta($post->ID, '_enable_subscription', true);

    if ($is_subscription && $enable_subscription == 'yes' && $product_type === 'simple') {
        add_action('woocommerce_before_add_to_cart_button', 'rorder_display_subscription_fields_for_simple_products', 10);
    }
}

add_action('wp_ajax_fetch_subscription_fields_single', 'rorder_fetch_subscription_fields_for_simple_product');
add_action('wp_ajax_nopriv_fetch_subscription_fields_single', 'rorder_fetch_subscription_fields_for_simple_product');


/**
 * Store subscription product data in the cart.
 *
 * @param array $cart_item_data The cart item data.
 * @param int $product_id The product ID.
 * @return array The updated cart item data.
 */
add_filter('woocommerce_add_cart_item_data', 'rorder_store_subscription_product_data_in_cart', 10, 2);

function rorder_store_subscription_product_data_in_cart($cart_item_data, $product_id) {
    if (isset($_POST['selected_subscription']) && !empty($_POST['selected_subscription'])) {
        $selected_subscription = sanitize_text_field(wp_unslash($_POST['selected_subscription']));
        $cart_item_data['_selected_subscription'] = $selected_subscription;

        // Get the subscription fields from the product
        $subscription_fields = get_post_meta($product_id, '_subscription_fields', true);

        // Check if the selected subscription exists in the subscription fields
        if (isset($subscription_fields[$selected_subscription])) {
            $selected_subscription_data = $subscription_fields[$selected_subscription];

            // Get the regular product price
            $product = wc_get_product($product_id); // Fetch the product object
            $regular_price = floatval($product->get_regular_price()); // Ensure we get the regular price

            // Get the subscription price and discount type
            $subscription_price = floatval($selected_subscription_data['_subs_price']);
            $subscription_discount_type = isset($selected_subscription_data['_subs_discount_type']) ? $selected_subscription_data['_subs_discount_type'] : 'flat';

            // Calculate the final price based on the discount type
            if ($subscription_discount_type === 'percentage') {
                $final_price = $regular_price - ($regular_price * ($subscription_price / 100)); // Apply percentage discount
            } elseif ($subscription_discount_type === 'amount_off') {
                $final_price = max(0, $regular_price - $subscription_price); // Apply amount off discount
            } else {
                $final_price = $subscription_price; // Use flat subscription price
            }

            // Store the final calculated price in cart item data
            $cart_item_data['_subscription_price'] = floatval($final_price);

            // Check for existing product ID in subscription details table
            global $wpdb;
            $table_name = $wpdb->prefix . 'subscription_details';
            $user_id = get_current_user_id();
            $cached_product_id = wp_cache_get("subscription_product_{$product_id}_user_{$user_id}", 'subscriptions');
            if ($cached_product_id === false) {
                $product_id = $wpdb->get_var($wpdb->prepare("SELECT product_id FROM $table_name WHERE product_id = %d AND user_id = %d", $product_id, $user_id));
                wp_cache_set("subscription_product_{$product_id}_user_{$user_id}", $product_id, 'subscriptions');
            }
            if (empty($product_id)) {
                $cart_item_data['_create_subscription'] = 'yes'; // Hidden key to identify subscription creation
            }
        }
    }
    return $cart_item_data;
}

/**
 * Update cart item price based on subscription.
 *
 * @param WC_Cart $cart The WooCommerce cart object.
 */
add_action('woocommerce_before_calculate_totals', 'rorder_update_cart_item_price_based_on_subscription', 10, 1);

function rorder_update_cart_item_price_based_on_subscription($cart) {
    if (is_admin() && !defined('DOING_AJAX')) {
        return;
    }

    // Loop through the cart items
    foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
        if (isset($cart_item['_subscription_price'])) {
            // Log the price being applied for debugging
            error_log('Applying subscription price to cart item: ' . $cart_item['_subscription_price']);

            // Set the final calculated subscription price
            $cart_item['data']->set_price($cart_item['_subscription_price']);

            error_log('Cart item price updated to: ' . $cart_item['data']->get_price());
        }
    }
}

/**
 * Apply subscription price to product.
 *
 * @param float $price The product price.
 * @param WC_Product $product The WooCommerce product object.
 * @return float The updated product price.
 */
add_filter('woocommerce_product_get_price', 'rorder_apply_subscription_price', 10, 2);
add_filter('woocommerce_product_get_sale_price', 'rorder_apply_subscription_price', 10, 2);
add_filter('woocommerce_product_get_regular_price', 'rorder_apply_subscription_price', 10, 2);

function rorder_apply_subscription_price($price, $product) {
    // Get the cart to find the current product in it
    $cart = WC()->cart;

    if (is_admin() && !defined('DOING_AJAX')) {
        return $price; // Don't alter price in admin
    }

    // Loop through the cart items
    foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
        if ($cart_item['product_id'] == $product->get_id() && isset($cart_item['_subscription_price'])) {
            // Use the subscription price instead of the regular/sale price
            $subscription_price = floatval($cart_item['_subscription_price']);
            if ($subscription_price > 0) {
                return $subscription_price;
            }
        }
    }

    // Return the original price if no subscription price is set
    return $price;
}

/**
 * Add subscription details to order items.
 *
 * @param WC_Order_Item_Product $item The WooCommerce order item object.
 * @param string $cart_item_key The cart item key.
 * @param array $values The cart item values.
 * @param WC_Order $order The WooCommerce order object.
 */
add_action('woocommerce_checkout_create_order_line_item', 'rorder_add_subscription_details_to_order_items', 10, 4);

function rorder_add_subscription_details_to_order_items($item, $cart_item_key, $values, $order) {

  

   
    if (isset($values['plan_type'])) {

        if(!empty($values['variation_id'])){


             $item->add_meta_data('_subscription_period', $values['plan_details']['interval']);
             $item->add_meta_data('_subscription_interval', $values['plan_details']['period']);
             $item->add_meta_data('_subscription_price', $values['plan_details']['price']);


         }else{
        //   // Add the subscription meta data to the order item
            $item->add_meta_data('_subscription_period', $values['plan_details']['period']);
            $item->add_meta_data('_subscription_interval', $values['plan_details']['interval']);
            $item->add_meta_data('_subscription_price', $values['plan_details']['price']);
        

         }
    }
}

/**
 * Display subscription details in the cart.
 *
 * @param array $item_data The cart item data.
 * @param array $cart_item The cart item.
 * @return array The updated cart item data.
 */
add_filter('woocommerce_get_item_data', 'rorder_display_subscription_product_data_in_cart', 10, 2);

function rorder_display_subscription_product_data_in_cart($item_data, $cart_item) {
    if (isset($cart_item['_selected_subscription'])) {
        $selected_subscription_key = $cart_item['_selected_subscription'];
        $product_id = $cart_item['product_id'];
        // Retrieve subscription fields for the current product
        $subscription_fields = get_post_meta($product_id, '_subscription_fields', true);
        if (isset($subscription_fields[$selected_subscription_key])) {
            $subscription_detail = $subscription_fields[$selected_subscription_key];

            $subscription_period = isset($subscription_detail['_subs_period']) ? $subscription_detail['_subs_period'] : '';
            $subscription_period_interval = isset($subscription_detail['_subs_period_interval']) ? $subscription_detail['_subs_period_interval'] : '';
            $subscription_price = isset($subscription_detail['_subs_price']) ? wc_price($subscription_detail['_subs_price']) : '';

            $subscription_period_display = $subscription_period . ' ' . $subscription_period_interval;

            $item_data[] = array(
                'key' => __('Subscription Period', 'reordere-lite-subcriptions-for-woocommerce'),
                'value' => $subscription_period_display
            );
            $item_data[] = array(
                'key' => __('Subscription Price', 'reordere-lite-subcriptions-for-woocommerce'),
                'value' => $subscription_price
            );
        } else {
            $item_data[] = array(
                'key' => __('Subscription Information', 'reordere-lite-subcriptions-for-woocommerce'),
                'value' => __('One Time Buy.', 'reordere-lite-subcriptions-for-woocommerce')
            );
        }
    }

    return $item_data;
}

/**
 * Display subscription data on the order page.
 *
 * @param int $item_id The order item ID.
 * @param WC_Order_Item $item The WooCommerce order item object.
 * @param WC_Order $order The WooCommerce order object.
 * @param bool $plain_text Whether to display in plain text.
 */
add_action('woocommerce_order_item_meta_end', 'rorder_display_subscription_data_on_order_page', 10, 4);

function rorder_display_subscription_data_on_order_page($item_id, $item, $order, $plain_text) {
    $subscription_period = wc_get_order_item_meta($item_id, '_subscription_period', true);
    $subscription_interval = wc_get_order_item_meta($item_id, '_subscription_interval', true);
    $subscription_price = wc_get_order_item_meta($item_id, '_subscription_price', true);

    if ($subscription_period && $subscription_interval && $subscription_price) {
        echo '<p><strong>' . esc_html__('Subscription Details', 'reordere-lite-subcriptions-for-woocommerce') . ':</strong></p>';
        echo '<p>' . esc_html__('Period', 'reordere-lite-subcriptions-for-woocommerce') . ': ' . esc_html($subscription_period) . '</p>';
        echo '<p>' . esc_html__('Interval', 'reordere-lite-subcriptions-for-woocommerce') . ': ' . esc_html($subscription_interval) . '</p>';
        echo '<p>' . esc_html__('Price', 'reordere-lite-subcriptions-for-woocommerce') . ': ' . wp_kses_post(wc_price($subscription_price)) . '</p>';
    }
}

/**
 * Store the subscription details in the database.
 *
 * @param int $order_id The order ID.
 */
add_action('woocommerce_thankyou', 'rorder_store_subscription_details_onvfdfvevfedwfcdwdcw', 10, 1);

function rorder_store_subscription_details_onvfdfvevfedwfcdwdcw($order_id) {
    global $wpdb;

    // Define your custom table name
    $table_name = $wpdb->prefix . 'subscription_details';

    // Get order details
    $order = wc_get_order($order_id);
    if (!$order) {
        return;
    }

    // Loop through order items
    foreach ($order->get_items() as $item_id => $item) {
        $product_id = $item->get_product_id();
        $product = wc_get_product($product_id);

        // Retrieve the subscription details stored in the order item meta
        $subscription_period = wc_get_order_item_meta($item_id, '_subscription_period', true);
        $subscription_interval = wc_get_order_item_meta($item_id, '_subscription_interval', true);
        $subscription_price = wc_get_order_item_meta($item_id, '_subscription_price', true);

        // Validate subscription details
        if (empty($subscription_period) || empty($subscription_interval) || empty($subscription_price)) {
            continue; // Skip if required subscription data is missing
        }

        // Ensure valid subscription period and interval
        if (!is_numeric($subscription_period) || !in_array($subscription_interval, ['day', 'week', 'month', 'year'])) {
            continue; // Skip invalid data
        }

        // Calculate subscription start and end dates
        $subscription_start_date = gmdate('Y-m-d H:i:s'); // Current date and time in MySQL format (Y-m-d H:i:s)
        $subscription_end_date = gmdate('Y-m-d H:i:s', strtotime("+$subscription_period $subscription_interval", strtotime($subscription_start_date)));

        $result = $wpdb->query(
            $wpdb->prepare(
                "INSERT INTO $table_name 
                (product_id, subscription_type, subscription_period, subscription_price, subscription_duration, subscription_start_date, subscription_end_date, user_id, sub_status, order_id, payment_mode) 
                VALUES (%d, %s, %d, %f, %s, %s, %s, %d, %s, %d, %s)",
                $product_id,
                $product->get_type(),
                $subscription_period,
                $subscription_price,
                $subscription_interval,
                $subscription_start_date,
                $subscription_end_date,
                $order->get_user_id(),
                'active',
                $order_id,
                $order->get_payment_method()
            )
        );

        // Error handling
        if ($result === false) {
            // error_log("Failed to insert subscription details for order ID $order_id and product ID $product_id: " . $wpdb->last_error); // Remove debug code
        }
    }
}

/**
 * Restrict multiple subscription products in the cart.
 *
 * @param bool $passed Whether the product can be added to the cart.
 * @param int $product_id The product ID.
 * @param int $quantity The quantity.
 * @return bool Whether the product can be added to the cart.
 */
add_filter('woocommerce_add_to_cart_validation', 'rorder_prevent_multiple_subscription_in_cart', 10, 3);

function rorder_prevent_multiple_subscription_in_cart($passed, $product_id, $quantity) {
    // Get subscription fields from the product being added
    $subscription_fields = get_post_meta($product_id, '_subscription_fields', true);
    
    // Check if the product being added is a subscription
    if ($subscription_fields) {
        // Loop through the cart and check if there is already a subscription product
        foreach (WC()->cart->get_cart() as $cart_item) {
            // Get subscription fields of the cart product
            $cart_product_subscription_fields = get_post_meta($cart_item['product_id'], '_subscription_fields', true);
            
            if ($cart_product_subscription_fields) {
                // If there's already a subscription product, prevent adding another one
                wc_add_notice(__('You can only have one subscription product in your cart.', 'reordere-lite-subcriptions-for-woocommerce'), 'error');
                return false;
            }
        }
    }
    
    return $passed;
}

/**
 * Check the cart after an item is removed.
 *
 * @param string $cart_item_key The cart item key.
 * @param WC_Cart $cart The WooCommerce cart object.
 */
add_action('woocommerce_cart_item_removed', 'rorder_check_cart_after_removal', 10, 2);

function rorder_check_cart_after_removal($cart_item_key, $cart) {
    // Check if there are any subscription products left in the cart
    $has_subscription = false;
    
    foreach (WC()->cart->get_cart() as $cart_item) {
        $cart_product_subscription_fields = get_post_meta($cart_item['product_id'], '_subscription_fields', true);
        
        if ($cart_product_subscription_fields) {
            $has_subscription = true;
            break;
        }
    }
    
    // If no subscription product is left, allow adding subscription products again
    if (!$has_subscription) {
        wc_add_notice(__('You can now add a new subscription product.', 'reordere-lite-subcriptions-for-woocommerce'), 'notice');
    }
}

/**
 * Ensure user is logged in before adding a subscription product to the cart.
 *
 * @param bool $passed Whether the product can be added to the cart.
 * @param int $product_id The product ID.
 * @param int $quantity The quantity.
 * @return bool Whether the product can be added to the cart.
 */
add_filter('woocommerce_add_to_cart_validation', 'rorder_require_login_for_subscription', 10, 3);

function rorder_require_login_for_subscription($passed, $product_id, $quantity) {
    // Check if the user is logged in
    if (!is_user_logged_in()) {
        // Get subscription fields from the product being added
        $subscription_fields = get_post_meta($product_id, '_subscription_fields', true);

        // Check if the product being added is a subscription
        if ($subscription_fields) {
            // Display an error message and prevent adding to cart
            wc_add_notice(__('You need to be logged in to purchase a subscription product.', 'reordere-lite-subcriptions-for-woocommerce'), 'error');
            return false;
        }
    }
    
    return $passed;
}

/**
 * Redirect to the WooCommerce login page if trying to purchase a subscription product.
 */
add_action('template_redirect', 'rorder_redirect_to_login_if_not_logged_in');

function rorder_redirect_to_login_if_not_logged_in() {
    // Check if the user is not logged in and viewing a product
    if (!is_user_logged_in() && is_product()) {
        global $post;
        
        // Get the current product object using the post ID
        $product = wc_get_product($post->ID);

        if ($product) {
            // Check if the product has subscription fields
            $is_subscription = get_post_meta($product->get_id(), '_subscription_fields', true);

            if ($is_subscription) {
                // Redirect to the WooCommerce login page
                $login_url = wc_get_page_permalink('myaccount'); // Redirect to "My Account" login
                wp_redirect($login_url);
                exit;
            }
        }
    }
}

/**
 * Display subscription fields for simple products.
 */
function rorder_display_subscription_fields_for_simple_products() {
    $selected_plan_view = get_option('rorder_selected_plan_view', 'simple_view');
    $classname = '';
    
    if($selected_plan_view == 'simple_view') {
        $classname = 'subscription-simple_view_container';
    } else if($selected_plan_view == 'card_view') {
        $classname = 'subscription-card_view_container';
    } else if( $selected_plan_view == 'list_view' ) {
        $classname = 'subscription-list_view_wrapper';
    } else if( $selected_plan_view == 'select_view' ) {
        $classname = 'subscription-select_view_wrapper';
    } else if( $selected_plan_view == 'adv_list_view' ) {
        $classname = 'subscription-adv_list_view_wrapper';
    } else if( $selected_plan_view == 'drawer_view' ) {
        $classname = 'subscription-drawer_view_wrapper';
    } else if( $selected_plan_view == 'popup_view' ) {
        $classname = 'subscription-popup_view_wrapper';
    }
    
    global $product;
    if ($product->is_type('simple')) {
        echo '<div id="subscription-fields-container" class="' . $classname . '"></div>';
    }
}

/**
 * Display subscription fields for variable products.
 */
add_action('woocommerce_before_add_to_cart_button', 'rorder_display_subscription_fields_for_variable_products', 10);

function rorder_display_subscription_fields_for_variable_products() {
    $selected_plan_view = get_option('rorder_selected_plan_view', 'simple_view');
    $classname = '';
    
    if($selected_plan_view == 'simple_view') {
        $classname = 'subscription-simple_view_container';
    } else if($selected_plan_view == 'card_view') {
        $classname = 'subscription-card_view_container';
    } else if( $selected_plan_view == 'list_view' ) {
        $classname = 'subscription-list_view_wrapper';
    } else if( $selected_plan_view == 'select_view' ) {
        $classname = 'subscription-select_view_wrapper';
    } else if( $selected_plan_view == 'adv_list_view' ) {
        $classname = 'subscription-adv_list_view_wrapper';
    } else if( $selected_plan_view == 'drawer_view' ) {
        $classname = 'subscription-drawer_view_wrapper';
    } else if( $selected_plan_view == 'popup_view' ) {
        $classname = 'subscription-popup_view_wrapper';
    }
    
    echo '<div id="subscription-fields-container" class="' . $classname . '" style="margin-top: 20px;">';
}

/**
 * Fetch subscription fields based on variation selection.
 */
add_action('wp_ajax_fetch_subscription_fields', 'rorder_fetch_subscription_fields_callback');
add_action('wp_ajax_nopriv_fetch_subscription_fields', 'rorder_fetch_subscription_fields_callback');

function rorder_fetch_subscription_fields_callback() {
    $variation_id = isset($_POST['variation_id']) ? intval($_POST['variation_id']) : 0;

    if ($variation_id) {
        $subscription_fields = get_post_meta($variation_id, '_subscription_fields', true);
        $is_subscription_enabled = get_post_meta($variation_id, '_enable_subscription', true);
        $selected_plan_view = get_option('rorder_selected_plan_view', 'simple_view');
        $one_time_purchase_title = get_option('one_time_purchase_title', 'One Time Purchase');
        $product = wc_get_product($variation_id);
        $adjusted_subscription_fields = [];
        $counter = 1; // Start adjusted keys from 1

        if ($is_subscription_enabled === 'yes' && !empty($subscription_fields) && $product) {
            echo '<h4>' . esc_html__('Subscription Plans:', 'reordere-lite-subcriptions-for-woocommerce') . '</h4>';
            
            // Set the proper wrapper class based on selected view
            $classname = '';
            if( $selected_plan_view == 'simple_view' ) {
                $classname = 'subscription-simple_view_wrapper';
            } else if( $selected_plan_view == 'card_view' ) {
                $classname = 'subscription-card_view_wrapper';
            } else if( $selected_plan_view == 'list_view' ) {
                $classname = 'subscription-list_view_wrapper';
            } else if( $selected_plan_view == 'select_view' ) {
                $classname = 'subscription-select_view_wrapper';
            } else if( $selected_plan_view == 'adv_list_view' ) {
                $classname = 'subscription-adv_list_view_wrapper';
            } else if( $selected_plan_view == 'drawer_view' ) {
                $classname = 'subscription-drawer_view_wrapper';
            } else if( $selected_plan_view == 'popup_view' ) {
                $classname = 'subscription-popup_view_wrapper';
            }
            
            echo '<div id="subscription-product-section" class="subscription-fields-wrapper variable_subscription_wrapper ' . $classname . '">';

            $first_plan_key = 'one-time-purchase'; // Default to one-time purchase
            $one_time_price = wc_price($product->get_price());

            // Render One-Time Purchase Option
            if ($selected_plan_view == 'simple_view') {
                echo '<div class="subscription-plan subscription-simple_view_card">
                        <div class="">
                            <label for="one-time-purchase" class="subscription-plan-card">
                                <h3>' . esc_html($one_time_purchase_title) . '</h3>
                                <div class="product-price">' . $one_time_price . '</div>
                            </label>
                            <input type="radio" id="one-time-purchase" name="selected_subscription" value="one-time-purchase" checked>
                        </div>
                    </div>';
            } elseif ($selected_plan_view == 'card_view') {
                // Card View - One-time purchase first
                echo '<div class="subscription-plans-row" style="display: flex; flex-wrap: wrap; gap: 20px; margin-top:30px;">'; // Flexbox for layout
                echo '<div class="subscription-plan subscription-plan_card_view" style="width:200px;">'; // Make each card take up 45% width, leaving space for the gap
                echo '<div>
                        <label for="one-time-purchase" class="subscription-plan-card">
                            <h3>' . esc_html__('Buy and Go', 'reordere-lite-subcriptions-for-woocommerce') . '</h3>
                            <div class="product-price">' . $one_time_price . '</div>
                        </label>
                        <input type="radio" id="one-time-purchase" name="selected_subscription" value="one-time-purchase" checked>
                        <p>' . esc_html__('One-time buy', 'reordere-lite-subcriptions-for-woocommerce') . '</p>
                    </div>';
                echo '</div>
                </div>';
                $counter = 2;
            } elseif ($selected_plan_view == "select_view") {
                echo '<div id="preview-box-3" class="preview-box-3 subscription-plan_select_view">
                        <div class="select-plan_adv_view">
                            <select name="selected_subscription_plan" id="sub_plan_adv" class="form-control">
                                <option value="">' . esc_html__('Select a plan', 'reordere-lite-subcriptions-for-woocommerce') . '</option>
                                <option value="one-time-purchase">' . esc_html($one_time_purchase_title) . ' - ' . $one_time_price . '</option>';
            } elseif ($selected_plan_view == "list_view") {
                echo '<div id="preview-box-3" class="preview-box-3 subscription-plan_select_view">
                        <ul class="subscription-plans">
                            <li class="subscription-plan">
                                <input type="radio" name="selected_subscription" id="one_time_purchase_list_view" class="list_radio" value="one-time-purchase" checked>
                                <label for="one_time_purchase_list_view">
                                    <div class="plan-title">' . esc_html($one_time_purchase_title) . '</div>
                                    <div class="plan-price">' . $one_time_price . '</div>
                                </label>
                            </li>';
            } elseif ($selected_plan_view == "adv_list_view") {
                echo '<div class="adv_list_view_delivery-frequency">
                        <div class="adv_list_frequency-option">
                            <div class="adv_list_ckbtn-text">
                                <input type="radio" name="selected_subscription" id="one_time_purchase_adv" class="" value="one-time-purchase" checked>
                                <label for="one_time_purchase_adv">
                                    <div class="plan-title">' . esc_html($one_time_purchase_title) . '</div>
                                </label>
                            </div>
                            <div class="adv_list_ckbtn-price">
                                <div class="plan-price">' . $one_time_price . '</div>
                            </div>
                        </div>';
            } elseif ($selected_plan_view == "drawer_view") {
                echo '<div class="drawer_view_wrapper">
                        <h6>' . esc_html__('Drawer view coming soon', 'reordere-lite-subcriptions-for-woocommerce') . '</h6>
                      </div>';
            } elseif ($selected_plan_view == "popup_view") {
                echo '<div class="popup_view_wrapper">
                        <h6>' . esc_html__('Popup view coming soon', 'reordere-lite-subcriptions-for-woocommerce') . '</h6>
                      </div>';
            }

            // Render Subscription Plans
            foreach ($subscription_fields as $original_key => $field) {
                $adjusted_subscription_fields[$counter] = $field;

                // Set the first plan key if it's the first iteration
                if ($first_plan_key === 'one-time-purchase') {
                    $first_plan_key = $counter;
                }

                $period = intval($field['_subs_period']);
                $interval = ucfirst($field['_subs_period_interval']);
                $subscription_price_field = floatval($field['_subs_price_field']); // User input value
                $regular_price = floatval($product->get_regular_price());
                $title = str_replace(['{{_subs_period}}', '{{_subs_interval}}'], [$period, $interval], sanitize_text_field($field['_subs_title']));

                // Discount Logic
                $subscription_discount_type = get_post_meta($variation_id, '_subscription_discount_type', true);
                $discount_type = $subscription_discount_type ?: 'flat';
                $final_price = $subscription_price_field;
                $discount_label = '';

                if ($discount_type === 'percentage' && $regular_price > 0) {
                    // For percentage: subscription_price_field is the discount percentage (e.g., 10 for 10% off)
                    $final_price = $regular_price - ($regular_price * ($subscription_price_field / 100));
                    $discount_label = sprintf('%d%% Off', intval($subscription_price_field));
                } elseif ($discount_type === 'amount_off' && $regular_price > 0) {
                    // For amount_off: subscription_price_field is the amount to subtract
                    $final_price = max(0, $regular_price - $subscription_price_field);
                    $discount_label = sprintf('Save %s', wc_price($subscription_price_field));
                } else {
                    // For flat: subscription_price_field is the final price
                    $final_price = $subscription_price_field;
                    // Calculate discount percentage for display
                    if ($regular_price > $final_price && $regular_price > 0) {
                        $discount_percentage = (($regular_price - $final_price) / $regular_price) * 100;
                        $discount_label = sprintf('%d%% Off', intval($discount_percentage));
                    }
                }

                $formatted_final_price = wc_price($final_price);

                if ($selected_plan_view == 'simple_view') {
                    // Simple view - vertical list
                    echo '<div class="subscription-plan subscription-simple_view_card">
                            <label for="plan-' . esc_attr($counter) . '" class="subscription-plan-card">
                                <h3>' . esc_html($title) . '</h3>
                                <div class="discount">' . ($regular_price > $final_price ? wc_price($regular_price) : '') . '</div>
                                <div class="product-price">' . $formatted_final_price . '</div>';
                    if ($discount_label) {
                        echo '<div class="discount-label">' . esc_html($discount_label) . '</div>';
                    }
                    echo '</label>
                            <input type="radio" id="plan-' . esc_attr($counter) . '" name="selected_subscription" value="' . esc_attr($counter) . '">
                          </div>';
                } elseif ($selected_plan_view == 'card_view') {
                    // Card View - Display in one row with 2 cards per row
                    echo '<div class="subscription-plan subscription-plan_card_view" style="flex: 1 1 45%;width:200px !important; margin-top:30px;">';
                    echo '<div>
                            <label for="plan-' . esc_attr($counter) . '" class="subscription-plan-card">
                                <h3>' . esc_html($title) . '</h3>
                                <div class="discount">' . ($regular_price > $final_price ? wc_price($regular_price) : '') . '</div>
                                <div class="product-price">' . $formatted_final_price . '</div>';
                    if ($discount_label) {
                        echo '<div class="discount-label">' . esc_html($discount_label) . '</div>';
                    }
                    echo '</label>
                            <input type="radio" id="plan-' . esc_attr($counter) . '" name="selected_subscription" value="' . esc_attr($counter) . '">
                        </div>';
                    echo '</div>';

                    if ($counter % 2 == 0) {
                        echo '</div>';
                        echo '<div class="subscription-plans-row" style="display: flex; flex-wrap: wrap; gap: 20px;">';
                    }
                } elseif ($selected_plan_view == 'select_view') {
                    // Select dropdown view
                    echo '<option value="' . esc_attr($counter) . '" data-price="' . esc_attr($final_price) . '">' . esc_html($title) . ' - ' . $formatted_final_price . '</option>';
                } elseif ($selected_plan_view == 'list_view') {
                    // List view with radio buttons
                    echo '<li class="subscription-plan">
                            <input type="radio" name="selected_subscription" id="plan-' . esc_attr($counter) . '" value="' . esc_attr($counter) . '">
                            <label for="plan-' . esc_attr($counter) . '">
                                <div class="plan-title">' . esc_html($title) . '</div>
                                <div class="plan-price">' . $formatted_final_price . '</div>';
                    if ($discount_label) {
                        echo '<div class="discount-label">' . esc_html($discount_label) . '</div>';
                    }
                    echo '</label>
                        </li>';
                } elseif ($selected_plan_view == 'adv_list_view') {
                    // Advanced list view
                    echo '<div class="adv_list_frequency-option">
                            <div class="adv_list_ckbtn-text">
                                <input type="radio" name="selected_subscription" id="plan-' . esc_attr($counter) . '" value="' . esc_attr($counter) . '">
                                <label for="plan-' . esc_attr($counter) . '">
                                    <div class="plan-title">' . esc_html($title) . '</div>
                                </label>
                            </div>
                            <div class="adv_list_ckbtn-price">
                                <div class="plan-price">' . $formatted_final_price . '</div>
                            </div>
                        </div>';
                }
                $counter++;
            }
            
            // Close wrappers for specific views
            if ($selected_plan_view == 'select_view') {
                echo '</select>
                    </div>
                </div>';
            }
            if ($selected_plan_view == 'list_view') {
                echo '</ul>
                    </div>';
            }
            if ($selected_plan_view == 'adv_list_view') {
                echo '</div>'; // Close adv_list_view_delivery-frequency
            }
            
            // Remove the old advanced_list_view handling
            if (false && $selected_plan_view == 'advanced_list_view') {
                echo '<li class="subscription-plan">
                        <input type="radio" name="selected_subscription" id="one_time_purchase_adv_view" value="one_time_purchase" class="adv-radio">
                        <label for="one_time_purchase_adv_view">
                            <h3>' . esc_html($one_time_purchase_title) . '</h3>
                            <div class="final-price">' . wc_price($product->get_regular_price()) . '</div>
                        </label>
                    </li>
                    
                    <li class="subscription-plan" id="custom_id">
                        <input type="radio" name="selected_subscription" id="subscription_plan_adv_view" value="subscription_plan" class="adv-radio">
                        <label for="subscription_plan_adv_view">
                            <h3>' . esc_html__('Subscription', 'reordere-lite-subcriptions-for-woocommerce') . '</h3>
                        </label>
                    </li>

                    <div class="select-plan_adv_view" id="subscription-select_adv_plan" style="display: none;">
                        <select name="selected_subscription_plan" id="sub_plan_adv" class="form-control">
                            <option value="">' . esc_html__('Select a plan', 'reordere-lite-subcriptions-for-woocommerce') . '</option>';
                foreach ($subscription_fields as $key => $field) {
                    $title = isset($field['name']) ? sanitize_text_field($field['name']) : esc_html__('(No Name)', 'reordere-lite-subcriptions-for-woocommerce');
                    echo '<option value="' . esc_attr($key) . '">' . esc_html($title) . ' - ' . wc_price($field['price']) . '</option>';
                }
                echo '</select>
                    </div>
                </ul>
            </div>';
            }
            if ($selected_plan_view == 'inherit_list_view') {
                echo '<li class="subscription-plan" style="width:360px;">
                        <input type="radio" name="selected_subscription" id="one_time_purchase_inherit_view" class="inherit_radio" value="one_time_purchase">
                        <label for="one_time_purchase">
                            <div class="plan-title">' . esc_html($one_time_purchase_title) . '</div>
                            <div class="plan-price">' . wc_price($product->get_regular_price()) . '</div>
                        </label>
                    </li>
                    
                    <li class="subscription-plan">
                        <input type="radio" name="selected_subscription_inherit_view" id="subscription_plan" value="subscription_plan" class="inherit_radio">
                        <label for="subscription_plan">
                            <div class="plan-title">' . esc_html__('Subscription', 'reordere-lite-subcriptions-for-woocommerce') . '</div>
                        </label>
                    </li>
                    
                    <div class="delivery-frequency" id="subscription-select_inherit" style="display: none;">';
                foreach ($subscription_fields as $key => $field) {
                    $title = isset($field['name']) ? sanitize_text_field($field['name']) : esc_html__('(No Name)', 'reordere-lite-subcriptions-for-woocommerce');
                    echo '<div class="frequency-option" id="subscription_plan_inherit_div" style="display: flex; align-items: center; margin-bottom: 10px;">
                            <input type="radio" name="selected_subscription" id="subscription_plan_inherit" value="' . esc_attr($key) . '" data-price="' . esc_html($field['price']) . '" style="margin-left: 130px;">
                            <label for="subscription_plan_' . esc_attr($key) . '" style="display: flex; align-items: center;">
                                <div class="plan-price" style="font-weight: bold; color: #333; margin-right:40px;">' . esc_html($title) . ' - ' . wc_price($field['price']) . '</div>
                            </label>
                        </div>';
                }
                echo '</div>
            </div>';
            }

            if ($counter % 2 != 0) {
                echo '</div>';
            }
            echo '</div>';
        } else {
            echo '<p>' . esc_html__('No subscription plans available for this variation.', 'reordere-lite-subcriptions-for-woocommerce') . '</p>';
        }
    } else {
        echo '<p>' . esc_html__('No variation selected.', 'reordere-lite-subcriptions-for-woocommerce') . '</p>';
    }
    echo '<input type="hidden" id="selected_plan_key" name="selected_plan_key" value="' . esc_attr($first_plan_key) . '">';
    echo '<input type="hidden" id="product_id" name="product_id" value="' . esc_attr($variation_id) . '">';
    echo '<button type="button" id="add-to-cart-button" class="single_add_to_cart_button button alt">' 
         . esc_html__('Subscribe Now', 'reordere-lite-subcriptions-for-woocommerce') . '</button>';
    echo '</div>';
    wp_die();
}

/**
 * Fetch subscription fields for simple products.
 */
function rorder_fetch_subscription_fields_for_simple_product() {
    if (!isset($_POST['security']) || !check_ajax_referer('subscription_nonce', 'security', false)) {
        wp_send_json_error('Nonce verification failed.');
    }

    $product_id = isset($_POST['product_id']) ? intval(wp_unslash($_POST['product_id'])) : 0;
    $is_subscription_enabled = get_post_meta($product_id, '_enable_subscription', true);

    if ($is_subscription_enabled === 'yes') {
        if ($product_id) {
            $subscription_fields = get_post_meta($product_id, '_subscription_fields', true);
            $selected_plan_view = get_option('rorder_selected_plan_view', 'simple_view');
            $one_time_purchase_title = get_option('one_time_purchase_title', 'One Time Purchase');
            $product = wc_get_product($product_id);
            $adjusted_subscription_fields = [];
            $counter = 1; // Start adjusted keys from 1

            if ($is_subscription_enabled === 'yes' && !empty($subscription_fields) && $product) {
                echo '<h4>' . esc_html__('Subscription Plans:', 'reordere-lite-subcriptions-for-woocommerce') . '</h4>';
                
                // Set the proper wrapper class based on selected view
                $classname = '';
                if( $selected_plan_view == 'simple_view' ) {
                    $classname = 'subscription-simple_view_wrapper';
                } else if( $selected_plan_view == 'card_view' ) {
                    $classname = 'subscription-card_view_wrapper';
                } else if( $selected_plan_view == 'list_view' ) {
                    $classname = 'subscription-list_view_wrapper';
                } else if( $selected_plan_view == 'select_view' ) {
                    $classname = 'subscription-select_view_wrapper';
                } else if( $selected_plan_view == 'adv_list_view' ) {
                    $classname = 'subscription-adv_list_view_wrapper';
                } else if( $selected_plan_view == 'drawer_view' ) {
                    $classname = 'subscription-drawer_view_wrapper';
                } else if( $selected_plan_view == 'popup_view' ) {
                    $classname = 'subscription-popup_view_wrapper';
                }
                
                echo '<div id="subscription-product-section" class="subscription-fields-wrapper simple_subscription_wrapper ' . $classname . '">';

                $first_plan_key = 'one-time-purchase'; // Default to one-time purchase
                $one_time_price = wc_price($product->get_price());

                // Render One-Time Purchase Option
                if ($selected_plan_view == 'simple_view') {
                    echo '<div class="subscription-plan subscription-simple_view_card">
                            <div class="">
                                <label for="one-time-purchase" class="subscription-plan-card">
                                    <h3>' . esc_html($one_time_purchase_title) . '</h3>
                                    <div class="product-price">' . $one_time_price . '</div>
                                </label>
                                <input type="radio" id="one-time-purchase" name="selected_subscription" value="one-time-purchase" checked>
                            </div>
                        </div>';
                } elseif ($selected_plan_view == 'card_view') {
                    // Card View - One-time purchase first
                    echo '<div class="subscription-plans-row" style="display: flex; flex-wrap: wrap; gap: 20px; margin-top:30px;">'; // Flexbox for layout
                    echo '<div class="subscription-plan subscription-plan_card_view" style="width:200px;">'; // Make each card take up 45% width, leaving space for the gap
                    echo '<div>
                            <label for="one-time-purchase" class="subscription-plan-card">
                                <h3>' . esc_html__('Buy and Go', 'reordere-lite-subcriptions-for-woocommerce') . '</h3>
                                <div class="product-price">' . $one_time_price . '</div>
                            </label>
                            <input type="radio" id="one-time-purchase" name="selected_subscription" value="one-time-purchase" checked>
                            <p>' . esc_html__('One-time buy', 'reordere-lite-subcriptions-for-woocommerce') . '</p>
                        </div>';
                    echo '</div>
                    </div>';
                    $counter = 2;
                } elseif ($selected_plan_view == "select_view") {
                    echo '<div id="preview-box-3" class="preview-box-3 subscription-plan_select_view">
                            <div class="select-plan_adv_view">
                                <select name="selected_subscription_plan" id="sub_plan_adv" class="form-control">
                                    <option value="">' . esc_html__('Select a plan', 'reordere-lite-subcriptions-for-woocommerce') . '</option>
                                    <option value="one-time-purchase">' . esc_html($one_time_purchase_title) . ' - ' . $one_time_price . '</option>';
                } elseif ($selected_plan_view == "list_view") {
                    echo '<div id="preview-box-3" class="preview-box-3 subscription-plan_select_view">
                            <ul class="subscription-plans">
                                <li class="subscription-plan">
                                    <input type="radio" name="selected_subscription" id="one_time_purchase_list_view" class="list_radio" value="one-time-purchase" checked>
                                    <label for="one_time_purchase_list_view">
                                        <div class="plan-title">' . esc_html($one_time_purchase_title) . '</div>
                                        <div class="plan-price">' . $one_time_price . '</div>
                                    </label>
                                </li>';
                } elseif ($selected_plan_view == "adv_list_view") {
                    echo '<div class="adv_list_view_delivery-frequency">
                            <div class="adv_list_frequency-option">
                                <div class="adv_list_ckbtn-text">
                                    <input type="radio" name="selected_subscription" id="one_time_purchase_adv" class="" value="one-time-purchase" checked>
                                    <label for="one_time_purchase_adv">
                                        <div class="plan-title">' . esc_html($one_time_purchase_title) . '</div>
                                    </label>
                                </div>
                                <div class="adv_list_ckbtn-price">
                                    <div class="plan-price">' . $one_time_price . '</div>
                                </div>
                            </div>';
                } elseif ($selected_plan_view == "drawer_view") {
                    echo '<div class="drawer_view_wrapper">
                            <h6>' . esc_html__('Drawer view coming soon', 'reordere-lite-subcriptions-for-woocommerce') . '</h6>
                          </div>';
                } elseif ($selected_plan_view == "popup_view") {
                    echo '<div class="popup_view_wrapper">
                            <h6>' . esc_html__('Popup view coming soon', 'reordere-lite-subcriptions-for-woocommerce') . '</h6>
                          </div>';
                }

                // Render Subscription Plans
                foreach ($subscription_fields as $original_key => $field) {
                    $adjusted_subscription_fields[$counter] = $field;

                    // Set the first plan key if it's the first iteration
                    if ($first_plan_key === 'one-time-purchase') {
                        $first_plan_key = $counter;
                    }

                    $period = intval($field['_subs_period']);
                    $interval = ucfirst($field['_subs_period_interval']);
                    $subscription_price_field = floatval($field['_subs_price_field']); // User input value
                    $regular_price = floatval($product->get_regular_price());
                    $title = str_replace(['{{_subs_period}}', '{{_subs_interval}}'], [$period, $interval], sanitize_text_field($field['_subs_title']));

                    // Discount Logic
                    $subscription_discount_type = get_post_meta($product_id, '_subscription_discount_type', true);
                    $discount_type = $subscription_discount_type ?: 'flat';
                    $final_price = $subscription_price_field;
                    $discount_label = '';

                    if ($discount_type === 'percentage' && $regular_price > 0) {
                        // For percentage: subscription_price_field is the discount percentage (e.g., 10 for 10% off)
                        $final_price = $regular_price - ($regular_price * ($subscription_price_field / 100));
                        $discount_label = sprintf('%d%% Off', intval($subscription_price_field));
                    } elseif ($discount_type === 'amount_off' && $regular_price > 0) {
                        // For amount_off: subscription_price_field is the amount to subtract
                        $final_price = max(0, $regular_price - $subscription_price_field);
                        $discount_label = sprintf('Save %s', wc_price($subscription_price_field));
                    } else {
                        // For flat: subscription_price_field is the final price
                        $final_price = $subscription_price_field;
                        // Calculate discount percentage for display
                        if ($regular_price > $final_price && $regular_price > 0) {
                            $discount_percentage = (($regular_price - $final_price) / $regular_price) * 100;
                            $discount_label = sprintf('%d%% Off', intval($discount_percentage));
                        }
                    }

                    $formatted_final_price = wc_price($final_price);

                    if ($selected_plan_view == 'simple_view') {
                        // Simple view - vertical list
                        echo '<div class="subscription-plan subscription-simple_view_card">
                                <label for="plan-' . esc_attr($counter) . '" class="subscription-plan-card">
                                    <h3>' . esc_html($title) . '</h3>
                                    <div class="discount">' . ($regular_price > $final_price ? wc_price($regular_price) : '') . '</div>
                                    <div class="product-price">' . $formatted_final_price . '</div>';
                        if ($discount_label) {
                            echo '<div class="discount-label">' . esc_html($discount_label) . '</div>';
                        }
                        echo '</label>
                                <input type="radio" id="plan-' . esc_attr($counter) . '" name="selected_subscription" value="' . esc_attr($counter) . '">
                              </div>';
                    } elseif ($selected_plan_view == 'card_view') {
                        // Card View - Display in one row with 2 cards per row
                        echo '<div class="subscription-plan subscription-plan_card_view" style="flex: 1 1 45%;width:200px !important; margin-top:30px;">';
                        echo '<div>
                                <label for="plan-' . esc_attr($counter) . '" class="subscription-plan-card">
                                    <h3>' . esc_html($title) . '</h3>
                                    <div class="discount">' . ($regular_price > $final_price ? wc_price($regular_price) : '') . '</div>
                                    <div class="product-price">' . $formatted_final_price . '</div>';
                        if ($discount_label) {
                            echo '<div class="discount-label">' . esc_html($discount_label) . '</div>';
                        }
                        echo '</label>
                                <input type="radio" id="plan-' . esc_attr($counter) . '" name="selected_subscription" value="' . esc_attr($counter) . '">
                            </div>';
                        echo '</div>';

                        if ($counter % 2 == 0) {
                            echo '</div>';
                            echo '<div class="subscription-plans-row" style="display: flex; flex-wrap: wrap; gap: 20px;">';
                        }
                        if ($counter % 2 == 0) {
                            echo '</div>';
                            echo '<div class="subscription-plans-row" style="display: flex; flex-wrap: wrap; gap: 20px;">';
                        }
                    } elseif ($selected_plan_view == 'select_view') {
                        // Select dropdown view
                        echo '<option value="' . esc_attr($counter) . '" data-price="' . esc_attr($final_price) . '">' . esc_html($title) . ' - ' . $formatted_final_price . '</option>';
                    } elseif ($selected_plan_view == 'list_view') {
                        // List view with radio buttons
                        echo '<li class="subscription-plan">
                                <input type="radio" name="selected_subscription" id="plan-' . esc_attr($counter) . '" value="' . esc_attr($counter) . '">
                                <label for="plan-' . esc_attr($counter) . '">
                                    <div class="plan-title">' . esc_html($title) . '</div>
                                    <div class="plan-price">' . $formatted_final_price . '</div>';
                        if ($discount_label) {
                            echo '<div class="discount-label">' . esc_html($discount_label) . '</div>';
                        }
                        echo '</label>
                            </li>';
                    } elseif ($selected_plan_view == 'adv_list_view') {
                        // Advanced list view
                        echo '<div class="adv_list_frequency-option">
                                <div class="adv_list_ckbtn-text">
                                    <input type="radio" name="selected_subscription" id="plan-' . esc_attr($counter) . '" value="' . esc_attr($counter) . '">
                                    <label for="plan-' . esc_attr($counter) . '">
                                        <div class="plan-title">' . esc_html($title) . '</div>
                                    </label>
                                </div>
                                <div class="adv_list_ckbtn-price">
                                    <div class="plan-price">' . $formatted_final_price . '</div>
                                </div>
                            </div>';
                    }
                    $counter++;
                }
                
                // Close wrappers for specific views
                if ($selected_plan_view == 'select_view') {
                    echo '</select>
                        </div>
                    </div>';
                }
                if ($selected_plan_view == 'list_view') {
                    echo '</ul>
                        </div>';
                }
                if ($selected_plan_view == 'adv_list_view') {
                    echo '</div>'; // Close adv_list_view_delivery-frequency
                }
                
                // Remove old view handling
                if (false && $selected_plan_view == 'advanced_list_view') {
                    echo '<li class="subscription-plan">
                            <input type="radio" name="selected_subscription" id="one_time_purchase_adv_view" value="one_time_purchase" class="adv-radio">
                            <label for="one_time_purchase_adv_view">
                                <h3>' . esc_html($one_time_purchase_title) . '</h3>
                                <div class="final-price">' . wc_price($product->get_regular_price()) . '</div>
                            </label>
                        </li>
                        
                        <li class="subscription-plan" id="custom_id">
                            <input type="radio" name="selected_subscription" id="subscription_plan_adv_view" value="subscription_plan" class="adv-radio">
                            <label for="subscription_plan_adv_view">
                                <h3>' . esc_html__('Subscription', 'reordere-lite-subcriptions-for-woocommerce') . '</h3>
                            </label>
                        </li>

                        <div class="select-plan_adv_view" id="subscription-select_adv_plan" style="display: none;">
                            <select name="selected_subscription_plan" id="sub_plan_adv" class="form-control">
                                <option value="">' . esc_html__('Select a plan', 'reordere-lite-subcriptions-for-woocommerce') . '</option>';
                    foreach ($subscription_fields as $key => $field) {
                        $title = isset($field['name']) ? sanitize_text_field($field['name']) : esc_html__('(No Name)', 'reordere-lite-subcriptions-for-woocommerce');
                        echo '<option value="' . esc_attr($key) . '">' . esc_html($title) . ' - ' . wc_price($field['price']) . '</option>';
                    }
                    echo '</select>
                        </div>
                    </ul>
                </div>';
                }
                if ($selected_plan_view == 'inherit_list_view') {
                    echo '<li class="subscription-plan" style="width:360px;">
                            <input type="radio" name="selected_subscription" id="one_time_purchase_inherit_view" class="inherit_radio" value="one_time_purchase">
                            <label for="one_time_purchase">
                                <div class="plan-title">' . esc_html($one_time_purchase_title) . '</div>
                                <div class="plan-price">' . wc_price($product->get_regular_price()) . '</div>
                            </label>
                        </li>
                        
                        <li class="subscription-plan">
                            <input type="radio" name="selected_subscription_inherit_view" id="subscription_plan" value="subscription_plan" class="inherit_radio">
                            <label for="subscription_plan">
                                <div class="plan-title">' . esc_html__('Subscription', 'reordere-lite-subcriptions-for-woocommerce') . '</div>
                            </label>
                        </li>
                        
                        <div class="delivery-frequency" id="subscription-select_inherit" style="display: none;">';
                    foreach ($subscription_fields as $key => $field) {
                        $title = isset($field['name']) ? sanitize_text_field($field['name']) : esc_html__('(No Name)', 'reordere-lite-subcriptions-for-woocommerce');
                        echo '<div class="frequency-option" id="subscription_plan_inherit_div" style="display: flex; align-items: center; margin-bottom: 10px;">
                                <input type="radio" name="selected_subscription" id="subscription_plan_inherit" value="' . esc_attr($key) . '" data-price="' . esc_html($field['price']) . '" style="margin-left: 130px;">
                                <label for="subscription_plan_' . esc_attr($key) . '" style="display: flex; align-items: center;">
                                    <div class="plan-price" style="font-weight: bold; color: #333; margin-right:40px;">' . esc_html($title) . ' - ' . wc_price($field['price']) . '</div>
                                </label>
                            </div>';
                    }
                    echo '</div>
                </div>';
                }

                if ($counter % 2 != 0) {
                    echo '</div>';
                }
                echo '</div>';
            }
        }
    }
    echo '<input type="hidden" id="selected_plan_key" name="selected_plan_key" value="' . esc_attr($first_plan_key) . '">';
    echo '<input type="hidden" id="product_id" name="product_id" value="' . esc_attr($product_id) . '">';
    echo '<button type="button" id="add-to-cart-button" class="single_add_to_cart_button button alt">' 
         . esc_html__('Subscribe Now', 'reordere-lite-subcriptions-for-woocommerce') . '</button>';
    echo '</div>';
    wp_die();
}

/**
 * Add to cart with selected plan.
 */
add_action('wp_ajax_add_to_cart_with_plan', 'rorder_add_to_cart_with_plan_callback');
add_action('wp_ajax_nopriv_add_to_cart_with_plan', 'rorder_add_to_cart_with_plan_callback');

function rorder_add_to_cart_with_plan_callback() {
    check_ajax_referer('subscription_nonce', 'security'); // Validate nonce

    $plan_key = isset($_POST['plan_key']) ? sanitize_text_field(wp_unslash($_POST['plan_key'])) : '';
    $product_id = isset($_POST['product_id']) ? intval(wp_unslash($_POST['product_id'])) : 0;

    if (!$product_id || !$plan_key) {
        wp_send_json_error(['message' => 'Invalid product or plan selection.']);
    }

    $product = wc_get_product($product_id);
    if (!$product) {
        wp_send_json_error(['message' => 'Invalid product.']);
    }

    // Fetch subscription fields for the product
    $subscription_fields = get_post_meta($product_id, '_subscription_fields', true);
    $adjusted_subscription_fields = [];
    $counter = 1;
    $cart_item_data = [];
    $price = $product->get_price(); // Default price

    foreach ($subscription_fields as $key => $field) {
        $adjusted_subscription_fields[$counter] = $field;
        $counter++;
    }

    if ($plan_key === 'one-time-purchase') {
        // Handle "One-Time Purchase"
        $cart_item_data['plan_type'] = 'one-time-purchase';
        $cart_item_data['price'] = $price; // Regular product price
    } elseif (isset($adjusted_subscription_fields[$plan_key])) {
        // Handle subscription plans
        $plan = $adjusted_subscription_fields[$plan_key];
        $cart_item_data['plan_type'] = 'subscription';
        $cart_item_data['plan_details'] = [
            'name' => sanitize_text_field($plan['_subs_title']),
            'price' => floatval($plan['_subs_price']),
            'interval' => sanitize_text_field($plan['_subs_period_interval']),
            'period' => intval($plan['_subs_period']),
        ];

        // Apply discount logic
        $regular_price = floatval($product->get_regular_price());
        $subscription_price_field = floatval($plan['_subs_price_field']); // User input value
        $discount_type = get_post_meta($product_id, '_subscription_discount_type', true) ?: 'flat'; // Default to flat rate

        if ($discount_type === 'percentage') {
            $discount_value = $subscription_price_field; // Treat as percentage (e.g., 10 for 10%)
            $cart_item_data['price'] = $regular_price - ($regular_price * ($discount_value / 100));
        } elseif ($discount_type === 'amount_off') {
            $cart_item_data['price'] = max(0, $regular_price - $subscription_price_field); // Amount-off discount
        } else {
            $cart_item_data['price'] = $subscription_price_field; // Flat rate
        }
    } else {
        wp_send_json_error(['message' => 'Invalid subscription plan selected.']);
    }

    // Add the product to the cart
    $added = WC()->cart->add_to_cart($product_id, 1, 0, [], $cart_item_data);

    if ($added) {
        wp_send_json_success(['redirect_url' => wc_get_cart_url()]);
    } else {
        wp_send_json_error(['message' => 'Failed to add product to cart.']);
    }
}

/**
 * Update cart item price based on selected plan.
 *
 * @param WC_Cart $cart The WooCommerce cart object.
 */
add_action('woocommerce_before_calculate_totals', 'rorder_update_cart_item_price_based_on_plan', 10, 1);
function rorder_update_cart_item_price_based_on_plan($cart) {
    if (is_admin() && !defined('DOING_AJAX')) {
        return;
    }

    foreach ($cart->get_cart() as $cart_item) {
        if (isset($cart_item['price'])) {
            $cart_item['data']->set_price($cart_item['price']); // Update cart item price
        }
    }
}

/**
 * Display subscription plan in the cart.
 *
 * @param array $item_data The cart item data.
 * @param array $cart_item The cart item.
 * @return array The updated cart item data.
 */
add_filter('woocommerce_get_item_data', 'rorder_display_subscription_plan_in_cart', 10, 2);
function rorder_display_subscription_plan_in_cart($item_data, $cart_item) {
    if (isset($cart_item['plan_type']) && $cart_item['plan_type'] === 'subscription') {
        $plan_details = $cart_item['plan_details'];
        
        $item_data[] = [
            'name' => __('Plan', 'reordere-lite-subcriptions-for-woocommerce'),
            'value' => sprintf(
                '%s (%s %s at %s)',
                $plan_details['name'],
                ucfirst($plan_details['interval']),
                $plan_details['period'],
                wc_price($cart_item['price']) // Use updated price
            ),
        ];
    } elseif (isset($cart_item['plan_type']) && $cart_item['plan_type'] === 'one-time-purchase') {
        $item_data[] = [
            'name' => __('Plan', 'reordere-lite-subcriptions-for-woocommerce'),
            'value' => __('One-Time Purchase', 'reordere-lite-subcriptions-for-woocommerce'),
        ];
    }

    return $item_data;
}

/**
 * Save subscription plan in the order.
 *
 * @param WC_Order_Item_Product $item The WooCommerce order item object.
 * @param string $cart_item_key The cart item key.
 * @param array $values The cart item values.
 * @param WC_Order $order The WooCommerce order object.
 */
add_action('woocommerce_checkout_create_order_line_item', 'rorder_save_subscription_plan_in_order', 10, 4);
function rorder_save_subscription_plan_in_order($item, $cart_item_key, $values, $order) {
    if (isset($values['plan_type'])) {
        $item->add_meta_data('Plan Type', $values['plan_type']);
        if ($values['plan_type'] === 'subscription' && isset($values['plan_details'])) {
            $plan_details = $values['plan_details'];
            $item->add_meta_data('Subscription Plan', sprintf(
                '%s (%s %s at %s)',
                $plan_details['name'],
                ucfirst($plan_details['interval']),
                $plan_details['period'],               
                wc_price($values['price'])
            ));
        }
        if (isset($values['price'])) {
            $item->add_meta_data('Plan Price', wc_price($values['price']));
        }
    }
}

