<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Public-specific functionality of the plugin.
 * 
 * Defines the plugin name, version, and examples of how to enqueue
 * the public-specific stylesheet and register menu pages.
 * 
 * @link       http://www.techspawn.com
 * @since      1.0.0
 * @package    subscription-products
 * @subpackage subscription-products/public
 * @author     Techspawn Solutions
 */

class Rorder_Subscription_Products_Public
{
    /**
     * The plugin name.
     * 
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     * 
     * @since    1.0.0
     * @access   private
     * @var      string  $version  The current version of this plugin.
     */
    private $version;

    /**
     * Class constructor to initialize properties.
     * 
     * @since    1.0.0
     * @param    string    $plugin_name    The name of the plugin.
     * @param    string    $version        The version of this plugin.
     */
    public function __construct($plugin_name, $version)
    {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        // Hook into WordPress' enqueue actions
        add_action('wp_enqueue_scripts', array($this, 'enqueue_styles'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_jquery_ui_dialog'));
    }

    /**
     * Enqueue public-facing stylesheets.
     */
    public function enqueue_styles()
    {
        // Enqueue the stylesheet
        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/subscription-products-public.css', array(), $this->version, 'all');
    }

    /**
     * Enqueue public-facing JavaScript files.
     */
    public function enqueue_scripts()
    {
        // Enqueue the script
        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-dialog');
        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/subscription-products-public.js', array('jquery'), $this->version, false);
        wp_localize_script($this->plugin_name, 'subscription_vars', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('subscription_nonce')
        ));
    }

    /**
     * Display subscription product data.
     */
    public function subscription_products_public_display()
    {
        include_once plugin_dir_path(__FILE__) . 'partials/class-subscription-products-public-display.php';
    }

    /**
     * Enqueue jQuery UI Dialog.
     */
    public function enqueue_jquery_ui_dialog() {
        // Enqueue jQuery (already bundled with WordPress)
        wp_enqueue_script('jquery');
        
        // Enqueue jQuery UI Datepicker and Dialog
        wp_enqueue_script('jquery-ui-datepicker');
        wp_enqueue_script('jquery-ui-dialog');
    
       
    }
    
}

/**
 * Add 'Subscriptions' tab in My Account page.
 *
 * @param array $items Existing account menu items.
 * @return array Modified account menu items.
 */
function rorder_add_subscriptions_tab_in_my_account($items) {
    // Add the 'Subscriptions' tab after 'Orders'
    $new_items = array();

    foreach ($items as $key => $item) {
        $new_items[$key] = $item;  // Keep existing items
        if ($key === 'orders') {   // Add 'subscriptions' after 'orders'
            $new_items['subscriptions'] = __('Subscriptions', 'reordere-lite-subcriptions-for-woocommerce');  // Add the Subscriptions menu
        }
    }

    return $new_items;
}
add_filter('woocommerce_account_menu_items', 'rorder_add_subscriptions_tab_in_my_account');

/**
 * Register the 'subscriptions' endpoint.
 */
function rorder_subscriptions_tab_endpoint() {
    add_rewrite_endpoint('subscriptions', EP_ROOT | EP_PAGES);
}
add_action('init', 'rorder_subscriptions_tab_endpoint');

/**
 * Display content for the 'subscriptions' tab.
 */
function rorder_subscriptions_tab_content() {
    echo '<h3>' . esc_html__('Your Subscriptions', 'reordere-lite-subcriptions-for-woocommerce') . '</h3>';

    $user_id = get_current_user_id();
    global $wpdb;
    $table_name = $wpdb->prefix . 'subscription_details';

    $subscriptions = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM $table_name WHERE user_id = %d",
        $user_id
    ));

    if ($subscriptions) {
        echo '<table class="shop_table shop_table_responsive my_account_subscriptions">';
        echo '<thead><tr><th>' . esc_html__('Product', 'reordere-lite-subcriptions-for-woocommerce') . '</th><th>' . esc_html__('Start Date', 'reordere-lite-subcriptions-for-woocommerce') . '</th><th>' . esc_html__('End Date', 'reordere-lite-subcriptions-for-woocommerce') . '</th><th>' . esc_html__('Status', 'reordere-lite-subcriptions-for-woocommerce') . '</th><th>' . esc_html__('Actions', 'reordere-lite-subcriptions-for-woocommerce') . '</th></tr></thead>';
        echo '<tbody>';

        foreach ($subscriptions as $subscription) {
            $product_name = wc_get_product($subscription->product_id) ? wc_get_product($subscription->product_id)->get_name() : esc_html__('Unknown Product', 'reordere-lite-subcriptions-for-woocommerce');

            echo '<tr>';
            echo '<td>' . esc_html($product_name) . '</td>';
            echo '<td>' . esc_html(gmdate('Y-m-d', strtotime($subscription->subscription_start_date))) . '</td>';
            echo '<td>' . esc_html(gmdate('Y-m-d', strtotime($subscription->subscription_end_date))) . '</td>';
            echo '<td>' . 
            ($subscription->paused_until 
                ? esc_html(__('Paused until: ', 'reordere-lite-subcriptions-for-woocommerce') . gmdate('Y-m-d', strtotime($subscription->paused_until))) 
                : '<button class="button active-button">' . esc_html__('Active', 'reordere-lite-subcriptions-for-woocommerce') . '</button>'
            ) . '</td>';

            echo '<td>';
            if ($subscription->paused_until) {
                echo '<a href="#" class="button resume-subscription" data-order-id="' . esc_attr($subscription->order_id) . '">' . esc_html__('Resume', 'reordere-lite-subcriptions-for-woocommerce') . '</a>';
            } else {
                echo '<a href="#" class="button pause-subscription" data-order-id="' . esc_attr($subscription->order_id) . '">' . esc_html__('Pause', 'reordere-lite-subcriptions-for-woocommerce') . '</a>';
            }
            echo '</td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
    } else {
        echo '<p>' . esc_html__('You have no active subscriptions.', 'reordere-lite-subcriptions-for-woocommerce') . '</p>';
    }

    ?>
    <!-- Pause Subscription Modal -->
    <div id="subscription-modal" style="display:none;" title="<?php esc_attr_e('Pause Subscription', 'reordere-lite-subcriptions-for-woocommerce'); ?>">
        <form id="subscription-action-form">
            <input type="hidden" id="subscription_order_id" name="order_id" value="">

            <p><label for="subscription_date"><?php esc_html_e('Select Pause Until Date:', 'reordere-lite-subcriptions-for-woocommerce'); ?></label></p>
            <p><input type="date" id="subscription_date" name="subscription_date" required></p>

            <?php wp_nonce_field('subscription_nonce', 'subscription_nonce'); ?>
            <!-- <button type="submit" class="button">Submit</button> -->
        </form>
    </div>
    <?php
}
add_action('woocommerce_account_subscriptions_endpoint', 'rorder_subscriptions_tab_content');

/**
 * Handle the pause subscription AJAX request.
 */
add_action('wp_ajax_pause_subscription', function() {
    // Properly sanitize the nonce for verification
    check_ajax_referer('subscription_nonce', 'security'); // Validate nonce

    $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
    $pause_until = isset($_POST['pause_until']) ? sanitize_text_field(wp_unslash($_POST['pause_until'])) : '';

    if (empty($order_id) || empty($pause_until)) {
        wp_send_json_error(esc_html__('Invalid input data.', 'reordere-lite-subcriptions-for-woocommerce'));
    }
    $order = wc_get_order($order_id);
    if (!$order || (int)$order->get_user_id() !== get_current_user_id()) { wp_send_json_error(esc_html__('You are not allowed to modify this subscription.', 'reordere-lite-subcriptions-for-woocommerce')); }

    global $wpdb;
    $table_name = $wpdb->prefix . 'subscription_details';

    // Fetch current subscription details
    $subscription = wp_cache_get($order_id, 'subscription_details');
    if (false === $subscription) {
        $subscription = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE order_id = %d", $order_id));
        wp_cache_set($order_id, $subscription, 'subscription_details');
    }

    if (!$subscription) {
        wp_send_json_error(esc_html__('Subscription not found.', 'reordere-lite-subcriptions-for-woocommerce'));
    }

    $current_end_date = $subscription->subscription_end_date;

    // Ensure the date is valid
    $pause_until_timestamp = strtotime($pause_until);
    if (!$pause_until_timestamp || $pause_until_timestamp <= time()) {
        wp_send_json_error(esc_html__('Pause date must be in the future and valid.', 'reordere-lite-subcriptions-for-woocommerce'));
    }

    // Calculate pause duration
    $pause_duration = (int) ceil(($pause_until_timestamp - time()) / (60 * 60 * 24)); // Number of days
    $new_end_date = gmdate('Y-m-d H:i:s', strtotime($current_end_date . " + $pause_duration days"));

    // Update subscription in the database
    $result = $wpdb->update(
        $table_name,
        [
            'paused_until' => gmdate('Y-m-d H:i:s', $pause_until_timestamp), // Save in MySQL-compatible format
            'subscription_end_date' => $new_end_date
        ],
        ['order_id' => $order_id],
        ['%s', '%s'],
        ['%d']
    );

    if ($result !== false) {
        wp_send_json_success(esc_html__('Subscription paused successfully. End date adjusted.', 'reordere-lite-subcriptions-for-woocommerce'));
    } else {
        wp_send_json_error(esc_html__('Failed to pause subscription.', 'reordere-lite-subcriptions-for-woocommerce'));
    }
});

/**
 * Handle the resume subscription AJAX request.
 */
add_action('wp_ajax_resume_subscription', function() {
    // Properly sanitize the nonce for verification
    check_ajax_referer('subscription_nonce', 'security'); // Validate nonce

    $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;

    if (empty($order_id)) {
        wp_send_json_error(esc_html__('Invalid order ID.', 'reordere-lite-subcriptions-for-woocommerce'));
    }
    $order = wc_get_order($order_id);
    if (!$order || (int)$order->get_user_id() !== get_current_user_id()) { wp_send_json_error(esc_html__('You are not allowed to modify this subscription.', 'reordere-lite-subcriptions-for-woocommerce')); }

    global $wpdb;
    $table_name = $wpdb->prefix . 'subscription_details';

    $result = $wpdb->update(
        $table_name,
        [
            'paused_until' => null,
            'sub_status' => 'active'
        ],
        ['order_id' => $order_id],
        ['%s', '%s'],
        ['%d']
    );

    if ($result !== false) {
        wp_send_json_success(esc_html__('Subscription resumed successfully.', 'reordere-lite-subcriptions-for-woocommerce'));
    } else {
        wp_send_json_error(esc_html__('Failed to resume subscription.', 'reordere-lite-subcriptions-for-woocommerce'));
    }
});

/**
 * Reactivate paused subscriptions via cron job.
 */
add_action('rorder_reactivate_paused_subscriptions', function() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'subscription_details';

    $wpdb->query("
        UPDATE $table_name
        SET sub_status = 'active', paused_until = NULL
        WHERE sub_status = 'paused' AND paused_until < NOW()
    ");
});

// Schedule the cron event if not already scheduled
if (!wp_next_scheduled('rorder_reactivate_paused_subscriptions')) {
    wp_schedule_event(time(), 'hourly', 'rorder_reactivate_paused_subscriptions');
}

/**
 * Flush rewrite rules on theme switch.
 */
function rorder_subscriptions_flush_rewrite_rules() {
    rorder_subscriptions_tab_endpoint();  // Ensure the endpoint is registered
    flush_rewrite_rules();  // Flush rewrite rules
}
add_action('after_switch_theme', 'rorder_subscriptions_flush_rewrite_rules');

/**
 * Handle pause and skip subscription actions.
 */
function rorder_handle_pause_skip_subscription_actions() {
    // Check if user is logged in
    if (!is_user_logged_in()) {
        wp_send_json_error(array('message' => esc_html__('You must be logged in to perform this action.', 'reordere-lite-subcriptions-for-woocommerce')));
        wp_die();
    }

    global $wpdb;
    $table_name = $wpdb->prefix . 'subscription_details';

    // Verify nonce for security with proper sanitization
    if (!isset($_POST['subscription_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['subscription_nonce'])), 'subscription_nonce')) {
        wp_send_json_error(array('message' => esc_html__('Invalid request.', 'reordere-lite-subcriptions-for-woocommerce')));
        wp_die();  // Terminate script
    }

    // Check if we have the required fields
    if (!isset($_POST['action_type'], $_POST['order_id'], $_POST['subscription_date'])) {
        wp_send_json_error(array('message' => esc_html__('Required fields are missing.', 'reordere-lite-subcriptions-for-woocommerce')));
        wp_die();  // Terminate script
    }

    $action_type = sanitize_text_field($_POST['action_type']);
    $order_id = intval($_POST['order_id']);
    $subscription_date = sanitize_text_field(wp_unslash($_POST['subscription_date']));
    $order = wc_get_order($order_id);
    if (!$order || (int)$order->get_user_id() !== get_current_user_id()) { wp_send_json_error(array('message' => esc_html__('You are not allowed to modify this subscription.', 'reordere-lite-subcriptions-for-woocommerce'))); wp_die(); }

    // Handle Pause Subscription
    if ($action_type === 'pause') {
        $updated = $wpdb->update(
            $table_name,
            array('sub_status' => 'paused', 'subscription_end_date' => $subscription_date),
            array('order_id' => $order_id)
        );

        if ($updated !== false) {
            wp_send_json_success(array('message' => esc_html__('Subscription paused until ', 'reordere-lite-subcriptions-for-woocommerce') . esc_html($subscription_date), 'updated_date' => esc_html($subscription_date)));
        } else {
            wp_send_json_error(array('message' => esc_html__('Failed to pause subscription.', 'reordere-lite-subcriptions-for-woocommerce')));
        }
    }

    // Handle Skip Subscription
    if ($action_type === 'skip') {
        $current_end_date = $wpdb->get_var($wpdb->prepare("SELECT subscription_end_date FROM $table_name WHERE order_id = %d", $order_id));

        if ($current_end_date) {
            // Set the new end date
            $new_end_date = gmdate('Y-m-d', strtotime($subscription_date));

            $updated = $wpdb->update(
                $table_name,
                array('subscription_end_date' => $new_end_date),
                array('order_id' => $order_id)
            );

            if ($updated !== false) {
                wp_send_json_success(array('message' => esc_html__('Subscription extended to ', 'reordere-lite-subcriptions-for-woocommerce') . esc_html($new_end_date), 'updated_date' => esc_html($new_end_date)));
            } else {
                wp_send_json_error(array('message' => esc_html__('Failed to skip subscription.', 'reordere-lite-subcriptions-for-woocommerce')));
            }
        } else {
            wp_send_json_error(array('message' => esc_html__('Subscription not found.', 'reordere-lite-subcriptions-for-woocommerce')));
        }
    }

    wp_send_json_error(array('message' => esc_html__('Invalid action.', 'reordere-lite-subcriptions-for-woocommerce')));
}
add_action('wp_ajax_handle_subscription_action', 'rorder_handle_pause_skip_subscription_actions');







