<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Stripe Payment Helper for Subscription Products (No SDK)
 *
 * @package TS_Subscription
 */

class Rorder_Stripe_Payment_Helper {

    /**
     * Automatically charge the saved card for a given order.
     *
     * @param int $order_id The order ID.
     * @return bool|WP_Error True if the charge was successful, WP_Error otherwise.
     */
    public function rorder_ts_stripe_auto_charge_saved_card($order_id) {
        global $wpdb;

        $order = wc_get_order($order_id);

        $options = get_option('woocommerce_stripe_settings');
        $stripeKey = 'yes' === $options['testmode'] ? $options['test_secret_key'] : $options['secret_key'];

        $user_id = $order->get_user_id(); 

        $stripe_meta_key = $wpdb->prefix . '_stripe_customer_id';

        $stripe_customer_id = get_user_meta($user_id, $stripe_meta_key, true);

        if (!$order) {
            return new WP_Error('invalid_order', esc_html__('Invalid order ID', 'reordere-lite-subcriptions-for-woocommerce'));
        }

        // Get the customer's payment tokens
        $tokens = WC_Payment_Tokens::get_customer_tokens($order->get_user_id(), 'stripe');

        if (empty($tokens)) {
            return new WP_Error('no_token', esc_html__('No saved Stripe payment token found for this customer.', 'reordere-lite-subcriptions-for-woocommerce'));
        }

        // Get the first token (adjust logic if needed)
        $token = current($tokens);

        // Ensure we have a valid token
        if (!$token || !$token->get_token()) {
            return new WP_Error('invalid_token', esc_html__('Invalid or missing payment token.', 'reordere-lite-subcriptions-for-woocommerce'));
        }

        // Get the Stripe payment method ID from the token
        $stripe_payment_method_id = $token->get_token(); // This is the Stripe payment method ID

        // Create the payment intent
        $response = wp_remote_post('https://api.stripe.com/v1/payment_intents', [
            'method'    => 'POST',
            'headers'   => [
                'Authorization' => 'Bearer ' . $stripeKey,
                'Content-Type'  => 'application/x-www-form-urlencoded',
            ],
            'body'      => [
                'amount'               => $order->get_total() * 100, // Amount in cents
                'currency'             => strtolower(get_woocommerce_currency()),
                'payment_method'       => $stripe_payment_method_id,
                'confirmation_method'  => 'automatic',
                'confirm'              => 'true',
                'off_session'          => 'true',
                'customer'             => $stripe_customer_id,
            ],
        ]);

        // Handle the API response
        if (is_wp_error($response)) {
            $order->update_status('failed', esc_html__('Automatic charge failed: ', 'reordere-lite-subcriptions-for-woocommerce') . $response->get_error_message());
            return new WP_Error('http_error', $response->get_error_message());
        }

        $response_body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($response_body['error'])) {
            $order->update_status('failed', esc_html__('Automatic charge failed: ', 'reordere-lite-subcriptions-for-woocommerce') . $response_body['error']['message']);
            return new WP_Error('stripe_error', $response_body['error']['message']);
        }

        if ($response_body['status'] === 'succeeded') {
            // Mark the order as complete
            $order->payment_complete();
            $order->add_order_note(esc_html__('Automatic charge via saved card was successful.', 'reordere-lite-subcriptions-for-woocommerce'));
            $order->update_meta_data('_transaction_id', sanitize_text_field($response_body['id']));
            $order->update_meta_data('_stripe_source_id', sanitize_text_field($response_body['payment_method']));
            $order->update_meta_data('_stripe_charge_captured', 'yes');
            $order->update_meta_data('_stripe_currency', sanitize_text_field($response_body['currency']));
            $order->update_meta_data('_stripe_customer_id', sanitize_text_field($response_body['customer']));
            $order->save();
            return true;
        } else {
            $order->update_status('failed', esc_html__('Automatic charge failed: Unknown error.', 'reordere-lite-subcriptions-for-woocommerce'));
            return new WP_Error('unknown_error', esc_html__('Unknown error occurred.', 'reordere-lite-subcriptions-for-woocommerce'));
        }
    }
}

// Initialize the Stripe payment helper.
$stripe_payment_helper = new Rorder_Stripe_Payment_Helper();

