<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * PayPal Payment Helper for Subscription Products
 *
 * @package TS_Subscription
 */

class Rorder_Paypal_Payment_Helper {

    /**
     * Automatically charge the saved card for a given order using PayPal.
     *
     * @param int $order_id The order ID.
     * @return void
     */
    public function rorder_ts_paypal_auto_charge_saved_card($order_id) {
        global $wpdb;

        $order = wc_get_order($order_id);
        $amount = $order->get_total();

        // Ensure WooCommerce and the PayPal Gateway is available
        if (!class_exists('WooCommerce')) {
            return;
        }

        // Get the WooCommerce order
        $payer_id = get_user_meta(1, '_ppcp_target_customer_id', true);

        // Ensure the Payer ID exists and it's a PayPal payment method
        if ($payer_id) {
            // Get PayPal credentials from WooCommerce settings
            $paypal_options = get_option('woocommerce-ppcp-settings');
            $client_id = $paypal_options['client_id'];
            $client_secret = $paypal_options['client_secret'];
            $sandbox_enabled = $paypal_options['sandbox_on'] === '1';

            // Set PayPal endpoint based on sandbox setting
            $api_url = $sandbox_enabled ? 'https://api-m.sandbox.paypal.com' : 'https://api-m.paypal.com';

            // Get OAuth token
            $token = $this->rorder_get_paypal_access_token($client_id, $client_secret, $api_url);

            $data = [
                "intent" => "sale", // Sale for immediate capture
                "payer" => [
                    "payment_method" => "paypal",
                    "payer_info" => [
                        "payer_id" => $payer_id // PayPal Vault customer ID (stored after first authorization)
                    ]
                ],
                "transactions" => [
                    [
                        "amount" => [
                            "total" => $amount,
                            "currency" => "USD"
                        ],
                        "description" => "Subscription Payment"
                    ]
                ],
                "redirect_urls" => [
                    "return_url" => esc_url("https://example.com/success"), // URL to redirect after payment approval
                    "cancel_url" => esc_url("https://example.com/cancel")  // URL to redirect after payment cancellation
                ]
            ];

            $response = wp_remote_post("$api_url/v1/payments/payment", [
                'body' => wp_json_encode($data),
                'headers' => [
                    "Content-Type" => "application/json",
                    "Authorization" => "Bearer " . sanitize_text_field($token)
                ],
            ]);

            $response_data = json_decode(wp_remote_retrieve_body($response), true);

            // Check the state of the payment
            if (isset($response_data['state']) && $response_data['state'] == 'created') {
                echo esc_html("Payment created but not captured. Payment ID: " . $response_data['id'] . "\n");
                // Capture the payment
                $this->rorder_capture_payment($response_data['id'], $token);
            } elseif (isset($response_data['state']) && $response_data['state'] == 'approved') {
                echo esc_html("Payment already approved. Payment ID: " . $response_data['id'] . "\n");
            } else {
                echo esc_html("Failed to create payment. Response: \n");
                print_r(esc_html($response_data));
            }
        }
    }

    /**
     * Get PayPal access token.
     *
     * @param string $client_id The PayPal client ID.
     * @param string $client_secret The PayPal client secret.
     * @param string $api_url The PayPal API URL.
     * @return string|false The access token or false on failure.
     */
    public function rorder_get_paypal_access_token($client_id, $client_secret, $api_url) {
        // Make the API call to get access token
        $response = wp_remote_post("$api_url/v1/oauth2/token", [
            'body' => 'grant_type=client_credentials',
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode("$client_id:$client_secret"),
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
        ]);

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        return isset($data['access_token']) ? sanitize_text_field($data['access_token']) : false;
    }

    /**
     * Capture the PayPal payment.
     *
     * @param string $payment_id The PayPal payment ID.
     * @param string $access_token The PayPal access token.
     * @return array|false The response data or false on failure.
     */
    public function rorder_capture_payment($payment_id, $access_token) {
        if (!$access_token) {
            echo esc_html("Error: Failed to get access token.\n");
            return false;
        }

        // Initialize request for capturing the payment
        $response = wp_remote_post("https://api-m.sandbox.paypal.com/v1/payments/payment/$payment_id/execute", [
            'headers' => [
                "Content-Type" => "application/json",
                "Authorization" => "Bearer " . sanitize_text_field($access_token)
            ],
        ]);

        if (is_wp_error($response)) {
            echo esc_html('Error while capturing payment: ' . $response->get_error_message());
            return false;
        }

        $response_data = json_decode(wp_remote_retrieve_body($response), true);

        // Debugging: log the response from capture
        // file_put_contents('paypal_capture_response.log', print_r($response_data, true), FILE_APPEND);

        if (isset($response_data['state']) && $response_data['state'] == 'completed') {
            echo esc_html("Payment captured successfully. Payment ID: " . $response_data['id'] . "\n");
        } else {
            echo esc_html("Failed to capture payment. Response: \n");
            print_r(esc_html($response_data));
        }

        return $response_data;
    }
}

// Initialize the PayPal payment helper.
$paypal_payment_helper = new Rorder_Paypal_Payment_Helper();
