<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Payment Controller for Subscription Products
 *
 * @package TS_Subscription
 */

class Rorder_Payments_Controller {

    /**
     * Charge customers based on the payment method.
     *
     * @param string $payment_method The payment method.
     * @param int $order_id The order ID.
     * @return bool True if the charge was successful, false otherwise.
     */
    public function rorder_ts_charge_customers($payment_method, $order_id) {
        if ($payment_method == "stripe") {
            $response = $this->rorder_ts_charge_stripe_subscription_amount($order_id);
        } else if ($payment_method == "paypal") {
            $response = $this->rorder_ts_charge_paypal_subscription_amount($order_id);
        } else if ($payment_method == "razorpay") {
            $response = $this->rorder_ts_charge_razorpay_subscription_amount($order_id);
        } else {
            $response = true;
        }

        return $response;
    }

    /**
     * Charge the subscription amount using Stripe.
     *
     * @param int $order_id The order ID.
     * @return bool True if the charge was successful, false otherwise.
     */
    public function rorder_ts_charge_stripe_subscription_amount($order_id) {
        $order = wc_get_order($order_id);
        $amount = $order->calculate_totals();

        if ($amount == 0) {
            $response = true;
        } else {
            $initaite_stripe = new Rorder_Stripe_Payment_Helper();
            $response = $initaite_stripe->ts_stripe_auto_charge_saved_card($order_id);
        }

        return $response;
    }

    /**
     * Charge the subscription amount using PayPal.
     *
     * @param int $order_id The order ID.
     * @return bool True if the charge was successful, false otherwise.
     */
    public function rorder_ts_charge_paypal_subscription_amount($order_id) {
        $order = wc_get_order($order_id);
        $amount = $order->calculate_totals();

        if ($amount == 0) {
            $response = true;
        } else {
            $initaite_paypal = new Rorder_Paypal_Payment_Helper();
            $response = $initaite_paypal->ts_paypal_auto_charge_saved_card($order_id);
        }

        return $response;
    }

    /**
     * Charge the subscription amount using Razorpay.
     *
     * @param int $order_id The order ID.
     * @return bool True if the charge was successful, false otherwise.
     */
    public function rorder_ts_charge_razorpay_subscription_amount($order_id) {
        $order = wc_get_order($order_id);
        $amount = $order->calculate_totals();

        if ($amount == 0) {
            $response = true;
        } else {
            // Check if this subscription is managed by Razorpay native subscriptions
            global $wpdb;
            $subscription = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}subscription_details WHERE order_id = %d",
                $order_id
            ));
            
            // Check meta table for Razorpay subscription ID
            $razorpay_subscription_id = '';
            if ($subscription) {
                $razorpay_subscription_id = reordere_get_subscription_meta($subscription->id, 'razorpay_subscription_id');
            }
            
            if ($subscription && !empty($razorpay_subscription_id)) {
                // This subscription is managed by Razorpay's native subscription system
                // Razorpay will automatically charge and send webhook
                // We just need to wait for the webhook
                error_log(sprintf(
                    'Razorpay Payment Controller: Skipping manual charge for order %d - managed by Razorpay subscription %s',
                    $order_id,
                    $razorpay_subscription_id
                ));
                
                // Mark as pending - webhook will complete it
                $order->update_status('pending', 'Awaiting automatic payment from Razorpay subscription.');
                return true;
            }
            
            // Fallback to token-based charging (for subscriptions without Razorpay subscription)
            error_log('Razorpay Payment Controller: Using token-based charging for order ' . $order_id);
            require_once plugin_dir_path(__FILE__) . 'razorpay/class-razorpay-payment-helper.php';
            $initiate_razorpay = new Razorpay_Payment_Helper();
            $response = $initiate_razorpay->razorpay_auto_charge_saved_card($order_id);
        }

        return $response;
    }
}

// Initialize the payment controller.
$payment_controller = new Rorder_Payments_Controller();
