<?php
/**
 * Razorpay Token Storage Hook
 * 
 * Captures and stores Razorpay customer ID and token when payment is successful
 * This allows automatic charging for subscription renewals
 * 
 * @package ReordereLite
 * @subpackage Payments
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Save Razorpay customer and token data when payment is completed
 * Hooks into woocommerce_payment_complete to capture payment data
 */
add_action('woocommerce_payment_complete', 'rorder_save_razorpay_payment_data', 10, 1);

function rorder_save_razorpay_payment_data($order_id) {
    $order = wc_get_order($order_id);
    
    if (!$order) {
        return;
    }
    
    // Check if this is a Razorpay payment
    $payment_method = $order->get_payment_method();
    
    if ($payment_method !== 'razorpay') {
        return;
    }
    
    // Get user ID
    $user_id = $order->get_user_id();
    
    if (!$user_id) {
        error_log('Razorpay Token Storage: No user ID for order ' . $order_id);
        return;
    }
    
    // Check if we already have saved data for this user
    $existing_customer_id = get_user_meta($user_id, '_razorpay_customer_id', true);
    $existing_token_id = get_user_meta($user_id, '_razorpay_token_id', true);
    
    if (!empty($existing_customer_id) && !empty($existing_token_id)) {
        error_log('Razorpay Token Storage: User ' . $user_id . ' already has saved payment method');
        return;
    }
    
    // Get Razorpay payment ID from order
    $razorpay_payment_id = $order->get_meta('_razorpay_payment_id');
    
    if (empty($razorpay_payment_id)) {
        // Try alternative meta key
        $razorpay_payment_id = $order->get_meta('razorpay_payment_id');
    }
    
    if (empty($razorpay_payment_id)) {
        error_log('Razorpay Token Storage: No payment ID found for order ' . $order_id);
        return;
    }
    
    // Initialize Razorpay API to fetch payment details
    try {
        // Load Razorpay SDK
        if (!class_exists('Razorpay\Api\Api')) {
            $razorpay_plugin_path = WP_PLUGIN_DIR . '/woo-razorpay/razorpay-sdk/Razorpay.php';
            
            if (file_exists($razorpay_plugin_path)) {
                require_once $razorpay_plugin_path;
            } else {
                error_log('Razorpay Token Storage: Razorpay SDK not found');
                return;
            }
        }
        
        // Get Razorpay credentials
        $razorpay_settings = get_option('woocommerce_razorpay_settings');
        
        if (!$razorpay_settings || !isset($razorpay_settings['key_id']) || !isset($razorpay_settings['key_secret'])) {
            error_log('Razorpay Token Storage: Razorpay credentials not found');
            return;
        }
        
        $api = new Razorpay\Api\Api($razorpay_settings['key_id'], $razorpay_settings['key_secret']);
        
        // Fetch payment details
        $payment = $api->payment->fetch($razorpay_payment_id);
        
        if (!$payment) {
            error_log('Razorpay Token Storage: Could not fetch payment ' . $razorpay_payment_id);
            return;
        }
        
        // Extract customer ID and token from payment
        $customer_id = null;
        $token_id = null;
        
        // Check if payment has customer_id
        if (isset($payment->customer_id) && !empty($payment->customer_id)) {
            $customer_id = $payment->customer_id;
        }
        
        // Check if payment has token
        if (isset($payment->token_id) && !empty($payment->token_id)) {
            $token_id = $payment->token_id;
        } elseif (isset($payment->token) && !empty($payment->token)) {
            $token_id = $payment->token;
        }
        
        // If no customer or token in payment, try to create them for future use
        if (empty($customer_id)) {
            // Create Razorpay customer
            $customer_data = [
                'name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                'email' => $order->get_billing_email(),
                'contact' => $order->get_billing_phone(),
            ];
            
            try {
                $customer = $api->customer->create($customer_data);
                
                if ($customer && isset($customer->id)) {
                    $customer_id = $customer->id;
                    error_log('Razorpay Token Storage: Created customer ' . $customer_id . ' for user ' . $user_id);
                }
            } catch (Exception $e) {
                error_log('Razorpay Token Storage: Failed to create customer - ' . $e->getMessage());
            }
        }
        
        // Save customer ID and token if available
        if (!empty($customer_id)) {
            update_user_meta($user_id, '_razorpay_customer_id', sanitize_text_field($customer_id));
            error_log('Razorpay Token Storage: Saved customer ID ' . $customer_id . ' for user ' . $user_id);
        }
        
        if (!empty($token_id)) {
            update_user_meta($user_id, '_razorpay_token_id', sanitize_text_field($token_id));
            error_log('Razorpay Token Storage: Saved token ID ' . $token_id . ' for user ' . $user_id);
        }
        
        // Also save the payment method for reference
        if (!empty($customer_id) || !empty($token_id)) {
            update_user_meta($user_id, '_razorpay_payment_method_saved', 'yes');
            $order->add_order_note('Razorpay payment method saved for future subscription renewals');
        }
        
    } catch (Exception $e) {
        error_log('Razorpay Token Storage Exception: ' . $e->getMessage());
    }
}

/**
 * Alternative hook: Save Razorpay data when order status changes to completed/processing
 * This is a backup in case woocommerce_payment_complete doesn't fire
 */
add_action('woocommerce_order_status_changed', 'rorder_razorpay_order_status_changed', 10, 4);

function rorder_razorpay_order_status_changed($order_id, $old_status, $new_status, $order) {
    // Only process if order is now processing or completed
    if (!in_array($new_status, ['processing', 'completed'])) {
        return;
    }
    
    // Check if payment method is Razorpay
    $payment_method = $order->get_payment_method();
    
    if ($payment_method !== 'razorpay') {
        return;
    }
    
    // Check if we've already saved the payment method
    $user_id = $order->get_user_id();
    
    if (!$user_id) {
        return;
    }
    
    $payment_method_saved = get_user_meta($user_id, '_razorpay_payment_method_saved', true);
    
    if ($payment_method_saved === 'yes') {
        return; // Already saved
    }
    
    // Try to save payment data
    rorder_save_razorpay_payment_data($order_id);
}

/**
 * Hook to capture Razorpay webhook data for token storage
 * This captures data directly from Razorpay webhooks
 */
add_action('razorpay_payment_authorized', 'rorder_razorpay_webhook_capture_token', 10, 2);

function rorder_razorpay_webhook_capture_token($order_id, $payment_data) {
    if (!$order_id || !$payment_data) {
        return;
    }
    
    $order = wc_get_order($order_id);
    
    if (!$order) {
        return;
    }
    
    $user_id = $order->get_user_id();
    
    if (!$user_id) {
        return;
    }
    
    // Extract customer_id and token from webhook data
    if (isset($payment_data['customer_id']) && !empty($payment_data['customer_id'])) {
        update_user_meta($user_id, '_razorpay_customer_id', sanitize_text_field($payment_data['customer_id']));
        error_log('Razorpay Webhook: Saved customer ID from webhook for user ' . $user_id);
    }
    
    if (isset($payment_data['token_id']) && !empty($payment_data['token_id'])) {
        update_user_meta($user_id, '_razorpay_token_id', sanitize_text_field($payment_data['token_id']));
        error_log('Razorpay Webhook: Saved token ID from webhook for user ' . $user_id);
    } elseif (isset($payment_data['token']) && !empty($payment_data['token'])) {
        update_user_meta($user_id, '_razorpay_token_id', sanitize_text_field($payment_data['token']));
        error_log('Razorpay Webhook: Saved token from webhook for user ' . $user_id);
    }
}
