<?php
/**
 * Razorpay Subscription Manager
 * 
 * Manages native Razorpay subscriptions for automatic recurring payments
 * Uses Razorpay's subscription API to create plans and subscriptions
 * Handles webhooks for automatic renewal order creation
 * 
 * @package ReordereLite
 * @subpackage Payments
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Razorpay_Subscription_Manager {
    
    /**
     * Razorpay API instance
     */
    private $api;
    
    /**
     * Razorpay Key ID
     */
    private $key_id;
    
    /**
     * Razorpay Key Secret
     */
    private $key_secret;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->initialize_api();
        $this->setup_hooks();
    }
    
    /**
     * Initialize Razorpay API
     */
    private function initialize_api() {
        try {
            // Get Razorpay credentials
            $razorpay_settings = get_option('woocommerce_razorpay_settings');
            
            if (!$razorpay_settings || !isset($razorpay_settings['key_id']) || !isset($razorpay_settings['key_secret'])) {
                error_log('Razorpay Subscription Manager: Credentials not found');
                return;
            }
            
            $this->key_id = $razorpay_settings['key_id'];
            $this->key_secret = $razorpay_settings['key_secret'];
            
            // Load Razorpay SDK
            if (!class_exists('Razorpay\Api\Api')) {
                $razorpay_plugin_path = WP_PLUGIN_DIR . '/woo-razorpay/razorpay-sdk/Razorpay.php';
                
                if (file_exists($razorpay_plugin_path)) {
                    require_once $razorpay_plugin_path;
                } else {
                    error_log('Razorpay Subscription Manager: SDK not found');
                    return;
                }
            }
            
            $this->api = new Razorpay\Api\Api($this->key_id, $this->key_secret);
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Initialization failed - ' . $e->getMessage());
        }
    }
    
    /**
     * Setup WordPress hooks
     */
    private function setup_hooks() {
        // Hook into woocommerce_thankyou AFTER subscription is created
        // The subscription is created at priority 10, so we run at priority 20
        add_action('woocommerce_thankyou', array($this, 'create_razorpay_subscription_on_payment'), 20, 1);
        
        // Hook into order completion to create Razorpay subscription
        // Using multiple hooks to catch different payment completion stages
        // Priority 999 to run AFTER subscriptions are created in database
        add_action('woocommerce_payment_complete', array($this, 'create_razorpay_subscription_on_payment'), 999, 1);
        add_action('woocommerce_order_status_processing', array($this, 'create_razorpay_subscription_on_payment'), 999, 1);
        add_action('woocommerce_order_status_completed', array($this, 'create_razorpay_subscription_on_payment'), 999, 1);
        
        // Hook specifically into Razorpay payment authorization
        add_action('woocommerce_razorpay_payment_authorized', array($this, 'create_razorpay_subscription_on_payment'), 999, 1);
        
        // Hook into Razorpay plugin's payment success to capture payment data
        add_action('razorpay_payment_successful', array($this, 'on_razorpay_payment_successful'), 10, 2);
        
        // Retry hook for delayed subscription creation
        add_action('reordere_retry_razorpay_subscription', array($this, 'create_razorpay_subscription_on_payment'), 10, 1);
        
        // Hook into Razorpay webhooks
        add_action('admin_post_nopriv_rzp_wc_webhook', array($this, 'handle_razorpay_webhook'), 5);
    }
    
    /**
     * Capture Razorpay payment data when payment is successful
     * This hooks into the Razorpay plugin's payment flow
     * 
     * @param int $order_id WooCommerce order ID
     * @param array $payment_data Razorpay payment data
     */
    public function on_razorpay_payment_successful($order_id, $payment_data = array()) {
        try {
            error_log('Razorpay Subscription Manager: razorpay_payment_successful hook called for order ' . $order_id);
            error_log('Razorpay Subscription Manager: Payment data: ' . print_r($payment_data, true));
            
            // Store payment data in order meta for later use
            $order = wc_get_order($order_id);
            if ($order && !empty($payment_data)) {
                $order->update_meta_data('_razorpay_payment_data', $payment_data);
                $order->save();
            }
            
            // Try to create subscription
            $this->create_razorpay_subscription_on_payment($order_id);
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Error in on_razorpay_payment_successful - ' . $e->getMessage());
        }
    }
    
    /**
     * Create Razorpay subscription when payment is completed
     * 
     * @param int $order_id WooCommerce order ID
     */
    public function create_razorpay_subscription_on_payment($order_id) {
        try {
            error_log('Razorpay Subscription Manager: create_razorpay_subscription_on_payment called for order ' . $order_id);
            
            $order = wc_get_order($order_id);
            
            if (!$order) {
                error_log('Razorpay Subscription Manager: Order not found for ID ' . $order_id);
                return;
            }
            
            // Check if payment method is Razorpay
            $payment_method = $order->get_payment_method();
            error_log('Razorpay Subscription Manager: Payment method is ' . $payment_method);
            
            if ($payment_method !== 'razorpay') {
                error_log('Razorpay Subscription Manager: Skipping - not a Razorpay order');
                return;
            }
            
            // Check if order has subscription products
            $has_subscription = false;
            foreach ($order->get_items() as $item) {
                $product_id = $item->get_product_id();
                
                // Check for variation first, then parent product
                $variation_id = $item->get_variation_id();
                if ($variation_id) {
                    $is_subscription = get_post_meta($variation_id, '_enable_subscription', true);
                    error_log('Razorpay Subscription Manager: Variation ' . $variation_id . ' subscription status: ' . $is_subscription);
                    
                    if ($is_subscription !== 'yes') {
                        // Check parent product if variation not enabled
                        $is_subscription = get_post_meta($product_id, '_enable_subscription', true);
                        error_log('Razorpay Subscription Manager: Parent Product ' . $product_id . ' subscription status: ' . $is_subscription);
                    }
                } else {
                    $is_subscription = get_post_meta($product_id, '_enable_subscription', true);
                    error_log('Razorpay Subscription Manager: Product ' . $product_id . ' subscription status: ' . $is_subscription);
                }
                
                if ($is_subscription === 'yes') {
                    $has_subscription = true;
                    error_log('Razorpay Subscription Manager: Found subscription product!');
                    break;
                }
            }
            
            if (!$has_subscription) {
                error_log('Razorpay Subscription Manager: No subscription products in order');
                return;
            }
            
            // Get subscription from database
            global $wpdb;
            $subscription = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}subscription_details WHERE order_id = %d",
                $order_id
            ));
            
            if (!$subscription) {
                error_log('Razorpay Subscription Manager: No subscription found for order ' . $order_id . ' - will retry');
                
                // Schedule a retry after 30 seconds
                // The subscription might not be created yet
                if (!wp_next_scheduled('reordere_retry_razorpay_subscription', array($order_id))) {
                    wp_schedule_single_event(time() + 30, 'reordere_retry_razorpay_subscription', array($order_id));
                    error_log('Razorpay Subscription Manager: Scheduled retry for order ' . $order_id);
                }
                
                return;
            }
            
            error_log('Razorpay Subscription Manager: Found subscription ID ' . $subscription->id . ' for order ' . $order_id);
            
            // Check if Razorpay subscription already created
            $existing_razorpay_id = reordere_get_subscription_meta($subscription->id, 'razorpay_subscription_id');
            if (!empty($existing_razorpay_id)) {
                error_log('Razorpay Subscription Manager: Razorpay subscription already exists for subscription ' . $subscription->id);
                return;
            }
            
            // Create Razorpay subscription
            $razorpay_subscription_id = $this->create_razorpay_subscription($subscription, $order);
            
            if ($razorpay_subscription_id) {
                $order->add_order_note(sprintf(
                    'Razorpay Subscription created: %s. Automatic renewals enabled.',
                    $razorpay_subscription_id
                ));
                
                error_log(sprintf(
                    'Razorpay Subscription Manager: Created subscription %s for ReorderRe subscription %d',
                    $razorpay_subscription_id,
                    $subscription->id
                ));
            }
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Error in create_razorpay_subscription_on_payment - ' . $e->getMessage());
        }
    }
    
    /**
     * Create Razorpay subscription
     * 
     * @param object $subscription Subscription details from database
     * @param WC_Order $order WooCommerce order
     * @return string|false Razorpay subscription ID or false on failure
     */
    public function create_razorpay_subscription($subscription, $order) {
        try {
            if (!$this->api) {
                error_log('Razorpay Subscription Manager: API not initialized');
                return false;
            }
            
            // Step 1: Get or create Plan
            $plan_id = $this->get_or_create_plan($subscription);
            
            if (!$plan_id) {
                error_log('Razorpay Subscription Manager: Failed to create plan');
                return false;
            }
            
            // Step 2: Get or create Customer
            $customer_id = $this->get_or_create_customer($order);
            
            if (!$customer_id) {
                error_log('Razorpay Subscription Manager: Failed to create customer');
                return false;
            }
            
            error_log('Razorpay Subscription Manager: Using customer_id: ' . $customer_id);
            
            // Step 3: Calculate total cycles from subscription dates
            $total_count = $this->calculate_total_cycles($subscription);
            
            error_log(sprintf(
                'Razorpay Subscription Manager: Calculated total_count: %d (from %s to %s, every %d %s)',
                $total_count,
                $subscription->subscription_start_date,
                $subscription->subscription_end_date,
                $subscription->subscription_period,
                $subscription->subscription_duration
            ));
            
            // Step 4: Calculate start timestamp (next billing date)
            $start_at = $this->get_start_timestamp($subscription);
            
            // Step 5: Create Subscription on Razorpay
            $subscription_data = array(
                'customer_id' => $customer_id,
                'plan_id' => $plan_id,
                'quantity' => !empty($subscription->subscription_quantity) ? (int)$subscription->subscription_quantity : 1,
                'total_count' => $total_count, // Always provide total_count (calculated from dates)
                'customer_notify' => 1,
                'notes' => array(
                    'reordere_subscription_id' => $subscription->id,
                    'woocommerce_order_id' => $order->get_id(),
                    'user_id' => $order->get_user_id(),
                    'source' => 'ReorderRe Subscription Plugin'
                )
            );
            
            error_log('Razorpay Subscription Manager: Subscription data prepared: ' . print_r($subscription_data, true));
            
            // Add start_at if next billing date is in future
            if ($start_at && $start_at > time()) {
                $subscription_data['start_at'] = $start_at;
            }
            
            error_log('Razorpay Subscription Manager: Creating subscription with data: ' . json_encode($subscription_data));
            
            $razorpay_subscription = $this->api->subscription->create($subscription_data);
            
            if (!$razorpay_subscription || !isset($razorpay_subscription['id'])) {
                error_log('Razorpay Subscription Manager: Invalid response from Razorpay');
                return false;
            }
            
            // Step 6: Save Razorpay subscription details to meta table
            reordere_update_subscription_meta($subscription->id, 'razorpay_subscription_id', $razorpay_subscription['id']);
            reordere_update_subscription_meta($subscription->id, 'razorpay_customer_id', $customer_id);
            reordere_update_subscription_meta($subscription->id, 'razorpay_plan_id', $plan_id);
            
            // Also save to order meta for quick reference
            $order->update_meta_data('_razorpay_subscription_id', $razorpay_subscription['id']);
            $order->save();
            
            return $razorpay_subscription['id'];
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in create_razorpay_subscription - ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get or create Razorpay Plan
     * 
     * @param object $subscription Subscription details
     * @return string|false Plan ID or false on failure
     */
    private function get_or_create_plan($subscription) {
        try {
            // Generate unique plan hash based on amount, period, and interval
            $plan_hash = md5(sprintf(
                '%s_%s_%s_%s',
                $subscription->subscription_price,
                $subscription->subscription_period,
                $subscription->subscription_duration,
                get_woocommerce_currency()
            ));
            
            // Check if plan exists in WordPress options
            $plan_id = get_option('reordere_razorpay_plan_' . $plan_hash);
            
            if ($plan_id) {
                // Verify plan exists on Razorpay
                try {
                    $plan = $this->api->plan->fetch($plan_id);
                    error_log('Razorpay Subscription Manager: Using existing plan ' . $plan_id);
                    return $plan['id'];
                } catch (Exception $e) {
                    // Plan doesn't exist on Razorpay, create new one
                    error_log('Razorpay Subscription Manager: Cached plan not found, creating new one');
                }
            }
            
            // Convert period to Razorpay format
            $period = $this->convert_period_to_razorpay($subscription->subscription_duration);
            $interval = (int)$subscription->subscription_period;
            
            if ($interval < 1) {
                $interval = 1;
            }
            
            // Create new plan
            $plan_data = array(
                'period' => $period,
                'interval' => $interval,
                'item' => array(
                    'name' => sprintf(
                        'Subscription - Every %d %s - %s',
                        $interval,
                        ucfirst($subscription->subscription_duration),
                        get_woocommerce_currency() . ' ' . number_format($subscription->subscription_price, 2)
                    ),
                    'amount' => (int)($subscription->subscription_price * 100), // Convert to paise
                    'currency' => get_woocommerce_currency()
                )
            );
            
            error_log('Razorpay Subscription Manager: Creating plan with data: ' . json_encode($plan_data));
            
            $plan = $this->api->plan->create($plan_data);
            
            if (!$plan || !isset($plan['id'])) {
                error_log('Razorpay Subscription Manager: Invalid plan response');
                return false;
            }
            
            // Save plan ID for future use
            update_option('reordere_razorpay_plan_' . $plan_hash, $plan['id']);
            
            error_log('Razorpay Subscription Manager: Created plan ' . $plan['id']);
            
            return $plan['id'];
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in get_or_create_plan - ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get or create Razorpay Customer
     * 
     * @param WC_Order $order WooCommerce order
     * @return string|false Customer ID or false on failure
     */
    private function get_or_create_customer($order) {
        try {
            $user_id = $order->get_user_id();
            
            // First, try to get customer from Razorpay payment data
            $payment_data = $order->get_meta('_razorpay_payment_data');
            $razorpay_payment_id = null;
            
            // Try multiple ways to get payment ID
            if (!empty($payment_data) && isset($payment_data['razorpay_payment_id'])) {
                $razorpay_payment_id = $payment_data['razorpay_payment_id'];
            } elseif ($order->get_transaction_id()) {
                // Transaction ID is often the payment ID
                $razorpay_payment_id = $order->get_transaction_id();
            } elseif ($order->get_meta('_razorpay_payment_id')) {
                $razorpay_payment_id = $order->get_meta('_razorpay_payment_id');
            }
            
            if ($razorpay_payment_id) {
                try {
                    error_log('Razorpay Subscription Manager: Fetching payment to get customer: ' . $razorpay_payment_id);
                    $payment = $this->api->payment->fetch($razorpay_payment_id);
                    
                    error_log('Razorpay Subscription Manager: Payment data: ' . print_r($payment, true));
                    
                    if (isset($payment['customer_id']) && !empty($payment['customer_id'])) {
                        error_log('Razorpay Subscription Manager: Found customer from payment: ' . $payment['customer_id']);
                        
                        // Save for future use
                        if ($user_id) {
                            update_user_meta($user_id, '_razorpay_customer_id', $payment['customer_id']);
                        }
                        
                        return $payment['customer_id'];
                    } else {
                        error_log('Razorpay Subscription Manager: No customer_id in payment object, will create new customer');
                    }
                } catch (Exception $e) {
                    error_log('Razorpay Subscription Manager: Error fetching payment: ' . $e->getMessage());
                }
            } else {
                error_log('Razorpay Subscription Manager: No payment ID found in order meta');
            }
            
            if (!$user_id) {
                // Guest checkout - create customer anyway
                error_log('Razorpay Subscription Manager: Creating customer for guest order');
            } else {
                // Check if customer already exists
                $customer_id = get_user_meta($user_id, '_razorpay_customer_id', true);
                
                if ($customer_id) {
                    // Verify customer exists on Razorpay
                    try {
                        $customer = $this->api->customer->fetch($customer_id);
                        error_log('Razorpay Subscription Manager: Using existing customer ' . $customer_id);
                        return $customer['id'];
                    } catch (Exception $e) {
                        // Customer doesn't exist, create new one
                        error_log('Razorpay Subscription Manager: Cached customer not found, creating new one');
                    }
                }
            }
            
            // Create new customer
            $customer_data = array(
                'name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                'email' => $order->get_billing_email(),
                'contact' => $order->get_billing_phone(),
                'fail_existing' => '0'  // Return existing customer if email matches
            );
            
            error_log('Razorpay Subscription Manager: Creating customer with email ' . $order->get_billing_email());
            
            $customer = $this->api->customer->create($customer_data);
            
            if (!$customer || !isset($customer['id'])) {
                error_log('Razorpay Subscription Manager: Invalid customer response');
                return false;
            }
            
            // Save customer ID
            if ($user_id) {
                update_user_meta($user_id, '_razorpay_customer_id', $customer['id']);
            }
            
            // Also save to order meta
            $order->update_meta_data('_razorpay_customer_id', $customer['id']);
            $order->save();
            
            error_log('Razorpay Subscription Manager: Created customer ' . $customer['id']);
            
            return $customer['id'];
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in get_or_create_customer - ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Convert subscription period to Razorpay format
     * 
     * @param string $period Subscription period (day, week, month, year)
     * @return string Razorpay period format
     */
    private function convert_period_to_razorpay($period) {
        $map = array(
            'day' => 'daily',
            'days' => 'daily',
            'week' => 'weekly',
            'weeks' => 'weekly',
            'month' => 'monthly',
            'months' => 'monthly',
            'year' => 'yearly',
            'years' => 'yearly'
        );
        
        $period_lower = strtolower(trim($period));
        
        return isset($map[$period_lower]) ? $map[$period_lower] : 'monthly';
    }
    
    /**
     * Calculate total billing cycles
     * 
     * @param object $subscription Subscription details
     * @return int Total cycles (calculated from start and end dates)
     */
    private function calculate_total_cycles($subscription) {
        // Calculate cycles from start and end dates
        if (!empty($subscription->subscription_start_date) && !empty($subscription->subscription_end_date)) {
            $start = strtotime($subscription->subscription_start_date);
            $end = strtotime($subscription->subscription_end_date);
            $period = (int)$subscription->subscription_period;
            $duration = $subscription->subscription_duration;
            
            if ($period < 1) {
                $period = 1;
            }
            
            // Calculate total months/days/weeks/years between start and end
            $diff = $end - $start;
            
            switch (strtolower($duration)) {
                case 'day':
                case 'days':
                    $total_days = floor($diff / (60 * 60 * 24));
                    $cycles = ceil($total_days / $period);
                    break;
                    
                case 'week':
                case 'weeks':
                    $total_weeks = floor($diff / (60 * 60 * 24 * 7));
                    $cycles = ceil($total_weeks / $period);
                    break;
                    
                case 'month':
                case 'months':
                    $start_date = new DateTime($subscription->subscription_start_date);
                    $end_date = new DateTime($subscription->subscription_end_date);
                    $interval = $start_date->diff($end_date);
                    $total_months = ($interval->y * 12) + $interval->m;
                    $cycles = ceil($total_months / $period);
                    break;
                    
                case 'year':
                case 'years':
                    $start_date = new DateTime($subscription->subscription_start_date);
                    $end_date = new DateTime($subscription->subscription_end_date);
                    $interval = $start_date->diff($end_date);
                    $total_years = $interval->y;
                    $cycles = ceil($total_years / $period);
                    break;
                    
                default:
                    $cycles = 12; // Default to 12 cycles if unknown duration
            }
            
            // Ensure at least 1 cycle
            return max(1, (int)$cycles);
        }
        
        // Default: 12 billing cycles (1 year for monthly, etc.)
        return 12;
    }
    
    /**
     * Get subscription start timestamp
     * 
     * @param object $subscription Subscription details
     * @return int|null Unix timestamp or null
     */
    private function get_start_timestamp($subscription) {
        if (empty($subscription->next_payment_date)) {
            return null;
        }
        
        $timestamp = strtotime($subscription->next_payment_date);
        
        // Only return if in future
        if ($timestamp > time()) {
            return $timestamp;
        }
        
        return null;
    }
    
    /**
     * Handle Razorpay webhooks
     */
    public function handle_razorpay_webhook() {
        try {
            $payload = file_get_contents('php://input');
            
            if (empty($payload)) {
                error_log('Razorpay Subscription Manager: Empty webhook payload');
                return;
            }
            
            $data = json_decode($payload, true);
            
            if (!$data || !isset($data['event'])) {
                error_log('Razorpay Subscription Manager: Invalid webhook data');
                return;
            }
            
            $event = $data['event'];
            
            error_log('Razorpay Subscription Manager: Received webhook event: ' . $event);
            error_log('Razorpay Subscription Manager: Full webhook payload: ' . print_r($data, true));
            
            // Handle different webhook events
            switch ($event) {
                case 'subscription.charged':
                    $this->handle_subscription_charged($data);
                    break;
                    
                case 'subscription.cancelled':
                    $this->handle_subscription_cancelled($data);
                    break;
                    
                case 'subscription.paused':
                    $this->handle_subscription_paused($data);
                    break;
                    
                case 'subscription.resumed':
                    $this->handle_subscription_resumed($data);
                    break;
                    
                case 'payment.failed':
                    $this->handle_payment_failed($data);
                    break;
                    
                default:
                    error_log('Razorpay Subscription Manager: Unhandled event type: ' . $event);
                    break;
            }
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in handle_razorpay_webhook - ' . $e->getMessage());
        }
    }
    
    /**
     * Handle subscription.charged webhook
     * 
     * @param array $data Webhook data
     */
    private function handle_subscription_charged($data) {
        try {
            if (!isset($data['payload']['subscription']['entity']['notes']['reordere_subscription_id'])) {
                // Not our subscription
                return;
            }
            
            $payment_id = $data['payload']['payment']['entity']['id'];
            $razorpay_subscription_id = $data['payload']['subscription']['entity']['id'];
            $reordere_subscription_id = $data['payload']['subscription']['entity']['notes']['reordere_subscription_id'];
            $paid_count = isset($data['payload']['subscription']['entity']['paid_count']) ? 
                         $data['payload']['subscription']['entity']['paid_count'] : 0;
            
            error_log(sprintf(
                'Razorpay Subscription Manager: subscription.charged - Payment: %s, Razorpay Sub: %s, ReorderRe Sub: %d, Paid Count: %d',
                $payment_id,
                $razorpay_subscription_id,
                $reordere_subscription_id,
                $paid_count
            ));
            
            // Get subscription from database
            global $wpdb;
            $subscription = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}subscription_details WHERE id = %d",
                $reordere_subscription_id
            ));
            
            if (!$subscription) {
                error_log('Razorpay Subscription Manager: Subscription not found: ' . $reordere_subscription_id);
                return;
            }
            
            // Check if this is first payment or renewal
            if ($paid_count <= 1) {
                // First payment - just mark original order as complete
                error_log('Razorpay Subscription Manager: First payment for subscription ' . $reordere_subscription_id);
                
                $order = wc_get_order($subscription->order_id);
                if ($order && $order->needs_payment()) {
                    $order->payment_complete($payment_id);
                    $order->add_order_note('First subscription payment completed via Razorpay. Automatic renewals enabled.');
                }
            } else {
                // Renewal payment - create renewal order
                error_log('Razorpay Subscription Manager: Renewal payment (cycle ' . $paid_count . ') for subscription ' . $reordere_subscription_id);
                
                $renewal_order = $this->create_renewal_order($subscription, $payment_id);
                
                if ($renewal_order) {
                    error_log('Razorpay Subscription Manager: Renewal order created: ' . $renewal_order->get_id());
                } else {
                    error_log('Razorpay Subscription Manager: Failed to create renewal order');
                }
            }
            
            // Update next payment date
            $this->update_next_payment_date($subscription);
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in handle_subscription_charged - ' . $e->getMessage());
        }
    }
    
    /**
     * Create renewal order
     * 
     * @param object $subscription Subscription details
     * @param string $payment_id Razorpay payment ID
     * @return WC_Order|false Renewal order or false on failure
     */
    private function create_renewal_order($subscription, $payment_id) {
        try {
            $original_order = wc_get_order($subscription->order_id);
            
            if (!$original_order) {
                error_log('Razorpay Subscription Manager: Original order not found: ' . $subscription->order_id);
                return false;
            }
            
            // Create new order
            $renewal_order = wc_create_order(array(
                'customer_id' => $original_order->get_customer_id(),
                'created_via' => 'reordere_razorpay_subscription'
            ));
            
            // Add products from subscription
            $product_ids = maybe_unserialize($subscription->subscription_product_ids);
            
            if (!is_array($product_ids)) {
                $product_ids = array($product_ids);
            }
            
            foreach ($product_ids as $product_id) {
                $product = wc_get_product($product_id);
                
                if ($product) {
                    $quantity = !empty($subscription->subscription_quantity) ? 
                               (int)$subscription->subscription_quantity : 1;
                    
                    $renewal_order->add_product($product, $quantity);
                }
            }
            
            // Set payment method
            $renewal_order->set_payment_method('razorpay');
            $renewal_order->set_payment_method_title('Razorpay (Automatic Subscription)');
            
            // Set transaction ID
            $renewal_order->set_transaction_id($payment_id);
            
            // Copy addresses from original order
            $renewal_order->set_address($original_order->get_address('billing'), 'billing');
            $renewal_order->set_address($original_order->get_address('shipping'), 'shipping');
            
            // Calculate totals
            $renewal_order->calculate_totals();
            
            // Mark as paid
            $renewal_order->payment_complete($payment_id);
            
            // Add notes
            $renewal_order->add_order_note(sprintf(
                'Renewal order created automatically via Razorpay Subscription. Payment ID: %s, Original Order: #%d',
                $payment_id,
                $subscription->order_id
            ));
            
            // Get Razorpay subscription ID from meta table
            $razorpay_sub_id = reordere_get_subscription_meta($subscription->id, 'razorpay_subscription_id');
            
            // Link to subscription
            $renewal_order->update_meta_data('_reordere_subscription_id', $subscription->id);
            $renewal_order->update_meta_data('_razorpay_subscription_id', $razorpay_sub_id);
            $renewal_order->update_meta_data('_is_renewal_order', 'yes');
            $renewal_order->save();
            
            // Update subscription cron table
            global $wpdb;
            $wpdb->insert(
                $wpdb->prefix . 'subscription_cron_order_details',
                array(
                    'subscription_id' => $subscription->id,
                    'order_id' => $renewal_order->get_id(),
                    'status' => 'completed',
                    'created_at' => current_time('mysql'),
                    'payment_method' => 'razorpay',
                    'payment_id' => $payment_id
                )
            );
            
            return $renewal_order;
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in create_renewal_order - ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update next payment date
     * 
     * @param object $subscription Subscription details
     */
    private function update_next_payment_date($subscription) {
        try {
            global $wpdb;
            
            // Calculate next payment date
            $duration = $subscription->subscription_duration; // 'day', 'week', 'month', 'year'
            $period = (int)$subscription->subscription_period; // 1, 3, 6, 12, etc.
            
            if ($period < 1) {
                $period = 1;
            }
            
            $current_date = current_time('mysql');
            $next_date = date('Y-m-d H:i:s', strtotime("+{$period} {$duration}", strtotime($current_date)));
            
            $wpdb->update(
                $wpdb->prefix . 'subscription_details',
                array('next_payment_date' => $next_date),
                array('id' => $subscription->id)
            );
            
            error_log(sprintf(
                'Razorpay Subscription Manager: Updated next payment date for subscription %d to %s',
                $subscription->id,
                $next_date
            ));
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in update_next_payment_date - ' . $e->getMessage());
        }
    }
    
    /**
     * Handle subscription.cancelled webhook
     * 
     * @param array $data Webhook data
     */
    private function handle_subscription_cancelled($data) {
        try {
            if (!isset($data['payload']['subscription']['entity']['notes']['reordere_subscription_id'])) {
                return;
            }
            
            $reordere_subscription_id = $data['payload']['subscription']['entity']['notes']['reordere_subscription_id'];
            
            error_log('Razorpay Subscription Manager: Subscription cancelled: ' . $reordere_subscription_id);
            
            // Update subscription status
            global $wpdb;
            $wpdb->update(
                $wpdb->prefix . 'subscription_details',
                array('status' => 'cancelled'),
                array('id' => $reordere_subscription_id)
            );
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in handle_subscription_cancelled - ' . $e->getMessage());
        }
    }
    
    /**
     * Handle subscription.paused webhook
     * 
     * @param array $data Webhook data
     */
    private function handle_subscription_paused($data) {
        try {
            if (!isset($data['payload']['subscription']['entity']['notes']['reordere_subscription_id'])) {
                return;
            }
            
            $reordere_subscription_id = $data['payload']['subscription']['entity']['notes']['reordere_subscription_id'];
            
            error_log('Razorpay Subscription Manager: Subscription paused: ' . $reordere_subscription_id);
            
            // Update subscription status
            global $wpdb;
            $wpdb->update(
                $wpdb->prefix . 'subscription_details',
                array('status' => 'paused'),
                array('id' => $reordere_subscription_id)
            );
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in handle_subscription_paused - ' . $e->getMessage());
        }
    }
    
    /**
     * Handle subscription.resumed webhook
     * 
     * @param array $data Webhook data
     */
    private function handle_subscription_resumed($data) {
        try {
            if (!isset($data['payload']['subscription']['entity']['notes']['reordere_subscription_id'])) {
                return;
            }
            
            $reordere_subscription_id = $data['payload']['subscription']['entity']['notes']['reordere_subscription_id'];
            
            error_log('Razorpay Subscription Manager: Subscription resumed: ' . $reordere_subscription_id);
            
            // Update subscription status
            global $wpdb;
            $wpdb->update(
                $wpdb->prefix . 'subscription_details',
                array('status' => 'active'),
                array('id' => $reordere_subscription_id)
            );
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in handle_subscription_resumed - ' . $e->getMessage());
        }
    }
    
    /**
     * Handle payment.failed webhook
     * 
     * @param array $data Webhook data
     */
    private function handle_payment_failed($data) {
        try {
            // Check if it's a subscription payment
            if (!isset($data['payload']['payment']['entity']['notes']['reordere_subscription_id'])) {
                return;
            }
            
            $payment_id = $data['payload']['payment']['entity']['id'];
            $reordere_subscription_id = $data['payload']['payment']['entity']['notes']['reordere_subscription_id'];
            
            error_log(sprintf(
                'Razorpay Subscription Manager: Payment failed for subscription %d, Payment: %s',
                $reordere_subscription_id,
                $payment_id
            ));
            
            // Get subscription
            global $wpdb;
            $subscription = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}subscription_details WHERE id = %d",
                $reordere_subscription_id
            ));
            
            if ($subscription) {
                // Send notification to admin/customer about failed payment
                $this->send_payment_failed_notification($subscription, $payment_id);
            }
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in handle_payment_failed - ' . $e->getMessage());
        }
    }
    
    /**
     * Send payment failed notification
     * 
     * @param object $subscription Subscription details
     * @param string $payment_id Payment ID
     */
    private function send_payment_failed_notification($subscription, $payment_id) {
        try {
            $order = wc_get_order($subscription->order_id);
            
            if (!$order) {
                return;
            }
            
            $to = $order->get_billing_email();
            $subject = sprintf(
                'Subscription Renewal Payment Failed - Order #%d',
                $subscription->order_id
            );
            
            $message = sprintf(
                "Dear Customer,\n\nYour subscription renewal payment has failed.\n\nSubscription ID: %d\nPayment ID: %s\n\nPlease update your payment method or contact support.\n\nThank you.",
                $subscription->id,
                $payment_id
            );
            
            wp_mail($to, $subject, $message);
            
            error_log('Razorpay Subscription Manager: Sent payment failed notification for subscription ' . $subscription->id);
            
        } catch (Exception $e) {
            error_log('Razorpay Subscription Manager: Exception in send_payment_failed_notification - ' . $e->getMessage());
        }
    }
}

// Initialize the Razorpay Subscription Manager
function initialize_razorpay_subscription_manager() {
    return new Razorpay_Subscription_Manager();
}

add_action('plugins_loaded', 'initialize_razorpay_subscription_manager');
