<?php
/**
 * Razorpay Payment Helper for Subscription Renewals
 * 
 * Handles automatic charging of Razorpay payment methods for subscription renewals
 * 
 * @package ReordereLite
 * @subpackage Payments
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Razorpay_Payment_Helper {

    /**
     * Razorpay API instance
     */
    private $api;

    /**
     * Razorpay Key ID
     */
    private $key_id;

    /**
     * Razorpay Key Secret
     */
    private $key_secret;

    /**
     * Constructor
     */
    public function __construct() {
        $this->initialize_razorpay_credentials();
    }

    /**
     * Initialize Razorpay credentials from plugin settings
     */
    private function initialize_razorpay_credentials() {
        // Get Razorpay settings from WooCommerce
        $razorpay_settings = get_option('woocommerce_razorpay_settings');
        
        if ($razorpay_settings && isset($razorpay_settings['key_id']) && isset($razorpay_settings['key_secret'])) {
            $this->key_id = $razorpay_settings['key_id'];
            $this->key_secret = $razorpay_settings['key_secret'];
        } else {
            error_log('Razorpay credentials not found in WooCommerce settings');
        }
    }

    /**
     * Get Razorpay API instance
     * 
     * @return Razorpay\Api\Api|false
     */
    private function get_razorpay_api() {
        if (!$this->key_id || !$this->key_secret) {
            error_log('Razorpay API credentials are missing');
            return false;
        }

        try {
            // Check if Razorpay SDK is loaded
            if (!class_exists('Razorpay\Api\Api')) {
                // Load Razorpay SDK from the woo-razorpay plugin
                $razorpay_plugin_path = WP_PLUGIN_DIR . '/woo-razorpay/razorpay-sdk/Razorpay.php';
                
                if (file_exists($razorpay_plugin_path)) {
                    require_once $razorpay_plugin_path;
                } else {
                    error_log('Razorpay SDK not found. Please ensure woo-razorpay plugin is installed.');
                    return false;
                }
            }

            $this->api = new Razorpay\Api\Api($this->key_id, $this->key_secret);
            return $this->api;
        } catch (Exception $e) {
            error_log('Razorpay API initialization failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Auto charge saved Razorpay payment method for subscription renewal
     * 
     * @param int $order_id WooCommerce order ID
     * @return bool True if payment successful, false otherwise
     */
    public function razorpay_auto_charge_saved_card($order_id) {
        try {
            $order = wc_get_order($order_id);
            
            if (!$order) {
                error_log('Razorpay Auto Charge: Order not found - ' . $order_id);
                return false;
            }

            // Get user ID
            $user_id = $order->get_user_id();
            
            if (!$user_id) {
                error_log('Razorpay Auto Charge: No user ID found for order - ' . $order_id);
                $order->add_order_note('Razorpay renewal failed: No user ID associated with order');
                return false;
            }

            // Get saved Razorpay customer ID and token
            $customer_id = get_user_meta($user_id, '_razorpay_customer_id', true);
            $token_id = get_user_meta($user_id, '_razorpay_token_id', true);

            if (empty($customer_id) || empty($token_id)) {
                error_log('Razorpay Auto Charge: No saved payment method found for user - ' . $user_id);
                $order->add_order_note('Razorpay renewal failed: No saved payment method found');
                $order->update_status('failed', 'Razorpay: No saved payment method');
                return false;
            }

            // Get order details
            $amount = $order->get_total();
            $currency = $order->get_currency();
            $customer_email = $order->get_billing_email();
            $customer_phone = $order->get_billing_phone();

            // Convert amount to paise (Razorpay uses smallest currency unit)
            $amount_in_paise = $this->convert_to_paise($amount, $currency);

            // Initialize Razorpay API
            $api = $this->get_razorpay_api();
            
            if (!$api) {
                error_log('Razorpay Auto Charge: Failed to initialize Razorpay API');
                $order->add_order_note('Razorpay renewal failed: API initialization error');
                $order->update_status('failed', 'Razorpay: API initialization failed');
                return false;
            }

            // Create recurring payment
            $payment_data = [
                'email' => $customer_email,
                'contact' => $customer_phone,
                'amount' => $amount_in_paise,
                'currency' => $currency,
                'token' => $token_id,
                'customer_id' => $customer_id,
                'description' => 'Subscription Renewal - Order #' . $order_id,
            ];

            error_log('Razorpay Auto Charge: Creating recurring payment for order ' . $order_id);
            error_log('Razorpay Payment Data: ' . json_encode($payment_data));

            $payment = $api->payment->createRecurring($payment_data);

            error_log('Razorpay Payment Response: ' . json_encode($payment));

            // Check payment status
            if ($payment && isset($payment->id)) {
                $payment_id = $payment->id;
                $payment_status = $payment->status;

                // Update order with payment ID
                $order->update_meta_data('_razorpay_payment_id', $payment_id);
                $order->save();

                if ($payment_status === 'captured' || $payment_status === 'authorized') {
                    // Payment successful
                    $order->payment_complete($payment_id);
                    $order->add_order_note(
                        sprintf(
                            'Razorpay subscription renewal payment successful. Payment ID: %s, Status: %s',
                            $payment_id,
                            $payment_status
                        )
                    );
                    
                    error_log('Razorpay Auto Charge: Payment successful for order ' . $order_id . ' - Payment ID: ' . $payment_id);
                    return true;
                    
                } else {
                    // Payment pending or other status
                    $order->update_status('on-hold', sprintf('Razorpay payment %s (Payment ID: %s)', $payment_status, $payment_id));
                    $order->add_order_note(
                        sprintf(
                            'Razorpay subscription renewal payment status: %s. Payment ID: %s',
                            $payment_status,
                            $payment_id
                        )
                    );
                    
                    error_log('Razorpay Auto Charge: Payment status ' . $payment_status . ' for order ' . $order_id);
                    return false;
                }
                
            } else {
                // Payment creation failed
                error_log('Razorpay Auto Charge: Payment creation failed for order ' . $order_id);
                $order->add_order_note('Razorpay renewal failed: Unable to create payment');
                $order->update_status('failed', 'Razorpay: Payment creation failed');
                return false;
            }

        } catch (Exception $e) {
            error_log('Razorpay Auto Charge Exception: ' . $e->getMessage());
            
            if (isset($order)) {
                $order->add_order_note('Razorpay renewal error: ' . $e->getMessage());
                $order->update_status('failed', 'Razorpay: ' . $e->getMessage());
            }
            
            return false;
        }
    }

    /**
     * Convert amount to smallest currency unit (paise for INR, cents for USD, etc.)
     * 
     * @param float $amount Amount in main currency unit
     * @param string $currency Currency code
     * @return int Amount in smallest currency unit
     */
    private function convert_to_paise($amount, $currency) {
        // Most currencies use 100 subunits (cents, paise, etc.)
        // Some exceptions exist but INR uses 100 paise = 1 rupee
        $multiplier = 100;
        
        // Handle zero-decimal currencies (e.g., JPY, KRW)
        $zero_decimal_currencies = ['JPY', 'KRW', 'VND', 'CLP', 'BIF', 'DJF', 'GNF', 'ISK', 'KMF', 'XAF', 'XOF', 'XPF'];
        
        if (in_array(strtoupper($currency), $zero_decimal_currencies)) {
            $multiplier = 1;
        }
        
        return intval(round($amount * $multiplier));
    }

    /**
     * Get Razorpay customer ID for a user
     * 
     * @param int $user_id WordPress user ID
     * @return string|false Customer ID or false if not found
     */
    public function get_razorpay_customer_id($user_id) {
        $customer_id = get_user_meta($user_id, '_razorpay_customer_id', true);
        return !empty($customer_id) ? $customer_id : false;
    }

    /**
     * Get Razorpay token for a user
     * 
     * @param int $user_id WordPress user ID
     * @return string|false Token ID or false if not found
     */
    public function get_razorpay_token($user_id) {
        $token_id = get_user_meta($user_id, '_razorpay_token_id', true);
        return !empty($token_id) ? $token_id : false;
    }

    /**
     * Save Razorpay customer ID for a user
     * 
     * @param int $user_id WordPress user ID
     * @param string $customer_id Razorpay customer ID
     * @return bool True on success, false on failure
     */
    public function save_razorpay_customer_id($user_id, $customer_id) {
        return update_user_meta($user_id, '_razorpay_customer_id', sanitize_text_field($customer_id));
    }

    /**
     * Save Razorpay token for a user
     * 
     * @param int $user_id WordPress user ID
     * @param string $token_id Razorpay token ID
     * @return bool True on success, false on failure
     */
    public function save_razorpay_token($user_id, $token_id) {
        return update_user_meta($user_id, '_razorpay_token_id', sanitize_text_field($token_id));
    }
}
