<?php
/**
 * Register all actions and filters for the plugin.
 *
 * Maintain a list of all hooks that are registered throughout
 * the plugin, and register them with the WordPress API. Call the
 * run function to execute the list of actions and filters.
 *
 * @link       http://www.techspawn.com
 * @since      1.0.0
 * @package    Subscription Products
 * @subpackage Subscription Products/includes
 * @author     techspawn1 <contact@techspawn.com>
 */

class Rorder_Subscription_Products_Loader
{
	/**
	 * The array of actions registered with WordPress.
	 *
	 * @since    1.0.0
	 * @var      array    $actions    The actions registered with WordPress to fire when the plugin loads.
	 */
	protected array $actions = [];

	/**
	 * The array of filters registered with WordPress.
	 *
	 * @since    1.0.0
	 * @var      array    $filters    The filters registered with WordPress to fire when the plugin loads.
	 */
	protected array $filters = [];

	/**
	 * The array of shortcodes registered with WordPress.
	 *
	 * @since    1.0.0
	 * @var      array    $shortcodes    The shortcodes registered with WordPress.
	 */
	protected array $shortcodes = [];

	/**
	 * Add a new action to the collection to be registered with WordPress.
	 *
	 * @since    1.0.0
	 * @param    string   $hook          The name of the WordPress action that is being registered.
	 * @param    object   $component     A reference to the instance of the object on which the action is defined.
	 * @param    string   $callback      The name of the function definition on the $component.
	 * @param    int      $priority      Optional. The priority at which the function should be fired. Default is 10.
	 * @param    int      $accepted_args Optional. The number of arguments that should be passed to the $callback. Default is 1.
	 */
	public function add_action(string $hook, object $component, string $callback, int $priority = 10, int $accepted_args = 1): void
	{
		$this->actions = $this->add($this->actions, $hook, $component, $callback, $priority, $accepted_args);
	}

	/**
	 * Add a new filter to the collection to be registered with WordPress.
	 *
	 * @since    1.0.0
	 * @param    string   $hook          The name of the WordPress filter that is being registered.
	 * @param    object   $component     A reference to the instance of the object on which the filter is defined.
	 * @param    string   $callback      The name of the function definition on the $component.
	 * @param    int      $priority      Optional. The priority at which the function should be fired. Default is 10.
	 * @param    int      $accepted_args Optional. The number of arguments that should be passed to the $callback. Default is 1.
	 */
	public function add_filter(string $hook, object $component, string $callback, int $priority = 10, int $accepted_args = 1): void
	{
		$this->filters = $this->add($this->filters, $hook, $component, $callback, $priority, $accepted_args);
	}

	/**
	 * Add a new shortcode to the collection to be registered with WordPress.
	 *
	 * @since    1.0.0
	 * @param    string   $tag           The name of the shortcode.
	 * @param    object   $component     A reference to the instance of the object on which the shortcode is defined.
	 * @param    string   $callback      The name of the function definition on the $component.
	 */
	public function add_shortcode(string $tag, object $component, string $callback): void
	{
		$this->shortcodes[] = ['tag' => $tag, 'component' => $component, 'callback' => $callback];
	}

	/**
	 * A utility function that is used to register the actions and filters into a single collection.
	 *
	 * @since    1.0.0
	 * @param    array    $hooks         The collection of hooks (actions or filters).
	 * @param    string   $hook          The name of the WordPress hook being registered.
	 * @param    object   $component     A reference to the instance of the object on which the hook is defined.
	 * @param    string   $callback      The name of the function definition on the $component.
	 * @param    int      $priority      The priority at which the function should be fired.
	 * @param    int      $accepted_args The number of arguments that should be passed to the $callback.
	 * @return   array                    The updated collection of hooks.
	 */
	private function add(array $hooks, string $hook, object $component, string $callback, int $priority, int $accepted_args): array
	{
		$hooks[] = [
			'hook'          => $hook,
			'component'     => $component,
			'callback'      => $callback,
			'priority'      => $priority,
			'accepted_args' => $accepted_args,
		];
		return $hooks;
	}

	/**
	 * Register the filters, actions, and shortcodes with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run(): void
	{
		foreach ($this->filters as $hook) {
			add_filter($hook['hook'], [$hook['component'], $hook['callback']], $hook['priority'], $hook['accepted_args']);
		}

		foreach ($this->actions as $hook) {
			add_action($hook['hook'], [$hook['component'], $hook['callback']], $hook['priority'], $hook['accepted_args']);
		}

		foreach ($this->shortcodes as $shortcode) {
			add_shortcode($shortcode['tag'], [$shortcode['component'], $shortcode['callback']]);
		}
	}
}
