<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The core plugin class for Subscription Products.
 * 
 * This class defines internationalization, admin-specific hooks, and
 * public-facing hooks, while maintaining the unique identifier and version.
 * 
 * @link       http://www.techspawn.com
 * @since      1.0.0
 * @package    subscription-products
 * @subpackage subscription-products/includes
 * @author     techspawn1 <contact@techspawn.com>  
 */
class Rorder_Subscription_Products {

    /**
     * Loader for registering all hooks that power the plugin.
     *
     * @since 1.0.0
     * @var Rorder_Subscription_Products_Loader
     */
    protected $loader;

    /**
     * Unique identifier for the plugin.
     *
     * @since 1.0.0
     * @var string $plugin_name
     */
    protected $plugin_name;

    /**
     * Current version of the plugin.
     *
     * @since 1.0.0
     * @var string $version
     */
    protected $version;

    /**
     * Initialize the core functionality of the plugin.
     *
     * @since 1.0.0
     */
    public function __construct() {
        $this->plugin_name = 'reordere-lite-subcriptions-for-woocommerce';
        $this->version = '1.0.0';

        $this->load_dependencies();
        $this->define_admin_hooks();
        $this->define_public_hooks();
        $this->run(); // Ensure run is called after hooks are defined
    }

    /**
     * Load the required dependencies for the plugin.
     *
     * @since 1.0.0
     */
    private function load_dependencies() {
       require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-subscription-products-loader.php';
       require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-subscription-products-admin.php';
        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/partials/class-subscription-products-inventory-fields.php';
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-subscription-order-action-scheduler.php';
        

        // Public files
        require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-subscription-products-public.php';
        require_once plugin_dir_path(dirname(__FILE__)) . 'public/partials/class-subscription-products-public-display.php';
        
        //payment files
        require_once plugin_dir_path(dirname(__FILE__)) . 'payments/ts-payment-controller.php';
		require_once plugin_dir_path(dirname(__FILE__)) . 'payments/woopayments/ts-integrate-stripe-payments.php';
		require_once plugin_dir_path(dirname(__FILE__)) . 'payments/woopayments/ts-integrate-paypal-payments.php';

        // Initialize the loader
        $this->loader = new Rorder_Subscription_Products_Loader();
    }

    /**
     * Register all admin-specific hooks for the plugin.
     *
     * @since 1.0.0
     */
    private function define_admin_hooks() {
        $plugin_admin = new Rorder_Products_Admin($this->get_plugin_name(), $this->get_version());
        $this->loader->add_action('admin_menu', $plugin_admin, 'rorder_susbscription_register_menu_pages');
    }

    /**
     * Register all public-facing hooks for the plugin.
     *
     * @since 1.0.0
     */
    private function define_public_hooks() {
        $plugin_public = new Rorder_Subscription_Products_Public($this->get_plugin_name(), $this->get_version());

        // Enqueue styles and scripts on the frontend
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_styles');
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');

        // Add other actions for the public-facing functionality
        $this->loader->add_action('woocommerce_single_product_summary', $plugin_public, 'subscription_products_public_display');

        // $this->loader->add_action('wp_ajax_process_subscription_action', $plugin_public, 'process_subscription_action');
        // $this->loader->add_action('wp_ajax_nopriv_process_subscription_action', $plugin_public, 'process_subscription_action');
    }

    /**
     * Run the loader to execute all registered hooks.
     *
     * @since 1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * Get the plugin name.
     *
     * @since 1.0.0
     * @return string Plugin name.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * Get the loader instance.
     *
     * @since 1.0.0
     * @return Subscription_Products_Loader The loader instance.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Get the current version of the plugin.
     *
     * @since 1.0.0
     * @return string Plugin version.
     */
    public function get_version() {
        return $this->version;
    }
}

// Initialize the core plugin class
$init_subscription_products = new Rorder_Subscription_Products();

/**
 * Process subscription action via AJAX.
 */
function rorder_process_subscription_action() {
    // Added nonce verification
    if (!isset($_POST['subscription_nonce'])) {
        wp_send_json_error(array('message' => 'Nonce missing.'));
        wp_die();
    }
    $nonce = sanitize_text_field(wp_unslash($_POST['subscription_nonce']));
    if (!wp_verify_nonce($nonce, 'subscription_nonce')) {
        wp_send_json_error(array('message' => 'Invalid nonce.'));
        wp_die();
    }
    
    if (isset($_POST['action_type'], $_POST['order_id'], $_POST['subscription_date'])) {
        $action_type = sanitize_text_field(wp_unslash($_POST['action_type']));
        $order_id = intval(wp_unslash($_POST['order_id']));
        $subscription_date = sanitize_text_field(wp_unslash($_POST['subscription_date']));

        global $wpdb;
        // Sanitize the table name
        $table_name = esc_sql($wpdb->prefix . 'subscription_details');

        // Process Pause
        if ($action_type === 'pause') {
            $wpdb->update(
                $table_name,
                array(
                    'sub_status' => 'paused',
                    'subscription_end_date' => $subscription_date  // Pause until selected date
                ),
                array('order_id' => $order_id),
                array('%s', '%s'),
                array('%d')
            );

            wp_send_json_success(array('message' => 'Subscription paused until ' . esc_html($subscription_date)));
        }

        // Process Skip
        if ($action_type === 'skip') {
            $current_end_date = wp_cache_get('subscription_end_date_' . $order_id);
            if ($current_end_date === false) {
                $current_end_date = $wpdb->get_var($wpdb->prepare(
                    "SELECT subscription_end_date FROM {$table_name} WHERE order_id = %d",
                    $order_id
                ));
                wp_cache_set('subscription_end_date_' . $order_id, $current_end_date);
            }

            if ($current_end_date) {
                // Set the new end date based on user input
                $new_end_date = $subscription_date;

                $wpdb->update(
                    $table_name,
                    array('subscription_end_date' => $new_end_date),
                    array('order_id' => $order_id),
                    array('%s'),
                    array('%d')
                );

                wp_send_json_success(array('message' => 'Subscription extended to ' . esc_html($new_end_date)));
            }
        }

        wp_send_json_error(array('message' => 'Action failed.'));
    }

    wp_die();  // Always include this to properly terminate the AJAX request
}

/**
 * Handle subscription action via AJAX.
 */
function rorder_handle_subscription_action() {
    // Check if user is logged in
    if (!is_user_logged_in()) {
        wp_send_json_error(array('message' => 'You must be logged in to perform this action.'));
        wp_die();
    }

    global $wpdb;
    // Sanitize the table name
    $table_name = esc_sql($wpdb->prefix . 'subscription_details');

    // Verify nonce for security with sanitized input
    if (!isset($_POST['subscription_nonce'])) {
        wp_send_json_error(array('message' => 'Nonce missing.'));
        wp_die();
    }
    $nonce = sanitize_text_field(wp_unslash($_POST['subscription_nonce']));
    if (!wp_verify_nonce($nonce, 'subscription_nonce')) {
        wp_send_json_error(array('message' => 'Invalid nonce.'));
        wp_die();
    }

    // Check if we have the required fields
    if (!isset($_POST['action_type'], $_POST['order_id'], $_POST['subscription_date'])) {
        wp_send_json_error(array('message' => 'Required fields are missing.'));
        wp_die();
    }

    $action_type = sanitize_text_field(wp_unslash($_POST['action_type']));
    $order_id = intval(wp_unslash($_POST['order_id']));
    $subscription_date = sanitize_text_field(wp_unslash($_POST['subscription_date']));
    $order = wc_get_order($order_id);
    if (!$order || intval($order->get_user_id()) !== get_current_user_id()) { wp_send_json_error(array('message' => 'You are not allowed to modify this subscription.')); wp_die(); }

    // Handle Pause Subscription
    if ($action_type === 'pause') {
        $updated = $wpdb->update(
            $table_name,
            array('sub_status' => 'paused', 'subscription_end_date' => $subscription_date),
            array('order_id' => $order_id),
            array('%s', '%s'),
            array('%d')
        );

        if ($updated !== false) {
            wp_send_json_success(array('message' => 'Subscription paused until ' . esc_html($subscription_date), 'updated_date' => esc_html($subscription_date)));
        } else {
            wp_send_json_error(array('message' => 'Failed to pause subscription.'));
        }
    }

    // Handle Skip Subscription
    if ($action_type === 'skip') {
        $current_end_date = wp_cache_get('subscription_end_date_' . $order_id);
        if ($current_end_date === false) {
            $current_end_date = $wpdb->get_var($wpdb->prepare("SELECT subscription_end_date FROM {$table_name} WHERE order_id = %d", $order_id));
            wp_cache_set('subscription_end_date_' . $order_id, $current_end_date);
        }

        if ($current_end_date) {
            // Set the new end date
            $new_end_date = gmdate('Y-m-d', strtotime($subscription_date));

            $updated = $wpdb->update(
                $table_name,
                array('subscription_end_date' => $new_end_date),
                array('order_id' => $order_id),
                array('%s'),
                array('%d')
            );

            if ($updated !== false) {
                wp_send_json_success(array('message' => 'Subscription extended to ' . esc_html($new_end_date), 'updated_date' => esc_html($new_end_date)));
            } else {
                wp_send_json_error(array('message' => 'Failed to skip subscription.'));
            }
        } else {
            wp_send_json_error(array('message' => 'Subscription not found.'));
        }
    }

    wp_send_json_error(array('message' => 'Invalid action.'));
}

// Register AJAX actions for logged-in users
add_action('wp_ajax_handle_subscription_action', 'rorder_handle_subscription_action');
