<?php
/**
 * Insert near end-date subscriptions into cron table.
 */
function rorder_insert_near_enddate_subscriptions_to_cron() {
    global $wpdb;

    // Escape table names
    $subscription_details_table = esc_sql($wpdb->prefix . 'subscription_details');
    $cron_order_details_table   = esc_sql($wpdb->prefix . 'subscription_cron_order_details');

    // Get the current date
    $current_date = gmdate('Y-m-d');

    // Query to fetch active subscriptions whose end date is within the next 7 days
    $query = $wpdb->prepare(
        "SELECT id, user_id, product_id, subscription_end_date, order_id, subscription_price, payment_mode
        FROM `{$subscription_details_table}` 
        WHERE sub_status = 'active'
        AND subscription_end_date BETWEEN %s AND DATE_ADD(%s, INTERVAL 7 DAY)", 
        $current_date, $current_date
    );

    // Use caching for database calls
    $cache_key = 'subscriptions_near_end_' . $current_date;
    $subscriptions_near_end = wp_cache_get($cache_key);

    if ($subscriptions_near_end === false) {
        $subscriptions_near_end = $wpdb->get_results($query);
        wp_cache_set($cache_key, $subscriptions_near_end);
    }

    // Loop through active subscriptions and insert them into the cron_order_details table
    foreach ($subscriptions_near_end as $subscription) {
        // Check if the subscription is already in the cron table with a pending status
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM `{$cron_order_details_table}` WHERE subscription_id = %d AND sub_status = 'pending'",
            $subscription->id
        ));

        // Insert only if it doesn't exist
        if (!$exists) {
            // Insert the subscription into the cron_order_details table
            $insert_result = $wpdb->insert(
                $cron_order_details_table,
                array(
                    'subscription_id' => $subscription->id,
                    'user_id' => $subscription->user_id,
                    'product_id' => $subscription->product_id,
                    'order_id' => $subscription->order_id,
                    'order_amount' => $subscription->subscription_price,
                    'order_renewal_date' => $subscription->subscription_end_date, // Set the renewal date to the end date
                    'order_created_date' => current_time('mysql'),
                    'sub_status' => 'pending',
                    'payment_mode' => $subscription->payment_mode,
                )
            );

            if ($insert_result === false) {
                // Log if insertion fails
                error_log('Failed to insert subscription ID: ' . esc_html($subscription->id) . ' into cron_order_details. SQL Error: ' . esc_html($wpdb->last_error));
            } else {
                error_log('Subscription ID ' . esc_html($subscription->id) . ' successfully inserted into cron_order_details.');
            }
        }
    }
}

// Hook to run the function manually or on an interval
add_action('init', 'rorder_insert_near_enddate_subscriptions_to_cron');

/**
 * Schedule actions based on the cron table.
 */
function rorder_schedule_action_based_on_cron_table() {
    global $wpdb;

    // Escape table names
    $cron_order_details_table = esc_sql($wpdb->prefix . 'subscription_cron_order_details');

    // Get the current date
    $current_date = gmdate('Y-m-d H:i:s');

    // Query to fetch cron entries with 'pending' status and upcoming order_renewal_date
    $query = $wpdb->prepare(
        "SELECT subscription_id, user_id, product_id, order_renewal_date, order_id 
        FROM `{$cron_order_details_table}` 
        WHERE sub_status = 'pending'
        AND order_renewal_date <= %s", 
        $current_date
    );

    // Use caching for database calls
    $cache_key = 'cron_entries_' . $current_date;
    $cron_entries = wp_cache_get($cache_key);

    if ($cron_entries === false) {
        $cron_entries = $wpdb->get_results($query);
        wp_cache_set($cache_key, $cron_entries);
    }

    if (empty($cron_entries)) {
        // Log that no pending cron entries exist for scheduling
        return;
    }

    // Loop through each pending cron entry and schedule actions
    foreach ($cron_entries as $cron_entry) {
        $renewal_timestamp = strtotime($cron_entry->order_renewal_date);

        // Check if Action is already scheduled
        $is_scheduled = as_next_scheduled_action('process_subscription_renewal', array($cron_entry->subscription_id));

        if (!$is_scheduled) {
            // Schedule the action for renewal on the specified date
            as_schedule_single_action($renewal_timestamp, 'process_subscription_renewal', array($cron_entry->subscription_id));
            error_log('Scheduled action for subscription ID: ' . esc_html($cron_entry->subscription_id) . ' to run on ' . esc_html($cron_entry->order_renewal_date) . '.');
        } else {
            error_log('Action already scheduled for subscription ID: ' . esc_html($cron_entry->subscription_id));
        }
    }
}

// Hook to run the function manually
add_action('init', 'rorder_schedule_action_based_on_cron_table');

// Action Scheduler callback function
add_action('process_subscription_renewal', 'rorder_process_subscription_renewal', 10, 1);

/**
 * Process subscription renewal.
 *
 * @param int $subscription_id The subscription ID.
 */
function rorder_process_subscription_renewal($subscription_id) {
    global $wpdb;

    error_log('Processing subscription renewal for subscription ID: ' . esc_html($subscription_id));

    // Escape table names
    $cron_order_details_table = esc_sql($wpdb->prefix . 'subscription_cron_order_details');
    $subscription_details_table = esc_sql($wpdb->prefix . 'subscription_details');
    // Fetch the cron entry with 'pending' status
    $cron_entry = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM `{$cron_order_details_table}` WHERE subscription_id = %d AND sub_status = 'pending'",
        $subscription_id
    ));
    $subscription_entry = $wpdb->get_row($wpdb->prepare(
        "SELECT order_id FROM `{$subscription_details_table}` WHERE id = %d",
        $subscription_id
    ));
    if (!$cron_entry) {
        error_log('No matching pending cron entry found for subscription ID: ' . esc_html($subscription_id));
        return;
    }

    // Fetch the product and user details from the cron table
    $user_id = $cron_entry->user_id;
    $product_id = $cron_entry->product_id;
    $product = wc_get_product($product_id);

    if (!$product) {
        error_log('Product not found for Product ID: ' . esc_html($product_id));
        return;
    }

    // Get the current product price
    $updated_price = $product->get_price();
    error_log('Updated Product Price for Product ID ' . esc_html($product_id) . ': ' . esc_html($updated_price));

    // Fetch subscription details from the original order
    $original_order_id = $cron_entry->order_id;
    $original_order = wc_get_order($original_order_id);

    if (!$original_order) {
        error_log('Original order not found for order ID: ' . esc_html($original_order_id));
        return;
    }

    // Get subscription details (period, interval, type) from the original order items
    $subscription_period = '';
    $subscription_duration = '';
    $subscription_type = '';

    foreach ($original_order->get_items() as $item_id => $item) {
        if ($item->get_product_id() == $product_id) {
            $subscription_period = wc_get_order_item_meta($item_id, '_subscription_period', true);
            $subscription_duration = wc_get_order_item_meta($item_id, '_subscription_duration', true);
            $subscription_type = wc_get_order_item_meta($item_id, '_subscription_type', true);
            break;
        }
    }

    // Create the new renewal order (this creates a new WooCommerce order but does not update the subscription details)
    $new_order_id = rorder_create_subscription_renewal_order($user_id, $product_id, $updated_price, $original_order_id, $subscription_id, $subscription_period, $subscription_duration, $subscription_type);
    $order_ids = array();
    $order_ids[] = $subscription_entry->order_id;
    $order_ids[] = $new_order_id;
    error_log('Order IDs: ' . esc_html(implode(', ', $order_ids)));

    if (!is_wp_error($new_order_id)) {
        error_log('New order created successfully for subscription ID: ' . esc_html($subscription_id) . '. Order ID: ' . esc_html($new_order_id));

        // Mark the cron table entry as 'processed'
        $wpdb->update(
            $cron_order_details_table,
            array('order_id' => $new_order_id, 'order_created_date' => current_time('mysql'), 'sub_status' => 'processed'),
            array('subscription_id' => $subscription_id)
        );
        
        // Convert order IDs array to a comma-separated string
        $order_ids_string = implode(',', $order_ids);

        // Push the order_id to the subscription_details table as a comma-separated string
        $wpdb->update(
            $subscription_details_table,
            array('order_id' => $order_ids_string),  // Update with comma-separated string
            array('id' => $subscription_id)
        );

    } else {
        error_log('Failed to create new order for subscription ID ' . esc_html($subscription_id) . '. Error: ' . esc_html($new_order_id->get_error_message()));
    }

    // Log the completion of the action
    error_log('Completed processing subscription renewal for ID: ' . esc_html($subscription_id));
}

/**
 * Create a subscription renewal order with updated price and subscription details.
 *
 * @param int $user_id The user ID.
 * @param int $product_id The product ID.
 * @param float $updated_price The updated price.
 * @param int $original_order_id The original order ID.
 * @param int $subscription_id The subscription ID.
 * @param string $subscription_period The subscription period.
 * @param string $subscription_duration The subscription duration.
 * @param string $subscription_type The subscription type.
 * @return int|WP_Error The new order ID or WP_Error on failure.
 */
function rorder_create_subscription_renewal_order($user_id, $product_id, $updated_price, $original_order_id, $subscription_id, $subscription_period, $subscription_duration, $subscription_type) {
    if (!class_exists('WC_Order')) {
        return new WP_Error('woocommerce_missing', 'WooCommerce is not installed or activated.');
    }

    error_log('Creating new order for User ID: ' . esc_html($user_id) . ', Product ID: ' . esc_html($product_id));

    // Create a new order for the renewal
    $order = wc_create_order();
    $order->set_customer_id($user_id);
    $product = wc_get_product($product_id);
    if (!$product) {
        return new WP_Error('invalid_product', 'Invalid product ID.');
    }

    // Add the product to the order
    $item_id = $order->add_product($product, 1, array(
        'subtotal' => $updated_price,
        'total' => $updated_price,
    ));

    // Add subscription-related metadata (period, interval, price, type)
    wc_add_order_item_meta($item_id, '_subscription_period', $subscription_period);
    wc_add_order_item_meta($item_id, '_subscription_duration', $subscription_duration);
    wc_add_order_item_meta($item_id, '_subscription_price', $updated_price);
    wc_add_order_item_meta($item_id, '_subscription_id', $subscription_id);
    wc_add_order_item_meta($item_id, '_subscription_type', $subscription_type);

    $order->calculate_totals();
    $order->add_order_note('Order created for subscription renewal.');

    $payment_method = rorder_get_subscription_renewal_payment_method($subscription_id);
		
	$payment_obj = new ts_payments_controller();
		
	$charge_customer = $payment_obj->ts_charge_customers($payment_method, $order->get_id());
 
    $order->save();

    return $order->get_id();
}

/**
 * Get the payment method for subscription renewal.
 *
 * @param int $subscription_id The subscription ID.
 * @return string The payment method.
 */
function rorder_get_subscription_renewal_payment_method($subscription_id){
    global $wpdb;
    $subscription_table = esc_sql($wpdb->prefix . 'subscription_details');
    $SQL = $wpdb->prepare("SELECT payment_mode FROM `{$subscription_table}` WHERE id = %d", $subscription_id);
    $results = $wpdb->get_results($SQL);

    foreach($results as $result){
        $payment_method = $result->payment_mode;
    }

    return $payment_method;
}
