<?php
/**
 * Database Migration: Create Subscription Meta Table
 * 
 * Creates reordere_subscriptions_meta table to store additional subscription metadata
 * including Razorpay subscription IDs, customer IDs, and other provider-specific data
 * 
 * @package ReordereLite
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Create reordere_subscriptions_meta table
 */
function reordere_create_subscriptions_meta_table() {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'reordere_subscriptions_meta';
    $charset_collate = $wpdb->get_charset_collate();
    
    // Check if table already exists
    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
        
        $sql = "CREATE TABLE $table_name (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            subscription_id bigint(20) UNSIGNED NOT NULL,
            meta_key varchar(255) NOT NULL,
            meta_value longtext NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY subscription_id (subscription_id),
            KEY meta_key (meta_key),
            UNIQUE KEY subscription_meta_key (subscription_id, meta_key)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        error_log('ReorderRe: Created reordere_subscriptions_meta table');
    } else {
        error_log('ReorderRe: reordere_subscriptions_meta table already exists');
    }
}

/**
 * Get subscription meta value
 * 
 * @param int $subscription_id Subscription ID
 * @param string $meta_key Meta key
 * @param mixed $default Default value if not found
 * @return mixed Meta value or default
 */
function reordere_get_subscription_meta($subscription_id, $meta_key, $default = null) {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'reordere_subscriptions_meta';
    
    $value = $wpdb->get_var($wpdb->prepare(
        "SELECT meta_value FROM $table_name WHERE subscription_id = %d AND meta_key = %s",
        $subscription_id,
        $meta_key
    ));
    
    return $value !== null ? $value : $default;
}

/**
 * Update subscription meta value
 * 
 * @param int $subscription_id Subscription ID
 * @param string $meta_key Meta key
 * @param mixed $meta_value Meta value
 * @return bool Success
 */
function reordere_update_subscription_meta($subscription_id, $meta_key, $meta_value) {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'reordere_subscriptions_meta';
    
    // Check if meta already exists
    $existing = $wpdb->get_var($wpdb->prepare(
        "SELECT id FROM $table_name WHERE subscription_id = %d AND meta_key = %s",
        $subscription_id,
        $meta_key
    ));
    
    if ($existing) {
        // Update existing meta
        $result = $wpdb->update(
            $table_name,
            array('meta_value' => $meta_value),
            array('subscription_id' => $subscription_id, 'meta_key' => $meta_key),
            array('%s'),
            array('%d', '%s')
        );
    } else {
        // Insert new meta
        $result = $wpdb->insert(
            $table_name,
            array(
                'subscription_id' => $subscription_id,
                'meta_key' => $meta_key,
                'meta_value' => $meta_value
            ),
            array('%d', '%s', '%s')
        );
    }
    
    return $result !== false;
}

/**
 * Delete subscription meta
 * 
 * @param int $subscription_id Subscription ID
 * @param string $meta_key Meta key (optional, deletes all if not provided)
 * @return bool Success
 */
function reordere_delete_subscription_meta($subscription_id, $meta_key = null) {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'reordere_subscriptions_meta';
    
    if ($meta_key) {
        $result = $wpdb->delete(
            $table_name,
            array('subscription_id' => $subscription_id, 'meta_key' => $meta_key),
            array('%d', '%s')
        );
    } else {
        $result = $wpdb->delete(
            $table_name,
            array('subscription_id' => $subscription_id),
            array('%d')
        );
    }
    
    return $result !== false;
}

/**
 * Get subscription ID by Razorpay subscription ID
 * 
 * @param string $razorpay_subscription_id Razorpay subscription ID
 * @return int|null Subscription ID or null
 */
function reordere_get_subscription_by_razorpay_id($razorpay_subscription_id) {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'reordere_subscriptions_meta';
    
    $subscription_id = $wpdb->get_var($wpdb->prepare(
        "SELECT subscription_id FROM $table_name WHERE meta_key = 'razorpay_subscription_id' AND meta_value = %s",
        $razorpay_subscription_id
    ));
    
    return $subscription_id ? (int)$subscription_id : null;
}

// Run migration on plugin activation
register_activation_hook(dirname(dirname(__FILE__)) . '/reordere-lite-subcriptions-for-woocommerce.php', 'reordere_create_subscriptions_meta_table');

// Also run on admin_init to handle updates
add_action('admin_init', function() {
    $migration_version = get_option('reordere_subscriptions_meta_version', '0');
    
    if (version_compare($migration_version, '1.0.0', '<')) {
        reordere_create_subscriptions_meta_table();
        update_option('reordere_subscriptions_meta_version', '1.0.0');
    }
});
