<?php
/**
 * Subscription Product Reports
 *
 * @package TS_Subscription
 */

/**
 * Get subscription report data.
 *
 * @return array The subscription report data including status counts.
 */
function rorder_get_subscription_report_data() {
    global $wpdb;

    // Check if data is cached
    $cache_key = 'subscription_report_data';
    $data = wp_cache_get($cache_key);

    if ($data === false) {
        $results = $wpdb->get_results("
            SELECT sub_status, COUNT(*) as count
            FROM {$wpdb->prefix}subscription_details
            GROUP BY sub_status
        ", ARRAY_A);

        $data = [
            'active' => 0,
            'paused' => 0,
            'cancelled' => 0,
            'expired' => 0,
            'inactive' => 0,
        ];

        if ($results && is_array($results)) {
            foreach ($results as $row) {
                $status = strtolower(trim($row['sub_status']));
                if (isset($data[$status])) {
                    $data[$status] = (int) $row['count'];
                } else {
                    // Map any unknown status to 'expired' for display
                    $data['expired'] += (int) $row['count'];
                }
            }
        }

        // Set cache
        wp_cache_set($cache_key, $data);
    }

    return $data;
}

function rorder_get_columns() {
    $columns = [
        'id' => __( 'Subscription ID', 'reordere-lite-subcriptions-for-woocommerce' ),
        'sub_status' => __( 'Status', 'reordere-lite-subcriptions-for-woocommerce' ),
        'subscription_start_date' => __( 'Start Date', 'reordere-lite-subcriptions-for-woocommerce' ),
        'subscription_end_date' => __( 'End Date', 'reordere-lite-subcriptions-for-woocommerce' ),
        'product_name' => __( 'Product(s)', 'reordere-lite-subcriptions-for-woocommerce' ),
        'total_price' => __( 'Total Price', 'reordere-lite-subcriptions-for-woocommerce' ),
        'customer_name' => __( 'Customer', 'reordere-lite-subcriptions-for-woocommerce' )
    ];
    return $columns;
}

/**
 * Get subscription metrics for dashboard widgets.
 * 
 * @return array The subscription metrics.
 */
function rorder_get_subscription_metrics() {
    global $wpdb;
    
    // Get total subscriptions
    $total = $wpdb->get_var("
        SELECT COUNT(*) 
        FROM {$wpdb->prefix}subscription_details
    ");
    
    // Get active subscriptions
    $active = $wpdb->get_var("
        SELECT COUNT(*) 
        FROM {$wpdb->prefix}subscription_details
        WHERE sub_status = 'active'
    ");
    
    // Get inactive subscriptions
    $inactive = $wpdb->get_var("
        SELECT COUNT(*) 
        FROM {$wpdb->prefix}subscription_details
        WHERE sub_status = 'inactive'
    ");
    
    // Get paused subscriptions
    $paused = $wpdb->get_var("
        SELECT COUNT(*) 
        FROM {$wpdb->prefix}subscription_details
        WHERE sub_status = 'paused'
    ");
    
    // Get cancelled subscriptions
    $cancelled = $wpdb->get_var("
        SELECT COUNT(*) 
        FROM {$wpdb->prefix}subscription_details
        WHERE sub_status = 'cancelled'
    ");
    
    // Get total revenue from active subscriptions
    $revenue = $wpdb->get_var("
        SELECT SUM(subscription_price) 
        FROM {$wpdb->prefix}subscription_details
        WHERE sub_status = 'active'
    ");
    
    return [
        'total' => (int) $total,
        'active' => (int) $active,
        'inactive' => (int) $inactive,
        'paused' => (int) $paused,
        'cancelled' => (int) $cancelled,
        'revenue' => (float) $revenue,
    ];
}

/**
 * Get payment methods distribution.
 * 
 * @return array The payment methods distribution.
 */
function rorder_get_payment_methods_distribution() {
    global $wpdb;
    
    $results = $wpdb->get_results("
        SELECT payment_mode, COUNT(*) as count
        FROM {$wpdb->prefix}subscription_details
        WHERE payment_mode IS NOT NULL AND payment_mode != ''
        GROUP BY payment_mode
    ", ARRAY_A);
    
    $data = [];
    
    if ($results && is_array($results)) {
        foreach ($results as $row) {
            $payment_method = !empty($row['payment_mode']) ? ucfirst($row['payment_mode']) : 'Unknown';
            $data[$payment_method] = (int) $row['count'];
        }
    }
    
    // If no data, return empty array
    if (empty($data)) {
        $data = ['No Data' => 0];
    }
    
    return $data;
}

/**
 * Get top subscription products.
 * 
 * @param int $limit Number of products to retrieve
 * @return array The top subscription products.
 */
function rorder_get_top_subscription_products($limit = 5) {
    global $wpdb;
    
    // Join with wp_posts to get product names
    $results = $wpdb->get_results($wpdb->prepare("
        SELECT 
            p.post_title as product_name,
            COUNT(*) as subscription_count,
            SUM(CAST(sd.subscription_price AS DECIMAL(10,2))) as total_revenue
        FROM {$wpdb->prefix}subscription_details sd
        LEFT JOIN {$wpdb->prefix}posts p ON sd.product_id = p.ID
        WHERE p.post_title IS NOT NULL AND p.post_title != ''
        GROUP BY sd.product_id, p.post_title
        ORDER BY subscription_count DESC
        LIMIT %d
    ", $limit), ARRAY_A);
    
    // Return empty array if no results
    if (!$results || !is_array($results)) {
        return [];
    }
    
    return $results;
}

/**
 * Get recent subscription activity.
 * 
 * @param int $limit Number of activities to retrieve
 * @return array The recent subscription activity.
 */
function rorder_get_recent_subscription_activity($limit = 10) {
    global $wpdb;
    
    $results = $wpdb->get_results($wpdb->prepare("
        SELECT id, subscription_title, sub_status, subscription_start_date, user_id
        FROM {$wpdb->prefix}subscription_details
        ORDER BY created_at DESC
        LIMIT %d
    ", $limit));
    
    return $results;
}
