<?php
/**
 * Subscription List Table
 *
 * @package TS_Subscription
 */

if ( ! class_exists( 'WP_List_Table' ) ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class Rorder_Subscription_List_Table extends WP_List_Table {

    /**
     * Get a list of CSS classes for the table tag.
     *
     * @return array
     */
    protected function get_table_classes() {
        return array( 'widefat', 'fixed', 'striped', 'subscription-table-modern', $this->_args['plural'] );
    }

    /**
     * Define the columns for the table.
     *
     * @return array
     */
    public function get_columns() {
        $columns = [
            'id' => __( 'Subscription ID', 'reordere-lite-subcriptions-for-woocommerce' ),
            'sub_status' => __( 'Status', 'reordere-lite-subcriptions-for-woocommerce' ),
            'subscription_start_date' => __( 'Start Date', 'reordere-lite-subcriptions-for-woocommerce' ),
            'subscription_end_date' => __( 'End Date', 'reordere-lite-subcriptions-for-woocommerce' ),
            'product_name' => __( 'Product(s)', 'reordere-lite-subcriptions-for-woocommerce' ),
            'total_price' => __( 'Total Price', 'reordere-lite-subcriptions-for-woocommerce' ),
            'customer_name' => __( 'Customer', 'reordere-lite-subcriptions-for-woocommerce' )
        ];
        return $columns;
    }

    /**
     * Display product names as comma-separated and link to product edit page.
     *
     * @param int $item_id The subscription item ID.
     * @return string
     */
    public function product_name( $item_id ) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'subscription_details';
        // Query to fetch product ID - table names cannot be parameterized
        $query = $wpdb->prepare( "SELECT product_id FROM {$table_name} WHERE id = %d", $item_id );
        $item = $wpdb->get_row( $query, ARRAY_A );
        if ( ! $item ) {
            return esc_html( 'Unknown Product' );
        }
        $products = explode( ',', $item['product_id'] );
        foreach ( $products as $product_id ) {
            $edit_product_url = get_edit_post_link( $product_id );
            $product_names = get_the_title( $product_id );
            $linked_product_names[] = '<a href="' . esc_url( $edit_product_url ) . '">' . esc_html( $product_names ) . '</a>';
        }
        return $linked_product_names ? implode( ', ', $linked_product_names ) : esc_html( 'Unknown Product' );
    }

    /**
     * Display customer name.
     *
     * @param array $item The subscription item.
     * @return string
     */
    public function column_customer_name( $item ) {
        $user_name = isset( $item['customer_name'] ) ? $item['customer_name'] : 'Unknown User';
        return esc_html( $user_name );
    }

    /**
     * Make subscription ID clickable for details.
     *
     * @param array $item The subscription item.
     * @return string
     */
    public function column_id( $item ) {
        $subscription_id = esc_html( $item['id'] );
        $details_url = add_query_arg( [
            'page' => 'subscriptions',
            'action' => 'edit',
            'subscription_id' => $subscription_id
        ], admin_url( 'admin.php' ) );

        return '<a href="' . esc_url( $details_url ) . '">#' . $subscription_id . '</a>';
    }

    /**
     * Display total price.
     *
     * @param array $item The subscription item.
     * @return string
     */
    public function column_total_price( $item ) {
        return isset( $item['total_price'] ) ? wc_price( $item['total_price'] ) : wc_price( 0 );
    }

    /**
     * Display subscription status with badge.
     *
     * @param array $item The subscription item.
     * @return string
     */
    public function column_sub_status( $item ) {
        $status = isset( $item['sub_status'] ) ? $item['sub_status'] : 'N/A';
        $status_lower = strtolower( $status );
        $status_class = 'status-' . str_replace( ' ', '-', $status_lower );
        
        return '<span class="subscription-status-badge ' . esc_attr( $status_class ) . '">' . 
               esc_html( ucwords( $status ) ) . 
               '</span>';
    }

    /**
     * Default column handling.
     *
     * @param array  $item The subscription item.
     * @param string $column_name The column name.
     * @return string
     */
    public function column_default( $item, $column_name ) {
        switch ( $column_name ) {
            case 'id':
                return '#' . esc_html( $item['id'] );
            case 'sub_status':
                return $this->column_sub_status( $item );
            case 'subscription_start_date':
                return isset( $item['subscription_start_date'] ) ? esc_html( $item['subscription_start_date'] ) : esc_html( 'N/A' );
            case 'subscription_end_date':
                return isset( $item['subscription_end_date'] ) ? esc_html( $item['subscription_end_date'] ) : esc_html( 'N/A' );
            case 'product_name':
                return $this->product_name( $item['id'] );
            case 'customer_name':
            case 'total_price':
                return isset( $item[ $column_name ] ) ? esc_html( $item[ $column_name ] ) : esc_html( 'N/A' );
            default:
                return ''; // Fallback for any unexpected column.
        }
    }

    /**
     * Make columns sortable (optional).
     *
     * @return array
     */
    public function get_sortable_columns() {
        $sortable_columns = [
            'product_name' => [ 'product_name', true ],
            'total_price' => [ 'total_price', true ],
            'subscription_start_date' => [ 'subscription_start_date', true ]
        ];
        return $sortable_columns;
    }

    /**
     * Prepare the data and columns for the table.
     */
    public function prepare_items() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'subscription_details'; // Custom table.

        // Set pagination parameters.
        $per_page = 10;
        $current_page = $this->get_pagenum();

        // Count total items - table names cannot be parameterized
        $total_items = $wpdb->get_var( "SELECT COUNT(DISTINCT order_id) FROM {$table_name}" );

        // Fetch data from the database, grouping products by order ID.
        $offset = ( $current_page - 1 ) * $per_page;
        // Main query - table names cannot be parameterized
        $query = $wpdb->prepare("
            SELECT 
                sd.id, 
                sd.sub_status,
                sd.subscription_start_date, 
                sd.subscription_end_date,
                GROUP_CONCAT(p.post_title ORDER BY p.post_title ASC) AS product_name, 
                GROUP_CONCAT(sd.product_id) AS product_ids,
                SUM(sd.subscription_price) AS total_price, 
                u.display_name AS customer_name 
            FROM {$table_name} sd
            LEFT JOIN {$wpdb->prefix}posts p ON sd.product_id = p.ID
            LEFT JOIN {$wpdb->prefix}users u ON sd.user_id = u.ID
            GROUP BY sd.order_id
            LIMIT %d, %d
        ", $offset, $per_page );
        $data = $wpdb->get_results( $query, ARRAY_A );

        // Define the column headers.
        $this->items = $data;

        // Set column headers and pagination.
        $this->set_pagination_args( [
            'total_items' => $total_items,
            'per_page' => $per_page,
        ] );

        $this->_column_headers = [
            $this->get_columns(), // Column headers.
            [], // Hidden columns.
            $this->get_sortable_columns()  // Sortable columns.
        ];
    }
}

