<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Show Single Subscription Data
 *
 * @package TS_Subscription
 */

// Ensure the user is allowed to view this page.
if ( ! current_user_can( 'manage_options' ) ) {
    wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'reordere-lite-subcriptions-for-woocommerce' ) );
}

// Get the subscription ID from the query parameter.
$subscription_id = isset( $_GET['subscription_id'] ) ? absint( $_GET['subscription_id'] ) : 0;

if ( $subscription_id <= 0 ) {
    wp_die( esc_html__( 'Invalid Subscription ID.', 'reordere-lite-subcriptions-for-woocommerce' ) );
}

// Fetch subscription details from the database.
global $wpdb;
$table_name   = $wpdb->prefix . 'subscription_details';
$subscription = wp_cache_get( 'subscription_' . $subscription_id, 'subscriptions' );

if ( false === $subscription ) {
    $subscription = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table_name} WHERE id = %d", $subscription_id ), ARRAY_A );
    wp_cache_set( 'subscription_' . $subscription_id, $subscription, 'subscriptions' );
}

if ( ! $subscription ) {
    wp_die( esc_html__( 'Subscription not found.', 'reordere-lite-subcriptions-for-woocommerce' ) );
}

$order_id  = $subscription['order_id'];
$order_ids = explode( ',', $order_id );
$order_id  = $order_ids[0];

$order = wc_get_order( $order_id );

if ( $order ) {
    // Get order status.
    $order_status = $order->get_status(); // e.g., 'completed', 'processing', etc.

    // Get order date.
    $order_date = $order->get_date_created(); // Returns a DateTime object.

    // Format the order date.
    $formatted_date = $order_date ? $order_date->date( 'F j, Y' ) : esc_html__( 'Unknown date', 'reordere-lite-subcriptions-for-woocommerce' );

} else {
    // Show error message as a WP notice.
    ?>
    <div class="notice notice-error is-dismissible">
        <p><?php esc_html_e( 'Order not found.', 'reordere-lite-subcriptions-for-woocommerce' ); ?></p>
    </div>
    <?php
    return;
}

// Handle form submission to update subscription settings.
if ( isset( $_POST['update_subscription'] ) ) {
    // Verify nonce.
    $nonce = isset( $_POST['subscription_settings_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['subscription_settings_nonce'] ) ) : '';
    if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'update_subscription_settings' ) ) {
        wp_die( esc_html__( 'Nonce verification failed.', 'reordere-lite-subcriptions-for-woocommerce' ) );
    }

    // Sanitize and update the subscription data.
    $status     = isset( $_POST['status'] ) ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : '';
    $start_date = isset( $_POST['start_date'] ) ? sanitize_text_field( wp_unslash( $_POST['start_date'] ) ) : '';
    $end_date   = isset( $_POST['end_date'] ) ? sanitize_text_field( wp_unslash( $_POST['end_date'] ) ) : '';

    $wpdb->update(
        $table_name,
        [
            'status'                  => $status,
            'subscription_start_date' => $start_date,
            'subscription_end_date'   => $end_date,
        ],
        [ 'id' => $subscription_id ],
        [ '%s', '%s', '%s' ],
        [ '%d' ]
    );

    // Reload the page to see updated details.
    wp_safe_redirect( add_query_arg( 'page', 'subscription-details', admin_url( 'admin.php' ) ) . '&subscription_id=' . $subscription_id );
    exit;
}
?>

<div class="wrap">
    <h1 class="wp-heading-inline"><?php esc_html_e( 'Subscription', 'reordere-lite-subcriptions-for-woocommerce' ); ?></h1>

    <div class="subscription-container">
        <!-- Subscription Details -->
        <div class="subscription-details">
            <h2><?php esc_html_e( 'Subscription Details', 'reordere-lite-subcriptions-for-woocommerce' ); ?></h2>
            <!-- Subscription ID and Status -->
            <h1 style="margin-bottom: 2%;">
                <?php esc_html_e( 'Subscription ID:', 'reordere-lite-subcriptions-for-woocommerce' ); ?> #<?php echo esc_html( $subscription_id ); ?>
                <?php
                $status = strtolower( $subscription['sub_status'] );
                $status_class = 'status-' . $status;
                $status_bg = '#d1fae5'; // Default active
                $status_color = '#065f46';
                
                if ( $status === 'inactive' ) {
                    $status_bg = '#fee2e2';
                    $status_color = '#991b1b';
                } elseif ( $status === 'cancelled' ) {
                    $status_bg = '#fef3c7';
                    $status_color = '#92400e';
                } elseif ( $status === 'paused' ) {
                    $status_bg = '#e1f5fe';
                    $status_color = '#01579b';
                }
                ?>
                <span style="font-size: 16px; background-color: <?php echo esc_attr( $status_bg ); ?>; color: <?php echo esc_attr( $status_color ); ?>; padding: 8px 16px; border-radius: 5px; font-weight: 600;">
                    <?php echo esc_html( ucwords( $subscription['sub_status'] ) ); ?>
                </span>
            </h1>
            <table class="form-table">
                <tr>
                    <th><?php esc_html_e( 'Subscription ID:', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                    <td>#<?php echo esc_html( $subscription['id'] ); ?></td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Start Date:', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                    <td><?php echo esc_html( gmdate( 'F j, Y', strtotime( $subscription['subscription_start_date'] ) ) ); ?></td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'End Date:', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                    <td><?php echo esc_html( gmdate( 'F j, Y', strtotime( $subscription['subscription_end_date'] ) ) ); ?></td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Product Purchased:', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                    <td>
                        <?php 
                        $product_ids = $subscription['product_id']; // Assuming $subscription['product_id'] contains a comma-separated string of product IDs.

                        if ( ! empty( $product_ids ) ) {
                            // Split the product IDs into an array.
                            $product_ids_array = explode( ',', $product_ids );

                            // Initialize an array to store product names with links.
                            $product_names_with_links = array();

                            foreach ( $product_ids_array as $product_id ) {
                                // Trim any extra spaces.
                                $product_id = absint( trim( $product_id ) );

                                // Get product by product ID.
                                $product = wc_get_product( $product_id );

                                if ( $product ) {
                                    // Get the product name and URL.
                                    $product_name = $product->get_name();
                                    $product_url  = get_permalink( $product_id );

                                    // Store the product name with link in the array.
                                    $product_names_with_links[] = sprintf(
                                        '<a href="%s" target="_blank" style="text-decoration: none;">%s</a>',
                                        esc_url( $product_url ),
                                        esc_html( $product_name )
                                    );
                                } else {
                                    // Handle unknown product cases.
                                    $product_names_with_links[] = esc_html__( 'Unknown Product (ID: ', 'reordere-lite-subcriptions-for-woocommerce' ) . esc_html( $product_id ) . ')';
                                }
                            }

                            // Output the product names with links separated by commas.
                            echo wp_kses_post( implode( ', ', $product_names_with_links ) );
                        } else {
                            echo esc_html__( 'No Product ID found', 'reordere-lite-subcriptions-for-woocommerce' );
                        }
                        ?>
                    </td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Customer:', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                    <td>
                        <?php 
                        $user_id   = absint( $subscription['user_id'] );
                        $user_info = get_userdata( $user_id );

                        if ( $user_info ) {
                            $customer_name = $user_info->display_name;
                            $customer_url  = get_edit_user_link( $user_id );
                            echo sprintf(
                                '<a href="%s" target="_blank" style="text-decoration: none;">%s</a>',
                                esc_url( $customer_url ),
                                esc_html( $customer_name )
                            );
                        } else {
                            echo esc_html__( 'Unknown User', 'reordere-lite-subcriptions-for-woocommerce' );
                        }
                        ?>
                    </td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Product Price:', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                    <td>
                        <?php 
                        $order = wc_get_order( $order_id ); // Assuming you have $order_id available.
                        if ( empty( $order ) ) {
                            return;
                        }
                        $items = $order->get_items();
                        foreach ( $items as $item ) {
                            $product_name     = $item->get_name(); // Get the product name.
                            $product_quantity = intval( $item->get_quantity() ); // Get the quantity purchased.
                            $product_price    = floatval( $item->get_total() ) / $product_quantity; // Get the total price divided by the quantity to get the unit price.
                            echo esc_html( $product_name ) . ' - ' . wp_kses_post( wc_price( $product_price ) ) . '<br>';
                        }
                        ?>
                    </td>
                </tr>
                <tr>
                    <?php 
                    // Step 1: Get the order details.
                    $order = wc_get_order( $order_id ); // Assuming you have $order_id available.

                    // Step 2: Get the payment method title (e.g., 'Cash on Delivery', 'Credit Card', etc.).
                    $payment_method_title = $order->get_payment_method_title(); // This will return the human-readable payment method title.
                    ?>

                    <th><?php esc_html_e( 'Payment Method:', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                    <td><?php echo esc_html( $payment_method_title ); ?></td>
                </tr>
            </table>
        </div>
        <!-- Subscription Settings -->
        <div class="subscription-settings">
            <h2><?php esc_html_e( 'Subscription Settings', 'reordere-lite-subcriptions-for-woocommerce' ); ?></h2>
            <form method="post" action="">
                <?php wp_nonce_field( 'update_subscription_settings', 'subscription_settings_nonce' ); ?>
                
                <table class="form-table">
                    <tr>
                        <th><?php esc_html_e( 'Status:', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                        <td>
                            <select name="status">
                                <option value="active" <?php selected( $subscription['sub_status'], 'active' ); ?>><?php esc_html_e( 'Active', 'reordere-lite-subcriptions-for-woocommerce' ); ?></option>
                                <option value="inactive" <?php selected( $subscription['sub_status'], 'inactive' ); ?>><?php esc_html_e( 'Inactive', 'reordere-lite-subcriptions-for-woocommerce' ); ?></option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Start Date:', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                        <td>
                            <input type="date" name="start_date" value="<?php echo esc_attr( $subscription['subscription_start_date'] ); ?>" readonly />
                        </td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'End Date:', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                        <td>
                            <input type="date" name="end_date" value="<?php echo esc_attr( $subscription['subscription_end_date'] ); ?>" readonly />
                        </td>
                    </tr>
                </table>
                <input type="hidden" name="update_subscription" value="1" />
                <input type="submit" class="button button-primary update_subs_btn" value="<?php esc_html_e( 'Update', 'reordere-lite-subcriptions-for-woocommerce' ); ?>" />
            </form>
        </div>
</div>
        <!-- Subscription Order History -->
        <div class="subscription-order-history" style="width: 75%; background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 0 10px rgba(0,0,0,0.1); margin-top:3%;">
    <h2><?php esc_html_e( 'Subscription Order History', 'reordere-lite-subcriptions-for-woocommerce' ); ?></h2>
    <table class="form-table" style="width: 100%; border-collapse: collapse;">
        <thead>
            <tr>
                <th style="text-align: left; padding: 8px; border-bottom: 1px solid #ddd;"><?php esc_html_e( 'Order ID', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                <th style="text-align: left; padding: 8px; border-bottom: 1px solid #ddd;"><?php esc_html_e( 'Order Type', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th> 
                <th style="text-align: left; padding: 8px; border-bottom: 1px solid #ddd;"><?php esc_html_e( 'Status', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
                <th style="text-align: left; padding: 8px; border-bottom: 1px solid #ddd;"><?php esc_html_e( 'Paid On', 'reordere-lite-subcriptions-for-woocommerce' ); ?></th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ( $order_ids as $orderid ) { ?>
            <tr>
                <td style="padding: 8px; border-bottom: 1px solid #ddd;">
                    <a href="<?php echo esc_url( admin_url( 'post.php?post=' . $orderid . '&action=edit' ) ); ?>" target="_blank" style="text-decoration: none;">
                        <?php 
                        $orderid = '#' . esc_html( $orderid );
                        echo esc_html( $orderid ); 
                        ?>
                    </a>
                </td>
                <td style="padding: 8px; border-bottom: 1px solid #ddd;"><?php echo esc_html( 'First Order' ); ?></td>
                <td style="padding: 8px; border-bottom: 1px solid #ddd;"><?php echo esc_html( $order_status ); ?></td>
                <td style="padding: 8px; border-bottom: 1px solid #ddd;"><?php echo esc_html( $formatted_date ); ?></td>
            </tr>
            <?php } ?>
        </tbody>
    </table>
</div>

    <!-- </div> -->
</div>
