<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Create Bulk Subscription Products
 *
 * @package TS_Subscription
 */

// Get option for the subscription product to add/edit in bulk.
$subscription_products = get_option('subscription_products');
$subscription_products = json_decode($subscription_products, true);

// Get all the products.
$products = wp_cache_get('all_products');
if (false === $products) {
    $products = wc_get_products(array('limit' => -1));
    wp_cache_set('all_products', $products);
}

// Scripts are now enqueued properly in the main admin class - no need to enqueue here

// Create a form to add the subscription products.
?>

<div class="subscription-form-container bulk_subscription">
    <!-- Page Header -->
    <div class="bulk-subscription-header">
        <h1><i class="dashicons dashicons-plus-alt"></i> Create Subscription Products</h1>
        <p>Set up subscription plans for your products with flexible pricing and intervals</p>
    </div>

    <!-- Two Column Layout -->
    <div class="bulk-subscription-layout">
        <!-- Form Section -->
        <div class="bulk-subscription-form-section">
            <form method="post" action="" id="bulk-subscription-form">
                <?php wp_nonce_field('save_bulk_subscription'); ?>
                
                <!-- Basic Information Section -->
                <div class="section-header">
                    <i class="dashicons dashicons-info"></i>
                    <h2><?php esc_html_e('Basic Information', 'reordere-lite-subcriptions-for-woocommerce'); ?></h2>
                </div>

                <div class="form-group">
                    <label for="main_subscription_title"><?php esc_html_e('Subscription Title', 'reordere-lite-subcriptions-for-woocommerce'); ?><span class="astrik_icon">*</span></label>
                    <input type="text" name="main_subscription_title" id="subscription_title" class="form-control" placeholder="e.g., Premium Monthly Plan" required>
                </div>

                <div class="form-group">
                    <label for="subscription_products"><?php esc_html_e('Select Products', 'reordere-lite-subcriptions-for-woocommerce'); ?><span class="astrik_icon">*</span></label>
                    <select name="subscription_products[]" id="subscription_products" class="form-control" multiple required>
                        <?php
                        // Fetch all products.
                        foreach ($products as $product) {
                            $enable_sub = get_post_meta( $product->get_id(), '_enable_subscription', true );
                            // Check if the product is already a subscription product.
                            if($enable_sub == 'yes'){
                                continue;
                            }
                            echo '<option data-regular-price="' . esc_attr( $product->get_regular_price() ) . '" value="' . esc_attr($product->get_id()) . '">' . esc_html($product->get_name()) . '</option>';
                        }
                        ?>
                    </select>
                </div>

                <!-- Discount Options Section -->
                <div class="section-header">
                    <i class="dashicons dashicons-tag"></i>
                    <h2><?php esc_html_e('Discount Options', 'reordere-lite-subcriptions-for-woocommerce'); ?></h2>
                </div>

                <div class="form-group bulk_subscription_options">
                    <label>
                        <input type="radio" name="subscription_discount_type" value="flat_rate">
                        <?php esc_html_e('Flat Rate', 'reordere-lite-subcriptions-for-woocommerce'); ?>
                    </label>
                    <label>
                        <input type="radio" name="subscription_discount_type" value="percentage">
                        <?php esc_html_e('Percentage off', 'reordere-lite-subcriptions-for-woocommerce'); ?>
                    </label>
                    <label>
                        <input type="radio" name="subscription_discount_type" value="amount">
                        <?php esc_html_e('Amount off', 'reordere-lite-subcriptions-for-woocommerce'); ?>
                    </label>
                </div>

                <!-- Plan Details Section -->
                <div class="section-header">
                    <i class="dashicons dashicons-calendar-alt"></i>
                    <h2><?php esc_html_e('Subscription Plans', 'reordere-lite-subcriptions-for-woocommerce'); ?></h2>
                </div>
                <div id="subscription-fields">
                    <div class="form-row subscription-field-row" data-row-id="0">
                        <div class="form-group">
                            <label for="_subs_title_1"><?php esc_html_e('Plan Name', 'reordere-lite-subcriptions-for-woocommerce'); ?></label>
                            <input type="text" name="subscription_fields[0][_subs_title]" placeholder="e.g., Monthly Plan" class="form-control" required>
                        </div>
                        <div class="form-group">
                            <label for="_subs_period_1"><?php esc_html_e('Period', 'reordere-lite-subcriptions-for-woocommerce'); ?></label>
                            <input type="number" name="subscription_fields[0][_subs_period]" placeholder="1" class="form-control" min="1" required>
                        </div>

                        <div class="form-group">
                            <label for="_subs_period_interval_1"><?php esc_html_e('Interval', 'reordere-lite-subcriptions-for-woocommerce'); ?></label>
                            <select name="subscription_fields[0][_subs_period_interval]" class="form-control">
                                <option value="day"><?php esc_html_e('Day(s)', 'reordere-lite-subcriptions-for-woocommerce'); ?></option>
                                <option value="week"><?php esc_html_e('Week(s)', 'reordere-lite-subcriptions-for-woocommerce'); ?></option>
                                <option value="month" selected><?php esc_html_e('Month(s)', 'reordere-lite-subcriptions-for-woocommerce'); ?></option>
                                <option value="year"><?php esc_html_e('Year(s)', 'reordere-lite-subcriptions-for-woocommerce'); ?></option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="_subs_price_1"><?php esc_html_e('Price', 'reordere-lite-subcriptions-for-woocommerce'); ?></label>
                            <input type="number" name="subscription_fields[0][_subs_price_field]" placeholder="0.00" step="0.01" class="form-control">
                        </div>

                    </div>
                </div>

                <!-- Form Actions -->
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" id="add-more-fields">
                        <i class="dashicons dashicons-plus-alt"></i> <?php esc_html_e('Add More Plans', 'reordere-lite-subcriptions-for-woocommerce'); ?>
                    </button>
                    <button type="submit" class="btn btn-primary" id="save-bulk-subscription">
                        <i class="dashicons dashicons-yes"></i> <?php esc_html_e('Save Subscription', 'reordere-lite-subcriptions-for-woocommerce'); ?>
                    </button>
                </div>
            </form>
        </div>
        
        <!-- Preview Panel (Right Side) -->
        <div class="bulk-subscription-preview">
            <div class="preview-header">
                <i class="dashicons dashicons-visibility"></i>
                <h3><?php esc_html_e('Live Preview', 'reordere-lite-subcriptions-for-woocommerce'); ?></h3>
            </div>
            <div class="preview-content">
                <div class="preview-empty">
                    <i class="dashicons dashicons-format-gallery"></i>
                    <p><?php esc_html_e('Select products and configure plans to see preview', 'reordere-lite-subcriptions-for-woocommerce'); ?></p>
                </div>
            </div>
        </div>
    </div><!-- .bulk-subscription-layout -->
</div><!-- .subscription-form-container -->

<?php
if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST') {

    // Verify nonce using sanitize_text_field then unslash.
    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'save_bulk_subscription')) {
        die('Security check failed');
    }

    $errors = [];

    // Validate Subscription Title.
    $main_subscription_title = isset($_POST['main_subscription_title']) ? sanitize_text_field(wp_unslash($_POST['main_subscription_title'])) : '';
    if (empty($main_subscription_title)) {
        $errors[] = __('Subscription Title is required.', 'reordere-lite-subcriptions-for-woocommerce');
    }

    // Validate Selected Products.
    $subscription_products = isset($_POST['subscription_products']) ? array_map('intval', wp_unslash($_POST['subscription_products'])) : [];
    if (empty($subscription_products)) {
        $errors[] = __('Please select at least one product.', 'reordere-lite-subcriptions-for-woocommerce');
    }

    // Validate Discount Type.
    $subscription_discount_type = isset($_POST['subscription_discount_type']) ? sanitize_text_field(wp_unslash($_POST['subscription_discount_type'])) : '';
    if (empty($subscription_discount_type)) {
        $errors[] = __('Please select a discount type.', 'reordere-lite-subcriptions-for-woocommerce');
    }

    // Validate Subscription Fields - properly sanitize the array
    $subscription_fields_raw = isset($_POST['subscription_fields']) ? map_deep(wp_unslash($_POST['subscription_fields']), 'sanitize_text_field') : [];
    $subscription_fields = [];
    
    // No need for additional sanitization since we used map_deep above
    $subscription_fields = $subscription_fields_raw;
    
    if (empty($subscription_fields) || !is_array($subscription_fields)) {
        $errors[] = __('Please provide subscription plan details.', 'reordere-lite-subcriptions-for-woocommerce');
    } else {
        foreach ($subscription_fields as $field) {
            if (empty($field['_subs_period'])) {
                $errors[] = __('Plan Period is required.', 'reordere-lite-subcriptions-for-woocommerce');
            }
            if (empty($field['_subs_period_interval'])) {
                $errors[] = __('Plan Interval is required.', 'reordere-lite-subcriptions-for-woocommerce');
            }
            if (empty($field['_subs_price_field'])) {
                $errors[] = __('Plan Price is required.', 'reordere-lite-subcriptions-for-woocommerce');
            }
        }
    }

    if (!empty($errors)) {
        // Display errors and prevent further processing.
        foreach ($errors as $error) {
            echo '<p style="color: red;">' . esc_html($error) . '</p>';
        }
        return;
    }

    // Proceed with saving the subscription data.
    $product_ids = $subscription_products;
    foreach ($product_ids as $product_id) {
        // Enable subscription for the product
        update_post_meta($product_id, '_enable_subscription', 'yes');
        
        // Set discount type
        update_post_meta($product_id, '_subscription_discount_type', $subscription_discount_type);
        
        // Set subscription title
        update_post_meta($product_id, '_main_subscription_title', $main_subscription_title);
        
        // Set subscription fields/plans
        update_post_meta($product_id, '_subscription_fields', $subscription_fields);
    }

    echo '<p style="color: green;">' . esc_html__('Subscription saved successfully!', 'reordere-lite-subcriptions-for-woocommerce') . '</p>';
}
?>
