/**
 * Modern Subscription List & Details JavaScript
 * Adds interactive enhancements and better UX
 */

(function($) {
    'use strict';

    $(document).ready(function() {
        
        // Initialize modern subscription list
        initSubscriptionList();
        
        // Initialize subscription details
        initSubscriptionDetails();
        
        // Add search and filter functionality
        initFiltersAndSearch();
        
        // Add smooth animations
        initAnimations();
    });
    
    /**
     * Initialize subscription list enhancements
     */
    function initSubscriptionList() {
        const $listWrap = $('.wrap:has(.widefat)').first();
        
        if ($listWrap.length) {
            // Add modern wrapper class
            $listWrap.addClass('subscription-list-wrap');
            
            // Enhance header
            const $header = $listWrap.find('h1.wp-heading-inline').first();
            if ($header.length) {
                $header.wrap('<div class="subscription-list-header"></div>');
            }
            
            // Add modern class to table
            const $table = $listWrap.find('.widefat');
            if ($table.length) {
                $table.addClass('subscription-table-modern');
                
                // Enhance status display
                enhanceStatusBadges($table);
                
                // Add tooltips to long product names
                addProductTooltips($table);
            }
            
            // Add filters section if it doesn't exist
            if (!$('.subscription-filters').length) {
                addFiltersSection($listWrap);
            }
            
            // Handle empty state
            if ($table.find('tbody tr').length === 0) {
                showEmptyState($listWrap);
            }
        }
    }
    
    /**
     * Initialize subscription details enhancements
     */
    function initSubscriptionDetails() {
        const $detailsWrap = $('.wrap:has(.subscription-container)');
        
        if ($detailsWrap.length) {
            // Add modern wrapper class
            $detailsWrap.addClass('subscription-details-wrap');
            
            // Enhance header with ID and status
            enhanceDetailsHeader($detailsWrap);
            
            // Add icons to section headers
            addSectionIcons($detailsWrap);
            
            // Enhance form interactions
            enhanceFormInteractions($detailsWrap);
            
            // Add confirmation for updates
            addUpdateConfirmation($detailsWrap);
        }
    }
    
    /**
     * Enhance status badges with modern styling
     */
    function enhanceStatusBadges($table) {
        $table.find('tbody tr').each(function() {
            const $row = $(this);
            const $statusCell = $row.find('td').eq(1); // Assuming status is second column
            
            if ($statusCell.length) {
                const statusText = $statusCell.text().trim().toLowerCase();
                
                if (statusText && !$statusCell.find('.subscription-status-badge').length) {
                    const statusClass = 'status-' + statusText.replace(/\s+/g, '-');
                    $statusCell.html(
                        '<span class="subscription-status-badge ' + statusClass + '">' + 
                        statusText.charAt(0).toUpperCase() + statusText.slice(1) + 
                        '</span>'
                    );
                }
            }
        });
    }
    
    /**
     * Add tooltips to product names
     */
    function addProductTooltips($table) {
        $table.find('tbody td a[href*="post.php"]').each(function() {
            const $link = $(this);
            const productName = $link.text();
            
            if (productName.length > 30) {
                $link.attr('title', productName);
                $link.css('cursor', 'help');
            }
        });
    }
    
    /**
     * Add filters section
     */
    function addFiltersSection($wrap) {
        const filtersHTML = `
            <div class="subscription-filters">
                <div class="subscription-filter-group">
                    <label>Search</label>
                    <input type="text" id="subscription-search" placeholder="Search subscriptions...">
                </div>
                <div class="subscription-filter-group">
                    <label>Status</label>
                    <select id="subscription-status-filter">
                        <option value="">All Statuses</option>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                        <option value="cancelled">Cancelled</option>
                        <option value="paused">Paused</option>
                    </select>
                </div>
                <div class="subscription-filter-group">
                    <label>Date Range</label>
                    <input type="date" id="subscription-date-from" placeholder="From">
                </div>
                <div class="subscription-filter-group">
                    <label>&nbsp;</label>
                    <input type="date" id="subscription-date-to" placeholder="To">
                </div>
                <div class="subscription-filter-group">
                    <label>&nbsp;</label>
                    <button type="button" class="subscription-filter-btn" id="apply-filters">
                        Apply Filters
                    </button>
                </div>
            </div>
        `;
        
        $wrap.find('h1.wp-heading-inline').closest('.subscription-list-header').after(filtersHTML);
    }
    
    /**
     * Show empty state when no subscriptions
     */
    function showEmptyState($wrap) {
        const emptyStateHTML = `
            <div class="subscription-empty-state">
                <h3>No Subscriptions Found</h3>
                <p>There are no subscriptions matching your criteria.</p>
            </div>
        `;
        
        $wrap.find('.widefat').replaceWith(emptyStateHTML);
    }
    
    /**
     * Enhance details page header
     */
    function enhanceDetailsHeader($wrap) {
        const $mainHeader = $wrap.find('.subscription-details h1').first();
        
        if ($mainHeader.length) {
            const headerText = $mainHeader.html();
            const $status = $mainHeader.find('span');
            
            if ($status.length) {
                const statusText = $status.text().trim().toLowerCase();
                const statusClass = 'status-' + statusText.replace(/\s+/g, '-');
                
                $status.removeClass().addClass('subscription-status-badge ' + statusClass);
            }
            
            // Wrap in header container if not already
            if (!$mainHeader.parent().hasClass('subscription-details-header')) {
                $mainHeader.wrap('<div class="subscription-details-header"></div>');
            }
        }
    }
    
    /**
     * Add icons to section headers
     */
    function addSectionIcons($wrap) {
        // Icons are now added via CSS ::before pseudo-elements
        // This function can be used for dynamic icon management if needed
    }
    
    /**
     * Enhance form interactions
     */
    function enhanceFormInteractions($wrap) {
        const $form = $wrap.find('.subscription-settings form');
        
        if ($form.length) {
            // Add loading state on submit
            $form.on('submit', function() {
                const $submitBtn = $(this).find('.update_subs_btn');
                $submitBtn.prop('disabled', true).text('Updating...');
                
                // Add loading indicator
                $submitBtn.append(' <span class="spinner" style="display: inline-block; margin-left: 5px;"></span>');
            });
            
            // Highlight changed fields
            $form.find('input, select').on('change', function() {
                $(this).css('border-color', '#f59e0b');
            });
        }
    }
    
    /**
     * Add update confirmation
     */
    function addUpdateConfirmation($wrap) {
        const $form = $wrap.find('.subscription-settings form');
        
        if ($form.length) {
            $form.on('submit', function(e) {
                const hasChanges = $form.find('input[style*="border-color"], select[style*="border-color"]').length > 0;
                
                if (hasChanges) {
                    const confirmed = confirm('Are you sure you want to update this subscription?');
                    if (!confirmed) {
                        e.preventDefault();
                        return false;
                    }
                }
            });
        }
    }
    
    /**
     * Initialize filters and search
     */
    function initFiltersAndSearch() {
        // Search functionality
        $(document).on('input', '#subscription-search', debounce(function() {
            const searchTerm = $(this).val().toLowerCase();
            filterTableRows(searchTerm);
        }, 300));
        
        // Status filter
        $(document).on('change', '#subscription-status-filter', function() {
            const status = $(this).val().toLowerCase();
            filterByStatus(status);
        });
        
        // Apply filters button
        $(document).on('click', '#apply-filters', function() {
            applyAllFilters();
        });
        
        // Date range filters
        $(document).on('change', '#subscription-date-from, #subscription-date-to', function() {
            filterByDateRange();
        });
    }
    
    /**
     * Filter table rows by search term
     */
    function filterTableRows(searchTerm) {
        const $table = $('.subscription-table-modern tbody');
        
        $table.find('tr').each(function() {
            const $row = $(this);
            const rowText = $row.text().toLowerCase();
            
            if (rowText.includes(searchTerm)) {
                $row.show().addClass('fade-in');
            } else {
                $row.hide();
            }
        });
        
        checkEmptyResults();
    }
    
    /**
     * Filter by status
     */
    function filterByStatus(status) {
        const $table = $('.subscription-table-modern tbody');
        
        $table.find('tr').each(function() {
            const $row = $(this);
            const $statusBadge = $row.find('.subscription-status-badge');
            const rowStatus = $statusBadge.length ? 
                $statusBadge.text().toLowerCase() : 
                $row.find('td').eq(1).text().toLowerCase();
            
            if (status === '' || rowStatus.includes(status)) {
                $row.show().addClass('fade-in');
            } else {
                $row.hide();
            }
        });
        
        checkEmptyResults();
    }
    
    /**
     * Filter by date range
     */
    function filterByDateRange() {
        const dateFrom = $('#subscription-date-from').val();
        const dateTo = $('#subscription-date-to').val();
        
        if (!dateFrom && !dateTo) return;
        
        const $table = $('.subscription-table-modern tbody');
        const fromDate = dateFrom ? new Date(dateFrom) : null;
        const toDate = dateTo ? new Date(dateTo) : null;
        
        $table.find('tr').each(function() {
            const $row = $(this);
            const dateText = $row.find('td').eq(2).text().trim(); // Start date column
            const rowDate = new Date(dateText);
            
            let show = true;
            
            if (fromDate && rowDate < fromDate) show = false;
            if (toDate && rowDate > toDate) show = false;
            
            if (show) {
                $row.show().addClass('fade-in');
            } else {
                $row.hide();
            }
        });
        
        checkEmptyResults();
    }
    
    /**
     * Apply all filters
     */
    function applyAllFilters() {
        const searchTerm = $('#subscription-search').val().toLowerCase();
        const status = $('#subscription-status-filter').val().toLowerCase();
        const dateFrom = $('#subscription-date-from').val();
        const dateTo = $('#subscription-date-to').val();
        
        const $table = $('.subscription-table-modern tbody');
        const fromDate = dateFrom ? new Date(dateFrom) : null;
        const toDate = dateTo ? new Date(dateTo) : null;
        
        $table.find('tr').each(function() {
            const $row = $(this);
            const rowText = $row.text().toLowerCase();
            const $statusBadge = $row.find('.subscription-status-badge');
            const rowStatus = $statusBadge.length ? 
                $statusBadge.text().toLowerCase() : 
                $row.find('td').eq(1).text().toLowerCase();
            const dateText = $row.find('td').eq(2).text().trim();
            const rowDate = new Date(dateText);
            
            let show = true;
            
            // Check search
            if (searchTerm && !rowText.includes(searchTerm)) show = false;
            
            // Check status
            if (status && !rowStatus.includes(status)) show = false;
            
            // Check date range
            if (fromDate && rowDate < fromDate) show = false;
            if (toDate && rowDate > toDate) show = false;
            
            if (show) {
                $row.show().addClass('fade-in');
            } else {
                $row.hide();
            }
        });
        
        checkEmptyResults();
        
        // Show success message
        showFilterMessage('Filters applied successfully!', 'success');
    }
    
    /**
     * Check if any results remain after filtering
     */
    function checkEmptyResults() {
        const $table = $('.subscription-table-modern tbody');
        const visibleRows = $table.find('tr:visible').length;
        
        if (visibleRows === 0 && !$('.subscription-empty-state').length) {
            $table.after(`
                <div class="subscription-empty-state" style="margin-top: 20px;">
                    <h3>No Results Found</h3>
                    <p>Try adjusting your filters or search criteria.</p>
                </div>
            `);
        } else if (visibleRows > 0) {
            $('.subscription-empty-state').remove();
        }
    }
    
    /**
     * Show filter message
     */
    function showFilterMessage(message, type) {
        const messageClass = type === 'success' ? 'success' : 'error';
        const $message = $(`
            <div class="subscription-notice ${messageClass}">
                ${message}
            </div>
        `);
        
        $('.subscription-filters').after($message);
        
        setTimeout(() => {
            $message.fadeOut(() => $message.remove());
        }, 3000);
    }
    
    /**
     * Initialize animations
     */
    function initAnimations() {
        // Add fade-in animation for table rows
        $('.subscription-table-modern tbody tr').each(function(index) {
            $(this).css({
                'opacity': '0',
                'transform': 'translateY(10px)'
            }).delay(index * 50).animate({
                'opacity': '1'
            }, 300).css('transform', 'translateY(0)');
        });
        
        // Add hover effects
        $('.subscription-table-modern tbody tr').hover(
            function() {
                $(this).css('box-shadow', '0 4px 12px rgba(0, 0, 0, 0.1)');
            },
            function() {
                $(this).css('box-shadow', 'none');
            }
        );
        
        // Smooth scroll to details
        $('a[href*="subscription_id"]').on('click', function(e) {
            if ($(this).attr('href').includes('action=edit')) {
                // Add loading indicator
                const $loadingOverlay = $(`
                    <div class="subscription-loading" style="position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(255, 255, 255, 0.9); z-index: 9999;">
                    </div>
                `);
                $('body').append($loadingOverlay);
            }
        });
    }
    
    /**
     * Debounce function for search
     */
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    /**
     * Add export functionality
     */
    function addExportButton() {
        const $header = $('.subscription-list-header');
        if ($header.length && !$('#export-subscriptions').length) {
            const $exportBtn = $(`
                <button type="button" id="export-subscriptions" class="button button-primary" style="margin-left: 10px;">
                    Export to CSV
                </button>
            `);
            
            $header.append($exportBtn);
            
            $exportBtn.on('click', function() {
                exportTableToCSV('subscriptions.csv');
            });
        }
    }
    
    /**
     * Export table to CSV
     */
    function exportTableToCSV(filename) {
        const $table = $('.subscription-table-modern');
        const csv = [];
        const rows = $table.find('tr:visible');
        
        rows.each(function() {
            const row = [];
            $(this).find('th, td').each(function() {
                row.push('"' + $(this).text().trim().replace(/"/g, '""') + '"');
            });
            csv.push(row.join(','));
        });
        
        const csvContent = csv.join('\n');
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);
        
        link.setAttribute('href', url);
        link.setAttribute('download', filename);
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        
        showFilterMessage('Export completed successfully!', 'success');
    }
    
    // Initialize export button
    setTimeout(addExportButton, 500);
    
    /**
     * Add scroll to top button
     */
    function addScrollToTop() {
        if (!$('#subscription-scroll-top').length) {
            const $scrollBtn = $('<button id="subscription-scroll-top" class="subscription-scroll-top">↑</button>');
            $('body').append($scrollBtn);
            
            // Show/hide based on scroll position
            $(window).on('scroll', function() {
                if ($(this).scrollTop() > 300) {
                    $scrollBtn.addClass('visible');
                } else {
                    $scrollBtn.removeClass('visible');
                }
            });
            
            // Scroll to top on click
            $scrollBtn.on('click', function() {
                $('html, body').animate({ scrollTop: 0 }, 600);
            });
        }
    }
    
    /**
     * Add tooltips to elements
     */
    function addTooltips() {
        // Add tooltips to status badges
        $('.subscription-status-badge').each(function() {
            const status = $(this).text().trim().toLowerCase();
            let tooltipText = '';
            
            switch(status) {
                case 'active':
                    tooltipText = 'Subscription is currently active';
                    break;
                case 'inactive':
                    tooltipText = 'Subscription has been deactivated';
                    break;
                case 'cancelled':
                    tooltipText = 'Subscription was cancelled';
                    break;
                case 'paused':
                    tooltipText = 'Subscription is temporarily paused';
                    break;
            }
            
            if (tooltipText) {
                $(this).addClass('subscription-tooltip').attr('data-tooltip', tooltipText);
            }
        });
    }
    
    /**
     * Add keyboard shortcuts
     */
    function addKeyboardShortcuts() {
        $(document).on('keydown', function(e) {
            // Ctrl/Cmd + F to focus search
            if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
                const $searchInput = $('#subscription-search');
                if ($searchInput.length) {
                    e.preventDefault();
                    $searchInput.focus();
                }
            }
            
            // ESC to clear filters
            if (e.key === 'Escape') {
                $('#subscription-search').val('');
                $('#subscription-status-filter').val('');
                $('#subscription-date-from').val('');
                $('#subscription-date-to').val('');
                $('.subscription-table-modern tbody tr').show();
                $('.subscription-active-filters').remove();
            }
        });
    }
    
    /**
     * Show active filters as tags
     */
    function showActiveFilters() {
        const searchTerm = $('#subscription-search').val();
        const status = $('#subscription-status-filter').val();
        const dateFrom = $('#subscription-date-from').val();
        const dateTo = $('#subscription-date-to').val();
        
        // Remove existing filter tags
        $('.subscription-active-filters').remove();
        
        if (!searchTerm && !status && !dateFrom && !dateTo) return;
        
        const $filterContainer = $('<div class="subscription-active-filters"></div>');
        
        if (searchTerm) {
            $filterContainer.append(`
                <span class="subscription-filter-tag">
                    Search: ${searchTerm}
                    <span class="subscription-filter-tag-remove" data-filter="search">×</span>
                </span>
            `);
        }
        
        if (status) {
            $filterContainer.append(`
                <span class="subscription-filter-tag">
                    Status: ${status}
                    <span class="subscription-filter-tag-remove" data-filter="status">×</span>
                </span>
            `);
        }
        
        if (dateFrom || dateTo) {
            const dateRange = `${dateFrom || 'Start'} - ${dateTo || 'End'}`;
            $filterContainer.append(`
                <span class="subscription-filter-tag">
                    Date: ${dateRange}
                    <span class="subscription-filter-tag-remove" data-filter="date">×</span>
                </span>
            `);
        }
        
        $('.subscription-filters').after($filterContainer);
        
        // Handle tag removal
        $('.subscription-filter-tag-remove').on('click', function() {
            const filter = $(this).data('filter');
            
            switch(filter) {
                case 'search':
                    $('#subscription-search').val('');
                    break;
                case 'status':
                    $('#subscription-status-filter').val('');
                    break;
                case 'date':
                    $('#subscription-date-from').val('');
                    $('#subscription-date-to').val('');
                    break;
            }
            
            applyAllFilters();
        });
    }
    
    // Initialize additional features
    setTimeout(() => {
        addScrollToTop();
        addTooltips();
        addKeyboardShortcuts();
    }, 1000);
    
    // Update apply filters to show active filters
    $(document).on('click', '#apply-filters', function() {
        applyAllFilters();
        showActiveFilters();
    });

})(jQuery);
