jQuery(document).ready(function ($) {
    
    // Load metrics on page load
    loadMetrics();
    loadStatusChart();
    loadPaymentChart();
    loadTopProducts();
    
    // Function to load subscription metrics
    function loadMetrics() {
        $.ajax({
            url: reportData.ajax_url,
            method: 'POST',
            data: {
                action: 'get_subscription_metrics',
                _ajax_nonce: reportData.nonce
            },
            success: function (response) {
                if (response.success) {
                    const metrics = response.data;
                    $('#revenue-value').text(reportData.currency_symbol + (metrics.revenue || 0).toFixed(2));
                    $('#active-count').text(metrics.active || 0);
                    $('#total-count').text(metrics.total || 0);
                    $('#cancelled-count').text(metrics.cancelled || 0);
                }
            },
            error: function (xhr, status, error) {
                console.log('Error loading metrics:', error);
            }
        });
    }
    
    // Function to load status distribution chart
    function loadStatusChart() {
        $.ajax({
            url: reportData.ajax_url,
            method: 'POST',
            data: {
                action: 'get_subscription_report_data',
                _ajax_nonce: reportData.nonce
            },
            success: function (response) {
                if (response.success) {
                    const statusData = response.data;
                    renderStatusChart(statusData);
                }
            },
            error: function (xhr, status, error) {
                console.log('Error loading status chart:', error);
            }
        });
    }
    
    // Function to render status distribution chart
    function renderStatusChart(data) {
        const ctx = document.getElementById('statusChart');
        if (!ctx) return;
        
        new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: ['Active', 'Inactive', 'Paused', 'Cancelled', 'Expired'],
                datasets: [{
                    data: [
                        data.active || 0,
                        data.inactive || 0,
                        data.paused || 0,
                        data.cancelled || 0,
                        data.expired || 0
                    ],
                    backgroundColor: [
                        '#28a745',  // Green for Active
                        '#17a2b8',  // Blue for Inactive
                        '#ffc107',  // Yellow for Paused
                        '#dc3545',  // Red for Cancelled
                        '#6c757d'   // Gray for Expired
                    ],
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                plugins: {
                    legend: {
                        position: 'bottom'
                    },
                    title: {
                        display: false
                    }
                }
            }
        });
    }
    
    // Function to load payment methods chart
    function loadPaymentChart() {
        $.ajax({
            url: reportData.ajax_url,
            method: 'POST',
            data: {
                action: 'get_payment_methods',
                _ajax_nonce: reportData.nonce
            },
            success: function (response) {
                if (response.success) {
                    const paymentData = response.data;
                    renderPaymentChart(paymentData);
                }
            },
            error: function (xhr, status, error) {
                console.log('Error loading payment chart:', error);
            }
        });
    }
    
    // Function to render payment methods chart
    function renderPaymentChart(data) {
        const ctx = document.getElementById('paymentChart');
        if (!ctx) return;
        
        const labels = Object.keys(data);
        const values = Object.values(data);
        
        new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Subscriptions',
                    data: values,
                    backgroundColor: '#00aaff',
                    borderColor: '#0088cc',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            precision: 0
                        }
                    }
                }
            }
        });
    }
    
    // Function to load top products table
    function loadTopProducts() {
        $.ajax({
            url: reportData.ajax_url,
            method: 'POST',
            data: {
                action: 'get_top_products',
                _ajax_nonce: reportData.nonce
            },
            success: function (response) {
                console.log('Top products response:', response);
                if (response.success) {
                    renderTopProducts(response.data);
                } else {
                    console.error('Top products error:', response);
                    $('#top-products-table tbody').html('<tr><td colspan="4">Error: ' + (response.data || 'Unknown error') + '</td></tr>');
                }
            },
            error: function (xhr, status, error) {
                console.log('Error loading top products:', error);
                console.log('XHR:', xhr);
                $('#top-products-table tbody').html('<tr><td colspan="4">Error loading data</td></tr>');
            }
        });
    }
    
    // Function to render top products table
    function renderTopProducts(products) {
        const tbody = $('#top-products-table');
        tbody.empty();
        
        if (!products || products.length === 0) {
            tbody.html('<tr><td colspan="3">No subscription products found</td></tr>');
            return;
        }
        
        products.forEach(function (product, index) {
            const row = `
                <tr>
                    <td>${product.product_name}</td>
                    <td>${product.subscription_count}</td>
                    <td>${reportData.currency_symbol}${parseFloat(product.total_revenue || 0).toFixed(2)}</td>
                </tr>
            `;
            tbody.append(row);
        });
    }
    
    const ctx = document.getElementById('reportChart').getContext('2d');
    let chart;

    // Function to fetch and render chart data
    function fetchChartData(timeframe) {
        $.ajax({
            url: reportData.ajax_url,
            method: 'POST',
            data: {
                action: 'get_chart_data',
                timeframe: timeframe,
                _ajax_nonce: reportData.nonce // Add nonce
            },
            success: function (response) {
                if (response.success) {
                    const chartData = response.data.data; // Accessing the dynamic data
                    const salesData = response.data.sales; // Accessing the sales data

                    let labels;
                    let data;

                    if (timeframe === 'day') {
                        // Static labels for days of the week (Mon-Sun)
                        labels = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"];
                        
                        // Map data to these days
                        data = mapDayData(chartData.data, chartData.labels[0]); // Get data for single day
                    } else {
                        // If it's not 'day', use the provided labels and data (e.g., for weeks/months)
                        labels = formatLabels(chartData.labels); // Format week/month labels
                        data = chartData.data;
                    }

                    // Update chart with the fetched data
                    updateChart(labels, data);
                    updateSalesCards(salesData); // Update sales info on the page
                } else {
                    console.log('Error fetching data:', response.data); // Log error message
                }
            },
            error: function (xhr, status, error) {
                console.log('AJAX error:', error);
            }
        });
    }

    // Map the data to the days of the week (Monday to Sunday)
    function mapDayData(data, date) {
        const daysOfWeek = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"];
        const dayIndex = new Date(date).getDay(); // Get the day index (0=Sun, 1=Mon, ..., 6=Sat)

        // Fill in all days with zero, and add the data on the corresponding day
        const mappedData = Array(7).fill(0);
        mappedData[dayIndex === 0 ? 6 : dayIndex - 1] = data[0]; // Map data to correct day

        return mappedData;
    }

    // Function to format week labels into readable format (e.g., "Week 202450" -> "Week 50, 2024")
    function formatLabels(labels) {
        return labels.map(label => {
            const match = label.match(/Week (\d{4})(\d{2})/); // Regex to capture year and week
            if (match) {
                const year = match[1]; // Captured year
                const week = match[2]; // Captured week number
                return `Week ${week}, ${year}`;
            }
            return label; // Return original if regex fails
        });
    }

    // Initialize the chart with default timeframe (e.g., daily)
    function initializeChart(timeframe = 'daily') {
        chart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: [],  // Labels will be dynamically filled
                datasets: [
                    {
                        label: 'Subscription Products Sold',
                        data: [],  // Data will be dynamically filled
                        borderColor: '#00aaff',
                        backgroundColor: 'rgba(0, 170, 255, 0.2)',
                        fill: true,
                        tension: 0.3,
                        pointRadius: 3,
                        pointBackgroundColor: '#00aaff',
                    },
                ],
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        display: true,
                        position: 'top',
                    },
                    tooltip: {
                        callbacks: {
                            label: function (context) {
                                return 'Products Sold: ' + context.raw.toLocaleString();
                            },
                        },
                    },
                },
                scales: {
                    x: {
                        title: {
                            display: false,
                            text: 'Timeframe',
                        },
                        ticks: {
                            autoSkip: true,
                            maxTicksLimit: 7, // Limit to 7 ticks (days in a week)
                        },
                    },
                    y: {
                        title: {
                            display: true,
                            text: 'Products Sold',
                        },
                        ticks: {
                            precision: 0,
                            callback: function (value) {
                                return value.toLocaleString();
                            },
                        },
                    },
                },
            },
        });
    }

    // Update chart data
    function updateChart(labels, data) {
        chart.data.labels = labels;
        chart.data.datasets[0].data = data;
        chart.update();
    }

    // Update sales cards with new data
    function updateSalesCards(sales) {
        $('#daily-revenue span').text(sales.daily.toLocaleString());
        $('#weekly-revenue span').text(sales.weekly.toLocaleString());
        $('#monthly-revenue span').text(sales.monthly.toLocaleString());
        $('#yearly-revenue span').text(sales.yearly.toLocaleString());
    }

    // Event listener for timeframe buttons
    $('.timeframe-button').on('click', function () {
        $('.timeframe-button').removeClass('active');    
        $(this).addClass('active');
        
        const timeframe = $(this).data('timeframe');
        fetchChartData(timeframe);
    });

    // Initialize the chart on page load with default data (e.g., daily data)
    initializeChart();
    fetchChartData('daily');
});
