<?php
/**
 * License Helper for Subscription Products for WooCommerce
 *
 * @package reordere-lite-subcriptions-for-woocommerce
 */

 if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class Rorder_Menu_Helper {

    public function __construct() {
        add_action('init', array($this, 'init_sb_menu'));
    }

    /**
     * Initialize the license menu and AJAX actions.
     */
    public function init_sb_menu() {
        add_action('admin_footer', array($this, 'sb_menu_ajax_sync_script'));
        add_action('wp_ajax_save_sb_menu_options', array($this, 'sb_menu_options_save'));
        add_action('wp_ajax_remove_sb_menu_options', array($this, 'sb_menu_options_remove'));
    }

    /**
     * Enqueue the license script and localize AJAX URL and nonce.
     */
    public function sb_menu_ajax_sync_script() {
        $js_file_url = esc_url(plugins_url('', __FILE__) . '/sb-license.js');

        wp_enqueue_script('sb-license-js', $js_file_url, array('jquery'), '1.0', true);
        wp_localize_script('sb-license-js', 'sbLicenseNonce', array(
            'licenseurl' => esc_url(admin_url('admin-ajax.php')),
            'nonce' => wp_create_nonce('sb_license_nonce'),
        ));
    }

    /**
     * Handle AJAX request to save license options.
     */
    public function sb_menu_options_save() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'sb_license_nonce')) {
            wp_send_json('Nonce verification failed.');
            return;
        }

        // Sanitize and validate input data
        $square_email = isset($_POST['reordere_email']) ? sanitize_email(wp_unslash($_POST['reordere_email'])) : '';
        $purchase_code = isset($_POST['purchase_code']) ? sanitize_text_field(wp_unslash($_POST['purchase_code'])) : '';

        $domain = esc_url(get_option('siteurl'));

        $response = $this->sb_verify_purchase_code($square_email, $purchase_code, $domain);

        wp_send_json($response);
    }

    /**
     * Handle AJAX request to remove license options.
     */
    public function sb_menu_options_remove() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'sb_license_nonce')) {
            wp_send_json('Nonce verification failed.');
            return;
        }

        update_option('rorder_license', 'invalid');
        update_option('rorder_email', '');
        update_option('rorder_license_key', '');
    }

    /**
     * Verify the purchase code with the server.
     *
     * @param string $sb_email The email address.
     * @param string $purchase_code The purchase code.
     * @param string $domain The domain.
     * @return string The response message.
     */
    public function sb_verify_purchase_code($sb_email, $purchase_code, $domain) {
        $json_array = array(
            "purchase_code" => $purchase_code,
            "purchase_username" => $sb_email,
            "domain" => $domain,
            "email" => $sb_email
        );
        $json_format = wp_json_encode($json_array);

        $response = wp_remote_post("https://verify.techspawn.com/wp-json/my-route/verify/", array(
            'body' => $json_format,
            'headers' => array(
                'accept' => 'application/json',
                'content-type' => 'application/json'
            ),
        ));

        if (is_wp_error($response)) {
            $message = "Invalid Email Or Purchase Code";
        } else {
            $response_arr = json_decode(wp_remote_retrieve_body($response));

            if ($response_arr->result == 1) {
                $message = "Your License Code Has Been Validated. Thank you for your purchase.";
                update_option('rorder_license', 'valid');
                update_option('rorder_email', $sb_email);
                update_option('rorder_license_key', $purchase_code);
            } else {
                $message = $response_arr->message;
                update_option('rorder_license', 'invalid');
                update_option('rorder_email', $sb_email);
                update_option('rorder_license_key', $purchase_code);
            }
        }

        return $message;
    }
}

// Initialize the license helper.
$sb_menu_helper = new Rorder_Menu_Helper();



