<?php

/**
 * Admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and examples of how to enqueue
 * the admin-specific stylesheet and register menu pages.
 *
 * @link       http://www.techspawn.com
 * @since      1.0.0
 * @package    reordere-lite-subcriptions-for-woocommerce
 * @subpackage reordere-lite-subcriptions-for-woocommerce/admin
 * @author     Techspawn Solutions
 */

if ( ! class_exists( 'Rorder_Products_Admin' ) ) {
    class Rorder_Products_Admin {

        /**
         * The plugin name.
         *
         * @since    1.0.0
         * @access   private
         * @var      string    $plugin_name    The ID of this plugin.
         */
        private $plugin_name;

        /**
         * The version of this plugin.
         *
         * @since    1.0.0
         * @access   private
         * @var      string    $version        The current version of this plugin.
         */
        private $version;

        /**
         * Class constructor to initialize properties.
         *
         * @since    1.0.0
         * @param    string    $plugin_name    The name of the plugin.
         * @param    string    $version        The version of this plugin.
         */
        public function __construct($plugin_name, $version) {
            $this->plugin_name = $plugin_name;
            $this->version = $version;

            // Include the reports functions
            require_once plugin_dir_path(__FILE__) . 'partials/class-subscription-product-reports.php';

            // Register the admin menu and styles
            add_action('admin_enqueue_scripts', array($this, 'enqueue_styles'));
            add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
            add_action('woocommerce_process_product_meta', array($this, 'save_subscription_product_fields'), 10, 1);
            // Hook to save variations
            add_action('woocommerce_save_product_variation', array($this, 'save_subscription_fields_for_variations'), 10, 2);
            add_action('admin_enqueue_scripts', array($this, 'enqueue_report_scripts'));
            add_action('wp_ajax_get_chart_data', array($this, 'get_chart_data'));
            add_action('wp_ajax_nopriv_get_chart_data', array($this, 'get_chart_data'));
            add_action('wp_ajax_get_subscription_metrics', array($this, 'get_subscription_metrics_ajax'));
            add_action('wp_ajax_get_subscription_report_data', array($this, 'get_subscription_report_data_ajax'));
            add_action('wp_ajax_get_payment_methods', array($this, 'get_payment_methods_ajax'));
            add_action('wp_ajax_get_top_products', array($this, 'get_top_products_ajax'));
            add_action('wp_ajax_save_retry_rule', array($this, 'rorder_save_retry_rule'));
            add_action('wp_ajax_delete_retry_rule', array($this, 'delete_retry_rule'));
            add_action('wp_ajax_get_retry_rules', array($this, 'rorder_get_retry_rules'));
            add_action('wp_ajax_get_product_image', array($this, 'get_product_image'));
            add_action('admin_enqueue_scripts', array($this, 'enqueue_font_awesome'));
            
            // Add the admin_menu hook to register the menu pages
           // add_action('admin_menu', array($this, 'rorder_susbscription_register_menu_pages'));
        }


        public function rorder_save_retry_rule() {
            // Verify nonce
            if (!isset($_POST['_ajax_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])), 'save_retry_rule_nonce')) {
                wp_send_json_error('Invalid nonce');
            }

            global $wpdb;
        
            // Sanitize and validate incoming data
            $rule_name = sanitize_text_field(wp_unslash($_POST['data']['name']));
            $retry_interval_value = intval(wp_unslash($_POST['data']['retry_interval_value']));
            $retry_interval_unit = sanitize_text_field(wp_unslash($_POST['data']['retry_interval_unit']));
            $max_retries = intval(wp_unslash($_POST['data']['max_attempts']));
            
            // Process the gateways and statuses arrays by imploding them into strings
            $payment_gateways = !empty($_POST['data']['applicable_gateways']) ? implode(',', array_map('sanitize_text_field', wp_unslash($_POST['data']['applicable_gateways']))) : null;
            $order_statuses = !empty($_POST['data']['applicable_statuses']) ? implode(',', array_map('sanitize_text_field', wp_unslash($_POST['data']['applicable_statuses']))) : null;
            
            // Get the notify_customer checkbox value (0 if unchecked, 1 if checked)
            $notify_customer = intval(wp_unslash($_POST['data']['notify_customer']));
        
            // Prepare the data to insert into the database
            $table_name = $wpdb->prefix . 'rorder_retry_rules';
        
            // Insert the new rule into the database
            $wpdb->insert(
                $table_name,
                [
                    'rule_name' => $rule_name,
                    'retry_interval_value' => $retry_interval_value,
                    'retry_interval_unit' => $retry_interval_unit,
                    'max_retries' => $max_retries,
                    'payment_method' => $payment_gateways,  // Comma-separated string
                    'status' => $order_statuses,          // Comma-separated string
                    'notify_customer' => $notify_customer,  // Checkbox value (0 or 1)
                    'created_at' => current_time('mysql'),
                    'updated_at' => current_time('mysql')
                ]
            );
        
            // Return success response
            wp_send_json_success(['message' => 'Rule saved successfully!']);
        }
        
        

    public function rorder_get_retry_rules() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        global $wpdb;

        $table_name = $wpdb->prefix . 'rorder_retry_rules';

        // Fetch all rules
        $rules = wp_cache_get('rorder_retry_rules');
        if ($rules === false) {
            $rules = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name"));
            wp_cache_set('rorder_retry_rules', $rules);
        }

        // Return rules as JSON
        wp_send_json_success($rules);
    }



    public function delete_retry_rule() {
        // Verify nonce
        if (!isset($_POST['_ajax_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])), 'delete_retry_rule_nonce')) {
            wp_send_json_error('Invalid nonce');
        }

        global $wpdb;
        $rule_id = intval(wp_unslash($_POST['rule_id']));
        $table_name = $wpdb->prefix . 'rorder_retry_rules';
        $wpdb->delete($table_name, ['id' => $rule_id]);
        wp_cache_delete('rorder_retry_rules');
        wp_send_json_success();
    }




        /**
         * Register the stylesheets for the admin area.
         *
         * @since    1.0.0
         */
        public function enqueue_styles() {
            $screen = get_current_screen();
            
            wp_enqueue_style('subscription-products-admin', plugin_dir_url(__FILE__) . 'css/subscription-products-admin.css', array(), $this->version, 'all' );
            
            // Load Select2 from CDN for lite version
            wp_enqueue_style('select2-css', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css', array(), '4.1.0', 'all');
            
            // Enqueue modern subscription list styles on subscription pages
            if ($screen && ($screen->id == 'reorderre_page_subscriptions' || 
                (isset($_GET['page']) && $_GET['page'] == 'subscriptions'))) {
                wp_enqueue_style('subscription-list-modern', plugin_dir_url(__FILE__) . 'css/subscription-list-modern.css', array(), $this->version, 'all' );
                wp_enqueue_style('subscription-list-animations', plugin_dir_url(__FILE__) . 'css/subscription-list-animations.css', array('subscription-list-modern'), $this->version, 'all' );
            }
            
            // Enqueue modern bulk subscription styles on bulk subscription page
            if ($screen && ($screen->id == 'reorderre_page_bulk-subscription' || 
                (isset($_GET['page']) && $_GET['page'] == 'bulk subscription'))) {
                wp_enqueue_style('dashicons');
                wp_enqueue_style('bulk-subscription-modern', plugin_dir_url(__FILE__) . 'css/bulk-subscription-modern.css', array(), $this->version, 'all' );
            }
            
            // Enqueue modern reports dashboard styles on reports page
            if ($screen && ($screen->id == 'reorderre_page_reports' || 
                (isset($_GET['page']) && $_GET['page'] == 'reports'))) {
                wp_enqueue_style('reports-dashboard-css', plugin_dir_url(__FILE__) . 'css/reports-dashboard.css', array(), $this->version, 'all' );
            }
            
            // Enqueue subscription settings styles on settings page
            if ($screen && ($screen->id == 'reorderre_page_settings' || 
                (isset($_GET['page']) && $_GET['page'] == 'settings'))) {
                wp_enqueue_style('subscription-settings-css', plugin_dir_url(__FILE__) . 'css/subscription-settings.css', array(), $this->version, 'all' );
                wp_enqueue_style('dashicons');
            }
        }

        /**
         * Register the JavaScript for the admin area.
         *
         * @since    1.0.0
         */
        public function enqueue_scripts() {
            $screen = get_current_screen();
            
            // Only enqueue subscription-products-admin.js on product edit pages, NOT on bulk subscription page
            if (!$screen || ($screen->id != 'reorderre_page_bulk-subscription' && (!isset($_GET['page']) || $_GET['page'] != 'bulk subscription'))) {
                wp_enqueue_script('subscription-products-admin', plugin_dir_url(__FILE__) . 'js/subscription-products-admin.js', array('jquery'), $this->version, true );
            }
            
            // Load Select2 from CDN for lite version
            wp_enqueue_script('select2-js', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js', array('jquery'), '4.1.0', true);
            
            // Enqueue modern subscription list scripts on subscription pages
            if ($screen && ($screen->id == 'reorderre_page_subscriptions' || 
                (isset($_GET['page']) && $_GET['page'] == 'subscriptions'))) {
                wp_enqueue_script('subscription-list-modern', plugin_dir_url(__FILE__) . 'js/subscription-list-modern.js', array('jquery'), $this->version, true );
            }
            
            // Enqueue modern bulk subscription scripts on bulk subscription page
            if ($screen && ($screen->id == 'reorderre_page_bulk-subscription' || 
                (isset($_GET['page']) && $_GET['page'] == 'bulk subscription'))) {
                wp_enqueue_script('bulk-subscription-modern', plugin_dir_url(__FILE__) . 'js/bulk-subscription-modern.js', array('jquery', 'select2-js'), $this->version, true );
                
                // Localize script data for bulk subscription
                wp_localize_script('bulk-subscription-modern', 'bulkSubscriptionData', array(
                    'plugin_url' => plugin_dir_url(dirname(__FILE__)),
                    'selected_plan_view' => get_option('selected_plan_view', 'simple_view'),
                    'one_time_purchase_title' => get_option('one_time_purchase_title', __('One Time Purchase', 'reordere-lite-subcriptions-for-woocommerce')),
                    'subscribe_button_text' => get_option('subscription_button_text', __('Subscribe', 'reordere-lite-subcriptions-for-woocommerce'))
                ));
            }
            
            // Enqueue admin-script.js for settings page (preview functionality)
            if ($screen && ($screen->id == 'reorderre_page_settings' || 
                (isset($_GET['page']) && $_GET['page'] == 'settings'))) {
                wp_enqueue_script('reordere-admin-script', plugin_dir_url(__FILE__) . 'js/admin-script.js', array('jquery'), $this->version, true );
            }
        }

        /**
         * Enqueue Font Awesome for admin pages
         */
        public function enqueue_font_awesome() {
            // Use Font Awesome from CDN
            wp_enqueue_style(
                'font-awesome',
                'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css',
                array(),
                '6.4.0'
            );
        }

        /**
         * Register the menu and submenu pages for the plugin.
         *
         * @since    1.0.0
         */
        public function rorder_susbscription_register_menu_pages() {
            // Define the icon URL
            $icon_url = rorder_assets_url('logo/reordere-icon.svg');

            // Register top-level menu
            add_menu_page(
                __('ReordeRe', 'reordere-lite-subcriptions-for-woocommerce'),
                __('ReordeRe', 'reordere-lite-subcriptions-for-woocommerce'),
                'manage_options',
                $this->plugin_name,
                array($this, 'render_subscription_products_landing_page'),
                $icon_url,
                6
            );

            // Add a dashboard submenu with the same slug as the top-level to override auto-creation
            add_submenu_page(
                $this->plugin_name,
                __('Dashboard', 'reordere-lite-subcriptions-for-woocommerce'),
                __('Dashboard', 'reordere-lite-subcriptions-for-woocommerce'),
                'manage_options',
                $this->plugin_name,
                array($this, 'render_subscription_products_landing_page')
            );

            // Add other submenu pages
            add_submenu_page(
                $this->plugin_name,
                __('All Subscription', 'reordere-lite-subcriptions-for-woocommerce'),
                __('All Subscription', 'reordere-lite-subcriptions-for-woocommerce'),
                'manage_options',
                'subscriptions',
                array($this, 'render_subscription_products_page')
            );

            add_submenu_page(
                $this->plugin_name,
                __('Bulk Subscription', 'reordere-lite-subcriptions-for-woocommerce'),
                __('Bulk Subscription', 'reordere-lite-subcriptions-for-woocommerce'),
                'manage_options',
                'bulk subscription',
                array($this, 'render_all_subscription_products_page')
            );

            add_submenu_page(
                $this->plugin_name,
                __('Reports', 'reordere-lite-subcriptions-for-woocommerce'),
                __('Reports', 'reordere-lite-subcriptions-for-woocommerce'),
                'manage_options',
                'reports',
                array($this, 'render_reports_subscription_products_page')
            );

            add_submenu_page(
                $this->plugin_name,
                __('Settings', 'reordere-lite-subcriptions-for-woocommerce'),
                __('Settings', 'reordere-lite-subcriptions-for-woocommerce'),
                'manage_options',
                'settings',
                array($this, 'render_settings_subscription_products_page')
            );
        }

        /**
         * Display content for the subscription products page.
         *
         * @since    1.0.0
         */
        public function render_subscription_products_page() {   
            if(isset($_GET['action']) && $_GET['action'] == 'edit' && $_GET['page'] == 'subscriptions') {
                require_once plugin_dir_path(dirname(__FILE__)) . 'admin/partials/class-show-single-subscription-data.php';
            } else {
                include_once plugin_dir_path(dirname(__FILE__)) . 'admin/partials/class-show-subscription-data.php';

                $table = new Rorder_Subscription_List_Table();
                $table->prepare_items();
                ?>
                <div class="wrap subscription-list-wrap">
                    <div class="subscription-list-header">
                        <h1>All Subscriptions</h1>
                    </div>
                    <form method="post">
                        <?php
                        $table->display();
                        ?>
                    </form>
                </div>
                <?php
            }
        }

        /**
         * Display content for the settings subscription products page.
         *
         * @since    1.0.0
         */
        public function render_settings_subscription_products_page() {
            require_once plugin_dir_path(dirname(__FILE__)) . 'admin/partials/class-subscription-product-settings.php';
        }

        /** */

        

        /**
         * Display content for the all subscription products submenu page.
         *
         * @since    1.0.0
         */
        public function render_all_subscription_products_page() {
            // Content for the all subscription products page goes here.
            require_once plugin_dir_path(dirname(__FILE__)) . 'admin/partials/class-create-bulk-susbcription-products.php';
        }

        /**
         * Display content for the reports submenu page.
         *
         * @since    1.0.0
         */
        public function render_reports_subscription_products_page() {
            //get currency symbol
            $currency_symbol = get_woocommerce_currency_symbol();
            ?>
            <div class="wrap rorder-dashboard">
                <!-- Top Navigation Bar -->
                <div class="rorder-top-bar">
                    <div class="rorder-top-bar-left">
                        <h1 class="rorder-page-title">
                            <span class="dashicons dashicons-chart-bar"></span>
                            <?php esc_html_e('Analytics Dashboard', 'reordere-lite-subcriptions-for-woocommerce'); ?>
                        </h1>
                    </div>
                    <div class="rorder-top-bar-right">
                        <div class="rorder-date-selector">
                            <span class="dashicons dashicons-calendar-alt"></span>
                            <select class="rorder-select" id="timeframe-selector">
                                <option value="daily"><?php esc_html_e('Daily', 'reordere-lite-subcriptions-for-woocommerce'); ?></option>
                                <option value="weekly" selected><?php esc_html_e('Weekly', 'reordere-lite-subcriptions-for-woocommerce'); ?></option>
                                <option value="monthly"><?php esc_html_e('Monthly', 'reordere-lite-subcriptions-for-woocommerce'); ?></option>
                                <option value="yearly"><?php esc_html_e('Yearly', 'reordere-lite-subcriptions-for-woocommerce'); ?></option>
                            </select>
                        </div>
                    </div>
                </div>
                
                <!-- Dashboard Container -->
                <div class="rorder-dashboard-container">
                    <!-- Metrics Cards Grid -->
                    <div class="rorder-metrics-grid">
                        <div class="rorder-metric-card">
                            <div class="rorder-metric-icon rorder-icon-blue">
                                <span class="dashicons dashicons-money-alt"></span>
                            </div>
                            <div class="rorder-metric-content">
                                <div class="rorder-metric-label"><?php esc_html_e('Revenue', 'reordere-lite-subcriptions-for-woocommerce'); ?></div>
                                <div class="rorder-metric-value" id="revenue-value"><?php echo esc_html($currency_symbol); ?>0</div>
                                <div class="rorder-metric-sublabel"><?php esc_html_e('Total subscription revenue', 'reordere-lite-subcriptions-for-woocommerce'); ?></div>
                            </div>
                        </div>
                        
                        <div class="rorder-metric-card">
                            <div class="rorder-metric-icon rorder-icon-green">
                                <span class="dashicons dashicons-update"></span>
                            </div>
                            <div class="rorder-metric-content">
                                <div class="rorder-metric-label"><?php esc_html_e('Active Subscriptions', 'reordere-lite-subcriptions-for-woocommerce'); ?></div>
                                <div class="rorder-metric-value" id="active-count">0</div>
                                <div class="rorder-metric-sublabel"><?php esc_html_e('Currently active', 'reordere-lite-subcriptions-for-woocommerce'); ?></div>
                            </div>
                        </div>
                        
                        <div class="rorder-metric-card">
                            <div class="rorder-metric-icon rorder-icon-purple">
                                <span class="dashicons dashicons-groups"></span>
                            </div>
                            <div class="rorder-metric-content">
                                <div class="rorder-metric-label"><?php esc_html_e('Total Subscriptions', 'reordere-lite-subcriptions-for-woocommerce'); ?></div>
                                <div class="rorder-metric-value" id="total-count">0</div>
                                <div class="rorder-metric-sublabel"><?php esc_html_e('All time', 'reordere-lite-subcriptions-for-woocommerce'); ?></div>
                            </div>
                        </div>
                        
                        <div class="rorder-metric-card">
                            <div class="rorder-metric-icon rorder-icon-orange">
                                <span class="dashicons dashicons-dismiss"></span>
                            </div>
                            <div class="rorder-metric-content">
                                <div class="rorder-metric-label"><?php esc_html_e('Cancelled', 'reordere-lite-subcriptions-for-woocommerce'); ?></div>
                                <div class="rorder-metric-value" id="cancelled-count">0</div>
                                <div class="rorder-metric-sublabel"><?php esc_html_e('Inactive subscriptions', 'reordere-lite-subcriptions-for-woocommerce'); ?></div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Charts Row -->
                    <div class="rorder-charts-row">
                        <div class="rorder-chart-card">
                            <div class="rorder-chart-header">
                                <h2 class="rorder-chart-title"><?php esc_html_e('Revenue Trends', 'reordere-lite-subcriptions-for-woocommerce'); ?></h2>
                            </div>
                            <div class="rorder-chart-body">
                                <canvas id="reportChart"></canvas>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Additional Charts Row -->
                    <div class="rorder-charts-row">
                        <!-- Status Distribution Chart -->
                        <div class="rorder-chart-card rorder-chart-half">
                            <div class="rorder-chart-header">
                                <h2 class="rorder-chart-title"><?php esc_html_e('Subscription Status', 'reordere-lite-subcriptions-for-woocommerce'); ?></h2>
                            </div>
                            <div class="rorder-chart-body">
                                <canvas id="statusChart"></canvas>
                            </div>
                        </div>
                        
                        <!-- Payment Methods Chart -->
                        <div class="rorder-chart-card rorder-chart-half">
                            <div class="rorder-chart-header">
                                <h2 class="rorder-chart-title"><?php esc_html_e('Payment Methods', 'reordere-lite-subcriptions-for-woocommerce'); ?></h2>
                            </div>
                            <div class="rorder-chart-body">
                                <canvas id="paymentChart"></canvas>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Top Products Table -->
                    <div class="rorder-table-card">
                        <div class="rorder-table-header">
                            <h2 class="rorder-table-title"><?php esc_html_e('Top Subscription Products', 'reordere-lite-subcriptions-for-woocommerce'); ?></h2>
                        </div>
                        <div class="rorder-table-body">
                            <table class="rorder-data-table">
                                <thead>
                                    <tr>
                                        <th><?php esc_html_e('Product', 'reordere-lite-subcriptions-for-woocommerce'); ?></th>
                                        <th><?php esc_html_e('Subscribers', 'reordere-lite-subcriptions-for-woocommerce'); ?></th>
                                        <th><?php esc_html_e('Revenue', 'reordere-lite-subcriptions-for-woocommerce'); ?></th>
                                    </tr>
                                </thead>
                                <tbody id="top-products-table">
                                    <tr>
                                        <td colspan="3" style="text-align: center; padding: 20px;"><?php esc_html_e('Loading...', 'reordere-lite-subcriptions-for-woocommerce'); ?></td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
            <?php
        }

        /**
         * Enqueue scripts for the reports page.
         *
         * @since    1.0.0
         */
        public function enqueue_report_scripts() {
            $screen = get_current_screen();
            
            // Only load on our reports page
            if ($screen && ($screen->id == 'reorderre_page_reports' || 
                (isset($_GET['page']) && $_GET['page'] == 'reports'))) {
                // Enqueue modern dashboard CSS (already done in enqueue_styles)
                
                // Use Chart.js from assets folder
                wp_enqueue_script('chart-js', plugin_dir_url(dirname(__FILE__)) . 'assets/external-assets/js/chart.js', array('jquery'), '3.9.1', true);
                
                // Load jQuery UI components
                wp_enqueue_script('jquery-ui-datepicker');
                wp_enqueue_script('jquery-ui-sortable');
                
                // Load the custom reports script
                wp_enqueue_script('report-scripts', plugin_dir_url(__FILE__) . 'js/custom-reports.js', 
                    array('jquery', 'chart-js', 'jquery-ui-datepicker', 'jquery-ui-sortable'), 
                    $this->version,
                    true
                );
                
                // Localize script with data for JavaScript
                wp_localize_script('report-scripts', 'reportData', [
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('subscription_reports_nonce'),
                    'currency_symbol' => function_exists('get_woocommerce_currency_symbol') ? get_woocommerce_currency_symbol() : '$',
                ]);
            }
        }

        /**
         * Handle AJAX request to get chart data.
         *
         * @since    1.0.0
         */
        public function get_chart_data() {
            // Verify nonce
            if (!isset($_POST['_ajax_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])), 'subscription_reports_nonce')) {
                wp_send_json_error('Invalid nonce');
            }

            // Make sure the request is valid
            if (!isset($_POST['timeframe'])) {
                wp_send_json_error('Timeframe parameter is missing.');
            }
        
            $timeframe = sanitize_text_field(wp_unslash($_POST['timeframe']));
            
            // Fetch the data based on the timeframe
            $data = $this->fetch_subscription_sales_data($timeframe);
            
            // Fetch the total subscription sales (products sold) by timeframe
            $total_sales = $this->fetch_subscription_sales_by_timeframe();
            
            // Send the response back to the JavaScript front-end
            wp_send_json_success([
                'data' => $data,
                'sales' => $total_sales,
            ]);
        }
        
        /**
         * Fetch subscription sales data based on the timeframe.
         *
         * @since    1.0.0
         * @param    string    $timeframe    The timeframe for which to fetch data.
         * @return   array                   The subscription sales data.
         */
        private function fetch_subscription_sales_data($timeframe) {
            global $wpdb;
            $now = current_time('mysql');
            
            // Initialize the labels and data arrays
            $labels = [];
            $data = [];
        
            switch ($timeframe) {
                case 'daily':
                    // Get sales for the current week (Sunday to Saturday)
                    // We'll use the current date to find the start of the week (Sunday)
                    $start_of_week = gmdate('Y-m-d', strtotime('last sunday', strtotime($now)));
                    $end_of_week = gmdate('Y-m-d', strtotime('next saturday', strtotime($now)));
        
                    // Query for sales data for the current week (from Sunday to Saturday)
                    $sales_data = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT COUNT(*) as sales, DATE(created_at) as date 
                             FROM {$wpdb->prefix}subscription_details 
                             WHERE DATE(created_at) BETWEEN %s AND %s 
                             GROUP BY date",
                            $start_of_week, $end_of_week
                        )
                    );
        
                    // Static labels for days of the week (Mon-Sun)
                    $week_days = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"];
                    
                    // Initialize the data array with zeros for each day of the week
                    $data = array_fill(0, 7, 0); // Fill with zeros (one for each day from Mon-Sun)
        
                    // Map sales data to the days of the week
                    foreach ($sales_data as $item) {
                        // Get the day of the week (0=Sun, 1=Mon, ..., 6=Sat)
                        $day_of_week = gmdate('w', strtotime($item->date)); // Get numeric day of the week (0=Sun, 6=Sat)
                        if ($day_of_week == 0) {
                            // If the day is Sunday (0), we map it to the 6th index (last day)
                            $day_of_week = 6;
                        } else {
                            $day_of_week--; // Adjust to map Mon (1) to index 0, Tue (2) to index 1, etc.
                        }
        
                        // Update the data array with the sales for that day
                        $data[$day_of_week] = (int) $item->sales;
                    }
        
                    // Labels are static (Mon-Sun)
                    $labels = $week_days;
                    break;
        
                case 'weekly':
                    // Get sales per week for the current year
                    // We'll dynamically create the weeks for the current year
                    $start_of_year = gmdate('Y-01-01', strtotime($now));
                    $end_of_year = gmdate('Y-12-31', strtotime($now));
        
                    // Query for sales data for the current year (week by week)
                    $sales_data = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT COUNT(*) as sales, YEARWEEK(created_at, 1) as week 
                             FROM {$wpdb->prefix}subscription_details 
                             WHERE created_at BETWEEN %s AND %s 
                             GROUP BY week ORDER BY week",
                            $start_of_year, $end_of_year
                        )
                    );
        
                    // Create dynamic labels for each week
                    foreach ($sales_data as $item) {
                        $labels[] = 'Week ' . $item->week;
                        $data[] = (int) $item->sales;
                    }
                    break;
        
                case 'monthly':
                    // Get sales for the current year (Jan-Dec)
                    // Query for sales data for each month of the current year
                    $sales_data = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT COUNT(*) as sales, MONTH(created_at) as month 
                             FROM {$wpdb->prefix}subscription_details 
                             WHERE YEAR(created_at) = YEAR(%s) 
                             GROUP BY month",
                            $now
                        )
                    );
        
                    // Static labels for months (Jan - Dec)
                    $month_names = [
                        "Jan", "Feb", "Mar", "Apr", "May", "Jun",
                        "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
                    ];
        
                    // Initialize the data array with zeros for each month
                    $data = array_fill(0, 12, 0); // Fill with zeros for each month from Jan-Dec
        
                    // Map sales data to the months
                    foreach ($sales_data as $item) {
                        $month_index = $item->month - 1; // Convert 1-12 to 0-11 index
                        $data[$month_index] = (int) $item->sales;
                    }
        
                    // Labels are static (Jan - Dec)
                    $labels = $month_names;
                    break;
        
                case 'yearly':
                    // Get sales for the current year
                    // Query for sales data for the current year
                    $sales_data = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT COUNT(*) as sales, YEAR(created_at) as year 
                             FROM {$wpdb->prefix}subscription_details 
                             WHERE YEAR(created_at) = YEAR(%s) 
                             GROUP BY year",
                            $now
                        )
                    );
        
                    // Create dynamic labels for years (current year onward)
                    $current_year = gmdate('Y');
                    for ($i = 0; $i < 5; $i++) {
                        $labels[] = (string)($current_year + $i); // Labels for current and next 4 years
                        $data[] = 0; // Initialize data to 0 (no sales yet)
                    }
        
                    // Map sales data to the years
                    foreach ($sales_data as $item) {
                        $year_index = $item->year - $current_year;
                        if ($year_index >= 0 && $year_index < count($data)) {
                            $data[$year_index] = (int) $item->sales;
                        }
                    }
                    break;
        
                default:
                    $sales_data = [];
            }
            
            return [
                'labels' => $labels,
                'data' => $data,
            ];
        }

        /**
         * Fetch subscription sales by timeframe.
         *
         * @since    1.0.0
         * @return   array    The subscription sales by timeframe.
         */
        private function fetch_subscription_sales_by_timeframe() {
            global $wpdb;
            $now = current_time('mysql');
            
            // Query to fetch the total revenue for each timeframe by summing the price
            $daily_revenue = wp_cache_get('daily_revenue');
            if ($daily_revenue === false) {
                $daily_revenue = $wpdb->get_var($wpdb->prepare("SELECT SUM(subscription_price) FROM {$wpdb->prefix}subscription_details WHERE DATE(created_at) = DATE(%s)", $now));
                wp_cache_set('daily_revenue', $daily_revenue);
            }

            $weekly_revenue = wp_cache_get('weekly_revenue');
            if ($weekly_revenue === false) {
                $weekly_revenue = $wpdb->get_var($wpdb->prepare("SELECT SUM(subscription_price) FROM {$wpdb->prefix}subscription_details WHERE YEARWEEK(created_at, 1) = YEARWEEK(%s, 1)", $now));
                wp_cache_set('weekly_revenue', $weekly_revenue);
            }

            $monthly_revenue = wp_cache_get('monthly_revenue');
            if ($monthly_revenue === false) {
                $monthly_revenue = $wpdb->get_var($wpdb->prepare("SELECT SUM(subscription_price) FROM {$wpdb->prefix}subscription_details WHERE MONTH(created_at) = MONTH(%s)", $now));
                wp_cache_set('monthly_revenue', $monthly_revenue);
            }

            $yearly_revenue = wp_cache_get('yearly_revenue');
            if ($yearly_revenue === false) {
                $yearly_revenue = $wpdb->get_var($wpdb->prepare("SELECT SUM(subscription_price) FROM {$wpdb->prefix}subscription_details WHERE YEAR(created_at) = YEAR(%s)", $now));
                wp_cache_set('yearly_revenue', $yearly_revenue);
            }
            
            // Return the total revenue for each timeframe
            return [
                'daily' => (float) $daily_revenue,
                'weekly' => (float) $weekly_revenue,
                'monthly' => (float) $monthly_revenue,
                'yearly' => (float) $yearly_revenue,
            ];
        }

        /**
         * Display content for the subscription products landing page.
         *
         * @since    1.0.0
         */
        public function render_subscription_products_landing_page() {

            
            $square_license = get_option('rorder_license');
            $square_email = get_option('rorder_email');
            $square_license_key = get_option('rorder_license_key');
        
            ?>
        
            <div class="plugin-dashboard">
                <header class="dashboard-header">
                <img src="<?php echo esc_url(rorder_assets_url('logo/reordere-full-width.svg')); ?>" alt="Reordere Logo" style="width: 200px; height: auto; margin-top: 10px;">   
            

                </header>
        
                <section class="features-grid">
                    <div class="feature-card">
                        <i class="fas fa-repeat feature-icon"></i>
                        <h2>Product Subscription</h2>
                        <p>Create a subscription plan for already existing woocommerce products</p>
                    </div>
                    <div class="feature-card">
                        <i class="fas fa-solid fa-list feature-icon"></i>
                        <h2>Bulk Subscription</h2>
                        <p>Create master subscription and apply to all desired woocommerce products</p>
                    </div>
                    <div class="feature-card">
                        <i class="fas fa-solid fa-money-check-dollar feature-icon"></i>
                        <h2>Dynamic Pricing</h2>
                        <p>Offer your customers a percentage OR flat amount discount for subscription.</p>
                    </div>
                    <div class="feature-card">
                        <i class="fas fa-solid fa-circle-pause feature-icon"></i>
                        <h2>Resume & Pause</h2>
                        <p>Offer your customers the ability to pause their subscription as desired.</p>
                    </div>
                    <div class="feature-card">
                        <i class="fas fa-solid fa-list-check feature-icon"></i>
                        <h2>Detailed Reports</h2>
                        <p>Get detailed reports about your subscription status and payment status.</p>
                    </div>
                    <div class="feature-card">
                        <i class="fa-brands fa-cc-stripe feature-icon" aria-hidden="true"></i>
                        <h2>Stripe Payments<span style="color: red;"><?php esc_html_e('(Pro Feature)', 'reordere-lite-subcriptions-for-woocommerce'); ?></span></h2>
                        <p>Auto-charge your customers using stripe payment gateway.</p>
                    </div>
                    <div class="feature-card">
                        <i class="fa-brands fa-cc-paypal feature-icon"></i>
                        <h2>Paypal Payments</h2>
                        <p>Auto-Charge your customer using Paypal Payment Gateway.</p>
                    </div>
                    <div class="feature-card">
                        <i class="fas fa-solid fa-credit-card feature-icon"></i>
                        <h2>Square Payments<span style="color: red;"><?php esc_html_e('(Pro Feature)', 'reordere-lite-subcriptions-for-woocommerce'); ?></span></h2>
                        <p>Auto-charge your customers using Square Payment Gateway.</p>
                    </div>
                     <div class="feature-card">
                        <i class="fas fa-solid fa-credit-card feature-icon"></i>
                        <h2>Automatic PaymentRetrive<span style="color: red;"><?php esc_html_e('(Pro Feature)', 'reordere-lite-subcriptions-for-woocommerce'); ?></span></h2>
                        <p>It means automatically collecting payments without needing to manually process each one.</p>
                    </div>
                </section>
        
        
                <div class="notices-section">
                    <h2>Notices</h2>
                    <div class="notice-card">
                        <i class="fas fa-info-circle notice-icon"></i>
                        <p>For any queries kindly reach contact us at - <b>support@techspawn.com</b></p>
                    </div>
                </div>
            </div>
            </div>
            <?php
        
        }

        /**
         * Save subscription product fields.
         *
         * @since    1.0.0
         * @param    int    $post_id    The ID of the post.
         */
        function save_subscription_product_fields($post_id) {
            // Verify nonce
            if (!isset($_POST['_subscription_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_subscription_nonce'])), 'save_subscription_fields')) {
                return;
            }

            // Check the enable_subscription checkbox
            if (isset($_POST['enable_subscription'])) {
                update_post_meta($post_id, '_enable_subscription', 'yes');
            } else {
                update_post_meta($post_id, '_enable_subscription', 'no');
            }

            // Check the flat rate and percentage discount radio button
            if (isset($_POST['subscription_discount_type'])) {
                update_post_meta($post_id, '_subscription_discount_type', sanitize_text_field(wp_unslash($_POST['subscription_discount_type'])));
            }

            // Check the _main_subscription_title
            if (isset($_POST['main_subscription_title'])) {
                update_post_meta($post_id, '_main_subscription_title', sanitize_text_field(wp_unslash($_POST['main_subscription_title'])));
            }
            
            // Check the subscription_product_row checkbox
            $save_individual = isset($_POST['subscription_product_row']) ? 'yes' : 'no';
            update_post_meta($post_id, '_subscription_product_row', $save_individual);
        
            // Get the postmeta for enable_subscription
            $enable_subscription = get_post_meta($post_id, '_enable_subscription', true);
        
            // If enable_subscription is not 'yes', return early
            if ($enable_subscription !== 'yes') {
                return;
            }
        
            // Check if the subscription fields are submitted
            if (isset($_POST['subscription_fields']) && is_array($_POST['subscription_fields'])) {
                $subscription_fields = wp_unslash($_POST['subscription_fields']);
        
                // Sanitize the fields' data before saving
                $sanitized_fields = array();
                foreach ($subscription_fields as $key => $field_data) {
                    $sanitized_fields[$key] = array_map('sanitize_text_field', $field_data);
                }
        
                // If 'subscription_product_row' is checked, save fields individually
                if ($save_individual === 'yes') {
                    foreach ($sanitized_fields as $index => $field_data) {
                        foreach ($field_data as $field_key => $field_value) {
                            update_post_meta($post_id, "_subscription_field_{$index}_{$field_key}", $field_value);
                        }
                    }
                } else {
                    // Save the fields as a single array in postmeta (normal behavior)
                    update_post_meta($post_id, '_subscription_fields', $sanitized_fields);
                }
            }
        }

        /**
         * Save subscription fields for variations.
         *
         * @since    1.0.0
         * @param    int    $variation_id    The ID of the variation.
         */
        function save_subscription_fields_for_variations($variation_id) {
            // Verify nonce
            if (!isset($_POST['_subscription_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_subscription_nonce'])), 'save_subscription_fields')) {
                return;
            }

            // Enable Subscription
            if (isset($_POST['enable_subscription'][$variation_id])) {
                update_post_meta($variation_id, '_enable_subscription', 'yes');
            } else {
                update_post_meta($variation_id, '_enable_subscription', 'no');
            }
        
            // Subscription Discount Type
            if (isset($_POST['subscription_discount_type'][$variation_id])) {
                update_post_meta($variation_id, '_subscription_discount_type', sanitize_text_field(wp_unslash($_POST['subscription_discount_type'][$variation_id])));
            }
        
            // Main Subscription Title
            if (isset($_POST['main_subscription_title'][$variation_id])) {
                update_post_meta($variation_id, '_main_subscription_title', sanitize_text_field(wp_unslash($_POST['main_subscription_title'][$variation_id])));
            }
        
            // Check if subscription fields are submitted
            if (isset($_POST['subscription_fields'][$variation_id]) && is_array($_POST['subscription_fields'][$variation_id])) {
                $subscription_fields = wp_unslash($_POST['subscription_fields'][$variation_id]);
        
                // Sanitize the fields before saving
                $sanitized_fields = [];
                foreach ($subscription_fields as $key => $field_data) {
                    $sanitized_fields[$key] = array_map('sanitize_text_field', $field_data);
                }
        
                // Save all fields for the variation
                update_post_meta($variation_id, '_subscription_fields', $sanitized_fields);

            } else {
                // If no fields are set, remove the meta
                delete_post_meta($variation_id, '_subscription_fields');
            }
        }
        
        /**
         * AJAX handler to get product image
         *
         * @since    1.0.0
         */
        public function get_product_image() {
            if (!isset($_POST['product_id'])) {
                wp_send_json_error(array('message' => 'Product ID is required'));
            }

            $product_id = intval($_POST['product_id']);
            $product = wc_get_product($product_id);

            if (!$product) {
                wp_send_json_error(array('message' => 'Product not found'));
            }

            $image_url = wp_get_attachment_image_url($product->get_image_id(), 'full');

            wp_send_json_success(array('image_url' => $image_url ?: ''));
        }
        
        /**
         * AJAX handler to get subscription metrics
         *
         * @since    1.0.0
         */
        public function get_subscription_metrics_ajax() {
            // Verify nonce
            if (!isset($_POST['_ajax_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])), 'subscription_reports_nonce')) {
                wp_send_json_error('Invalid nonce');
            }
            
            // Get metrics
            $metrics = rorder_get_subscription_metrics();
            
            wp_send_json_success($metrics);
        }
        
        /**
         * AJAX handler for getting subscription report data (status distribution)
         *
         * @since    1.0.0
         */
        public function get_subscription_report_data_ajax() {
            // Verify nonce
            if (!isset($_POST['_ajax_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])), 'subscription_reports_nonce')) {
                wp_send_json_error('Invalid nonce');
            }
            
            // Get status distribution
            $data = rorder_get_subscription_report_data();
            
            wp_send_json_success($data);
        }
        
        /**
         * AJAX handler for getting payment methods distribution
         *
         * @since    1.0.0
         */
        public function get_payment_methods_ajax() {
            // Verify nonce
            if (!isset($_POST['_ajax_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])), 'subscription_reports_nonce')) {
                wp_send_json_error('Invalid nonce');
            }
            
            // Get payment methods data
            $data = rorder_get_payment_methods_distribution();
            
            wp_send_json_success($data);
        }
        
        /**
         * AJAX handler for getting top subscription products
         *
         * @since    1.0.0
         */
        public function get_top_products_ajax() {
            // Verify nonce
            if (!isset($_POST['_ajax_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])), 'subscription_reports_nonce')) {
                wp_send_json_error('Invalid nonce');
            }
            
            // Get top products
            $data = rorder_get_top_subscription_products();
            
            wp_send_json_success($data);
        }
        
    }
    // Initialize the admin functionality
    $initialize_admin = new Rorder_Products_Admin('reordere-lite-subcriptions-for-woocommerce', '1.0.0');
}
