<?php
/**
 * Plugin Name:       Remove Background AI
 * Description:       Remove Background Instantly from Images using AI - www.rembg.com
 * Version:           1.0.0
 * Requires at least: 6.7
 * Requires PHP:      7.4
 * Author:            Arken AI LLC
 * License:           GPL-2.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       remove-background-ai
 *
 * @package CreateBlock
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
/**
 * Registers the block using a `blocks-manifest.php` file, which improves the performance of block type registration.
 * Behind the scenes, it also registers all assets so they can be enqueued
 * through the block editor in the corresponding context.
 *
 * @see https://make.wordpress.org/core/2025/03/13/more-efficient-block-type-registration-in-6-8/
 * @see https://make.wordpress.org/core/2024/10/17/new-block-type-registration-apis-to-improve-performance-in-wordpress-6-7/
 */
function rembgai_register_blocks() {
	/**
	 * Registers the block(s) metadata from the `blocks-manifest.php` and registers the block type(s)
	 * based on the registered block metadata.
	 * Added in WordPress 6.8 to simplify the block metadata registration process added in WordPress 6.7.
	 *
	 * @see https://make.wordpress.org/core/2025/03/13/more-efficient-block-type-registration-in-6-8/
	 */
	if ( function_exists( 'wp_register_block_types_from_metadata_collection' ) ) {
		wp_register_block_types_from_metadata_collection( __DIR__ . '/build', __DIR__ . '/build/blocks-manifest.php' );
		return;
	}

	/**
	 * Registers the block(s) metadata from the `blocks-manifest.php` file.
	 * Added to WordPress 6.7 to improve the performance of block type registration.
	 *
	 * @see https://make.wordpress.org/core/2024/10/17/new-block-type-registration-apis-to-improve-performance-in-wordpress-6-7/
	 */
	if ( function_exists( 'wp_register_block_metadata_collection' ) ) {
		wp_register_block_metadata_collection( __DIR__ . '/build', __DIR__ . '/build/blocks-manifest.php' );
	}
	/**
	 * Registers the block type(s) in the `blocks-manifest.php` file.
	 *
	 * @see https://developer.wordpress.org/reference/functions/register_block_type/
	 */
	$manifest_data = require __DIR__ . '/build/blocks-manifest.php';
	foreach ( array_keys( $manifest_data ) as $block_type ) {
		register_block_type( __DIR__ . "/build/{$block_type}" );
	}
}
add_action( 'init', 'rembgai_register_blocks' );

/**
 * Add admin menu for plugin settings
 */
function rembgai_add_admin_menu() {
	// Add main menu item in left sidebar
		add_menu_page(
		'Remove Background',              // Page title
		'Remove Background',              // Menu title
		'manage_options',                 // Capability
		'rembgai',                  // Menu slug
		'rembgai_main_page',        // Function
		'dashicons-format-image',        // Icon
		30                                // Position
	);

	// Add submenu for settings
	add_submenu_page(
		'rembgai',                  // Parent slug
		'Remove Background Settings',    // Page title
		'Settings',                      // Menu title
		'manage_options',                // Capability
		'rembgai-settings',        // Menu slug
		'rembgai_admin_page'        // Function
	);
}
add_action( 'admin_menu', 'rembgai_add_admin_menu' );

// No legacy redirects needed (plugin not previously published)

/**
 * Register plugin settings
 */
function rembgai_register_settings() {
	register_setting(
		'rembgai_settings',
		'rembgai_api_key',
		array(
			'type'              => 'string',
			'sanitize_callback' => 'rembgai_sanitize_api_key',
			'default'           => '',
		)
	);
}
add_action( 'admin_init', 'rembgai_register_settings' );

/**
 * Sanitize the API key option before saving.
 *
 * Strips tags, trims whitespace, and restricts to common API key characters.
 * Adjust the allowed character set if RemBG changes their key format.
 *
 * @param mixed $raw_value Raw submitted value.
 * @return string Sanitized API key.
 */
function rembgai_sanitize_api_key( $raw_value ) {
	$raw_string = is_string( $raw_value ) ? $raw_value : '';
	$trimmed    = trim( wp_strip_all_tags( $raw_string ) );
	// Allow letters, numbers, dash, underscore, and dot.
	$sanitized  = preg_replace( '/[^A-Za-z0-9._-]/', '', $trimmed );
	// Optionally enforce a reasonable max length to avoid abuse.
	return substr( $sanitized, 0, 128 );
}

/**
 * Main admin page
 */
function rembgai_main_page() {
	$rembg_logo_url = plugin_dir_url( __FILE__ ) . 'assets/rembg-branding-1024.png';
	?>
	<div class="wrap">
		<div style="display: flex; align-items: center; margin-bottom: 20px;">
			<img src="<?php echo esc_url( $rembg_logo_url ); ?>" alt="RemBG" style="height: 60px; margin-right: 15px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
			<h1 style="margin: 0;">Remove Background - RemBG.com</h1>
		</div>
		<div style="background: #fff; padding: 30px; border: 1px solid #ccd0d4; border-radius: 12px; margin-top: 20px; box-shadow: 0 4px 12px rgba(0,0,0,0.05);">
			<h2>Hello! 👋</h2>
			<p>Welcome to the RemBG.com WP plugin admin area.</p>
			<p>This plugin allows you to remove backgrounds from images automatically using advanced AI technology.</p>

			<div style="margin-top: 30px;">
				<h3>Quick Start:</h3>
				<ol>
					<li>Go to <strong>Settings</strong> to configure your API key</li>
					<li>Create a new post or page</li>
					<li>Add the "Remove Background" block</li>
					<li>Upload an image and watch the magic happen!</li>
				</ol>
			</div>

			<div style="margin-top: 30px; padding: 20px; background: linear-gradient(135deg, #f0f8ff 0%, #e6f3ff 100%); border-left: 4px solid #0073aa; border-radius: 8px;">
				<h4>Need Help?</h4>
			<p>Check out the <a href="<?php echo esc_url( admin_url( 'admin.php?page=rembgai-settings' ) ); ?>">Settings page</a> to configure your www.rembg.com API key.</p>
			</div>

			<div style="margin-top: 30px; padding: 20px; background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%); border-radius: 8px; text-align: center; border: 1px solid #e1e5e9;">
				<img src="<?php echo esc_url( $rembg_logo_url ); ?>" alt="Powered by RemBG" style="height: 40px; opacity: 0.8; margin-bottom: 10px;">
				<p style="margin: 0; font-size: 14px; color: #64748b;">Powered by <a href="<?php echo esc_url( 'https://www.rembg.com' ); ?>" target="_blank" style="color: #0073aa; text-decoration: none; font-weight: 600;">www.RemBG.com</a></p>
				<p style="margin: 10px 0 0 0; font-size: 12px; color: #94a3b8;">Professional AI-powered background removal service</p>
			</div>
		</div>
	</div>
	<?php
}

/**
 * Admin settings page
 */
function rembgai_admin_page() {
	$rembg_logo_url = plugin_dir_url( __FILE__ ) . 'assets/rembg-branding-1024.png';
	?>
	<div class="wrap">
		<div style="display: flex; align-items: center; margin-bottom: 20px;">
			<img src="<?php echo esc_url( $rembg_logo_url ); ?>" alt="RemBG" style="height: 50px; margin-right: 15px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
			<h1 style="margin: 0;">Remove Background Settings</h1>
		</div>

		<div style="background: #fff; padding: 30px; border: 1px solid #ccd0d4; border-radius: 12px; margin-bottom: 20px; box-shadow: 0 4px 12px rgba(0,0,0,0.05);">
			<form method="post" action="options.php">
				<?php
				settings_fields( 'rembgai_settings' );
				do_settings_sections( 'rembgai_settings' );
				$api_key = get_option( 'rembgai_api_key', '' );
				?>
				<table class="form-table">
					<tr>
						<th scope="row">
							<label for="rembgai_api_key">API Key</label>
						</th>
						<td>
							<div style="position: relative; display: inline-block;">
								<input
									type="password"
								id="rembgai_api_key"
								name="rembgai_api_key"
									value="<?php echo esc_attr( $api_key ); ?>"
									class="regular-text"
									placeholder="Enter your www.rembg.com API key"
									style="padding-right: 45px;"
								/>
								<button
									type="button"
									id="toggle_api_key_visibility"
									class="button button-secondary"
									style="position: absolute; right: 0px; top: 0px; height: calc(100% - 4px); padding: 3px 8px; border-radius: 3px; background: #f7f7f7; border: 1px solid #ccd0d4;"
									title="Show/Hide API Key"
								>
									<span class="dashicons dashicons-visibility" style="font-size: 16px; line-height: 1.2;"></span>
								</button>
							</div>
							<p class="description">
								Get your free API key from <a href="<?php echo esc_url( 'https://www.rembg.com/en/api-usage' ); ?>" target="_blank">www.RemBG.com</a>
							</p>
						</td>
					</tr>
				</table>
				<?php submit_button(); ?>
			</form>
		</div>

		<div style="padding: 20px; background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%); border-radius: 8px; text-align: center; border: 1px solid #e1e5e9;">
			<img src="<?php echo esc_url( $rembg_logo_url ); ?>" alt="Powered by RemBG" style="height: 40px; opacity: 0.8; margin-bottom: 10px;">
			<p style="margin: 0; font-size: 14px; color: #64748b;">Powered by <a href="https://www.rembg.com" target="_blank" style="color: #0073aa; text-decoration: none; font-weight: 600;">www.RemBG.com</a></p>
			<p style="margin: 10px 0 0 0; font-size: 12px; color: #94a3b8;">Professional AI-powered background removal service</p>
		</div>
	</div>
	<?php
}

/**
 * Enqueue admin scripts
 */
function rembgai_admin_scripts() {
	wp_enqueue_script( 'rembgai-admin', plugin_dir_url( __FILE__ ) . 'build/admin.js', array(), '1.0.0', true );
	wp_localize_script( 'rembgai-admin', 'rembgai', array(
		'apiKey' => get_option( 'rembgai_api_key', '' ),
		'apiUrl' => 'https://api.rembg.com/rmbg'
	) );
}
add_action( 'admin_enqueue_scripts', 'rembgai_admin_scripts' );

/**
 * Enqueue block editor scripts with API key
 */
function rembgai_enqueue_block_editor_assets() {
	// Provide data to the block editor via our own prefixed handle
	$handle = 'rembgai-block-data';
	wp_register_script( $handle, '', array(), '1.0.0', true );
	wp_enqueue_script( $handle );
	wp_add_inline_script(
		$handle,
		'window.rembgai = ' . wp_json_encode( array(
			'apiKey'    => get_option( 'rembgai_api_key', '' ),
			'apiUrl'    => 'https://api.rembg.com/rmbg',
			'pluginUrl' => plugin_dir_url( __FILE__ ),
		) ) . ';',
		'before'
	);
}
add_action( 'enqueue_block_editor_assets', 'rembgai_enqueue_block_editor_assets' );
