<?php

namespace ForgeSmith\Blocks;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}// Exit if accessed directly
use ForgeSmith\FoundryBlock;
use WP_Query;

class RelatedPosts extends FoundryBlock {
	public $bemClass = 'fndry-related-posts';
	public $query;

	/*
	 * Show related posts.
	 * Limit, specific taxonomies to filter by
	 * */
	public function __construct( $attributes, $content = '', $wp_block = '' ) {
		parent::__construct( $attributes, $content, $wp_block );

		$this->query = $this->constructBlockQuery();
	}

	private function constructBlockQuery() {
		/**
		 * Filter to override this block's generated query.
		 *
		 * @param null|WP_Query $query null query replacement arg. Default null to continue retrieving the result.
		 * @param RelatedPosts  $this  The RelatedPosts class. Inherits methods from FoundryBlock.
		 *
		 * @since 1.5.6
		 */
		$query = apply_filters( $this->name . '/construct-block-query', null, $this );
		if ( $query ) {
			return $this->query = $query;
		}

		$query_args = wp_parse_args( $this->getAttribute( 'query' ), [
			'perPage'  => 3,
			'offset'   => 0,
			'postType' => 'any',
			'order'    => 'asc',
			'orderBy'  => 'date',
			'exclude'  => [],
			'sticky'   => '',
			'inherit'  => true,
		] );

		$limit = $query_args['perPage'];

		// allow user to specify specific taxonomies to be related by, or just all of them.
		$allTerms = [];

		$registeredTaxes = get_object_taxonomies( get_post_type() );
		$customTaxes     = $this->getAttribute( 'taxonomyFilters' );

		$taxes = [];

		if ( ! empty( $customTaxes ) ) {
			// ensure we're only looking for valid taxes, in case of editor screwup
			$taxes = array_filter( $customTaxes, function ( $el ) use ( $registeredTaxes ) {
				return in_array( $el, $registeredTaxes );
			} );
		}

		if ( empty( $taxes ) ) {
			// if A) no taxes selected or B) taxes selected are all invalid
			$taxes = $registeredTaxes;
		}

		foreach ( $taxes as $tax ) {
			if ( $tax === 'tag' || $tax === 'category' ) {
				// use wp_get_post_terms() to avoid rendering all of the terms as objects:
				$terms = $tax === 'tag'
					? get_the_tags( null )
					: wp_get_post_terms(
						get_the_ID(),
						$tax,
						[ 'fields' => 'ids' ]
					);

				$allTerms[ $tax ] = is_array( $terms ) ? $terms : [];
			} else {
				$taxDef = get_taxonomy( $tax );
				// If $taxDef->query_var is not defined (as in author), do not set it as a taxonomy named 0
				if ( $taxDef && $taxDef->query_var ) {
					// use wp_get_post_terms() to avoid rendering all terms as objects:
					$terms                          = wp_get_post_terms( get_the_ID(), $tax, [ 'fields' => 'ids' ] );
					$allTerms[ $taxDef->query_var ] = is_array( $terms ) ? $terms : [];
				}
			}
		}

		// filter out empty tax keys
		$allTerms = array_filter( $allTerms );

		// Optimization for large content:  Often the $allTerms array is empty (e.g. when there is no post_tag)
		// this causes a WP_Query of ALL posts and a slow usort, which ends up just doing a posts query by date desc.

		$allTermCount = count( $allTerms );

		// if $allTerms is empty at this point, then there are no taxonomy terms to match on for this post / block combo.
		// this could just mean the current post has no terms in this specific taxonomy.
		// Instead, if we know there are no terms we can just do a simple default WP_Query with a posts_per_page of $limit,
		// because that was the end result of rendering the posts and then in the end sorting each row by date because there are no terms

		// if query attr's post type isn't set, just use current post type
		// OOP - if postType arg is set, and is either a string, or if it's an array and the array isn't empty, use it.
		// otherwise, we can assume that the user has set the value to "any"
		// (backwards compat - going forward, this will actually give us the value "any")
		$postType = ! empty( $query_args['postType'] ) &&
		            ( ! is_array( $query_args['postType'] ) ||
		              ( ! empty( array_filter( $query_args['postType'] ) ) )
		            ) ? $query_args['postType'] : 'any';

		// prep our $args;
		$args = [];

		// much faster and same end result (posts sorted by date)
		if ( $allTermCount === 0 ) {
			$args = [
				'post_type'           => $postType,
				'posts_per_page'      => $limit,
				'post__not_in'        => [ get_the_ID() ],
				'ignore_sticky_posts' => true,
			];
		} else {
			//  Otherwise, limit the query to all posts that have matching tax terms:
			// Use a SQL query to determine the posts with the most matching terms,
			// then limit the $query by those IDs only, to avoid rendering so many posts (really slow with thousands of posts and terms.)
			global $wpdb;

			// build a list of term_ids from the combined taxonomies applying to this post, this will be injected into the SQL query.
			$term_ids = [];
			foreach ( $allTerms as $tax => $terms ) {
				$term_ids += $terms;
			}

			// Prepare a SQL query to find related posts (with matching term_id's), and also sort the results based on <// of terms in common>, <post_date> DESC
			// the output of this query is the same list of posts as the old method ended up with by rendering all matching posts via WP_Query then sorting
			// manually afterwards in usort, but it's limited to just $limit posts.

			$post_type_clause = $postType !== 'any' ? "AND p.post_type IN (" . implode( ',',
					array_map( function ( $el ) {
						return "'${el}'";
					}, (array) $postType ) ) . ")" : '';

			$related_posts_query = $wpdb->prepare(
				"SELECT DISTINCT p.ID, p.post_date, p.post_type, COUNT(*) AS common_term_count
				FROM {$wpdb->posts} AS p
				INNER JOIN {$wpdb->term_relationships} AS tr ON p.ID = tr.object_id
				INNER JOIN {$wpdb->term_taxonomy} AS tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
				WHERE p.post_status = 'publish'
				{$post_type_clause}
				AND p.ID != %d
				AND tt.term_id IN (" . implode( ',', array_fill( 0, count( $term_ids ), '%d' ) ) . ")
				GROUP BY p.ID, p.post_date
				ORDER BY common_term_count DESC, p.post_date DESC",
				get_the_ID(),
				...$term_ids
			);

			// omit LIMIT clause if infinite
			if ( $limit > 0 ) {
				$related_posts_query .= ' LIMIT 0, ' . (int) $limit;
			}

			$related_posts    = $wpdb->get_results( $related_posts_query );
			$related_post_ids = array_map( function ( $post ) {
				return $post->ID;
			}, $related_posts );

			if ( ! empty( $related_post_ids ) ) {
				// We do have sorted matches come from the SQL query.
				// These are post ID's that are already pre-sorted correctly.
				// Fetch only the Post ID's from the SQL related posts query
				$args = [
					'post_type'           => $postType,
					'posts_per_page'      => $limit,
					'post__not_in'        => [ get_the_ID() ],
					'post__in'            => $related_post_ids,
					'ignore_sticky_posts' => true,
				];
			} elseif ( $this->getAttribute( 'backfillNoResults' ) ) {
				// if we had no matching related posts with common taxo terms (can happen for example when there are no post_tags),
				// then we fall back again to a standard WP_Query based on post_date, for $limit rows only.
				// Only do this if we explicitly want to - some folk just want to see a no results.
				$args = [
					'post_type'           => $postType,
					'posts_per_page'      => $limit,
					'post__not_in'        => [ get_the_ID() ],
					'ignore_sticky_posts' => true,
				];
			}
		}

		return new WP_Query( $args );
	}
}
