<?php

namespace ForgeSmith\Blocks;
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
use ForgeSmith\FoundryBlock;
use WP_Query;

class QueryBlock extends FoundryBlock {
	public $bemClass = 'fndry-query';
	public $query;

	public function __construct( $attributes, $content = '', $wp_block = '' ) {
		parent::__construct( $attributes, $content, $wp_block );

		if ( ! $this->query ) {
			$this->query = $this->constructBlockQuery();
		}
	}

	private function constructBlockQuery() {
		/**
		 * Filter to override this block's generated query.
		 *
		 * @param null|WP_Query $query null query replacement arg. Default null to continue retrieving the result.
		 * @param QueryBlock $this The QueryBlock class. Inherits methods from FoundryBlock.
		 *
		 * @since 1.5.6
		 */
		$query = apply_filters( $this->name . '/construct-block-query', null, $this );
		if ( $query ) {
			return $this->query = $query;
		}

		$queryId  = $this->getAttribute( 'queryId' ) ?? null;
		$page_key = isset( $queryId ) ? 'query-' . $queryId . '-page' : 'query-page';
		$page     = empty( fndry_get_query_var( $page_key ) ) ? 1 : (int) fndry_get_query_var( $page_key );

		$query = [
			'post_type'    => 'any',
			'order'        => 'DESC',
			'orderby'      => 'date',
			'post__not_in' => [],
		];

		$queryAttr = $this->attribute( 'query' );

		if ( $queryAttr ) {
			if ( ! empty( $queryAttr['postType'] ) ) {
				$post_type_param = $queryAttr['postType'];
				if ( is_post_type_viewable( $post_type_param ) ) {
					$query['post_type'] = $post_type_param;
				} else {
					// if it's not a viewable post type, why bother showing it to them?
					return null;
				}
			}
			if ( ! empty( $queryAttr['sticky'] ) ) {
				$sticky = get_option( 'sticky_posts' );
				if ( 'only' === $queryAttr['sticky'] ) {
					/*
					 * Passing an empty array to post__in will return have_posts() as true (and all posts will be returned).
					 * Logic should be used before hand to determine if WP_Query should be used in the event that the array
					 * being passed to post__in is empty.
					 *
					 * @see https://core.trac.wordpress.org/ticket/28099
					 */
					$query['post__in']            = ! empty( $sticky ) ? $sticky : [ 0 ];
					$query['ignore_sticky_posts'] = 1;
				} else {
					$query['post__not_in'] = array_merge( $query['post__not_in'], $sticky );
				}
			}
			if ( ! empty( $queryAttr['exclude'] ) ) {
				$excluded_post_ids     = array_map( 'intval', $queryAttr['exclude'] );
				$excluded_post_ids     = array_filter( $excluded_post_ids );
				$query['post__not_in'] = array_merge( $query['post__not_in'], $excluded_post_ids );
			}
			if (
				isset( $queryAttr['perPage'] ) &&
				is_numeric( $queryAttr['perPage'] )
			) {
				$per_page = (int) $queryAttr['perPage'];
				$offset   = 0;

				if (
					isset( $queryAttr['offset'] ) &&
					is_numeric( $queryAttr['offset'] )
				) {
					$offset = absint( $queryAttr['offset'] );
				}

				$query['offset']         = ( $per_page * ( $page - 1 ) ) + $offset;
				$query['posts_per_page'] = $per_page;
			}
			// Migrate `categoryIds` and `tagIds` to `tax_query` for backwards compatibility.
			if ( ! empty( $queryAttr['categoryIds'] ) || ! empty( $queryAttr['tagIds'] ) ) {
				$tax_query = [];
				if ( ! empty( $queryAttr['categoryIds'] ) ) {
					$tax_query[] = [
						'taxonomy'         => 'category',
						'terms'            => array_filter( array_map( 'intval', $queryAttr['categoryIds'] ) ),
						'include_children' => false,
					];
				}
				if ( ! empty( $queryAttr['tagIds'] ) ) {
					$tax_query[] = [
						'taxonomy'         => 'post_tag',
						'terms'            => array_filter( array_map( 'intval', $queryAttr['tagIds'] ) ),
						'include_children' => false,
					];
				}
				$query['tax_query'] = $tax_query;
			}
			if ( ! empty( $queryAttr['taxQuery'] ) ) {
				$query['tax_query'] = [];
				foreach ( $queryAttr['taxQuery'] as $taxonomy => $terms ) {
					if ( is_taxonomy_viewable( $taxonomy ) && ! empty( $terms ) ) {
						$query['tax_query'][] = [
							'taxonomy'         => $taxonomy,
							'terms'            => array_filter( array_map( 'intval', $terms ) ),
							'include_children' => false,
						];
					}
				}
			}
			if (
				isset( $queryAttr['order'] ) &&
				in_array( strtoupper( $queryAttr['order'] ), [ 'ASC', 'DESC' ], true )
			) {
				$query['order'] = strtoupper( $queryAttr['order'] );
			}
			if ( isset( $queryAttr['orderBy'] ) ) {
				$query['orderby'] = $queryAttr['orderBy'];
			}
			if (isset($queryAttr['metaKey'])) {
				$query['meta_key'] = $queryAttr['metaKey'];
			}			
			if (
				isset( $queryAttr['author'] ) &&
				(int) $queryAttr['author'] > 0
			) {
				$query['author'] = (int) $queryAttr['author'];
			}
			if ( ! empty( $queryAttr['search'] ) ) {
				$query['s'] = $queryAttr['search'];
			}
			if ( ! empty( $queryAttr['parents'] ) && is_post_type_hierarchical( $query['post_type'] ) ) {
				$query['post_parent__in'] = array_filter( array_map( 'intval', $queryAttr['parents'] ) );
			}
		}

		/**
		 * Filters the arguments which will be passed to `WP_Query` for the Query Loop Block.
		 * Anything to this filter should be compatible with the `WP_Query` API to form
		 * the query context which will be passed down to the Query Loop Block's children.
		 * This can help, for example, to include additional settings or meta queries not
		 * directly supported by the core Query Loop Block, and extend its capabilities.
		 * Please note that this will only influence the query that will be rendered on the
		 * front-end. The editor preview is not affected by this filter. Also, worth noting
		 * that the editor preview uses the REST API, so, ideally, one should aim to provide
		 * attributes which are also compatible with the REST API, in order to be able to
		 * implement identical queries on both sides.
		 *
		 * @param array $query Array containing parameters for `WP_Query` as parsed by the block context.
		 * @param WP_Block $block Block instance.
		 * @param int $page Current query's page.
		 *
		 * @since 6.1.0
		 */
		$query_args = apply_filters( 'query_loop_block_query_vars', $query, $this, $page );

		$taxFilters = $this->attribute( 'taxonomyFilters' );

		if ( ! empty( $taxFilters ) ) {
			$query_args['tax_query'] = $query_args['tax_query'] ?? [];
			foreach ( $taxFilters as $tax ) {
				if ( empty( $tax['slug'] ) ) {
					continue;
				}
				$slug          = $tax['slug'];
				$filterApplied = fndry_get_query_var( "query-$queryId-$slug" );
				if ( is_taxonomy_viewable( $slug ) && $filterApplied ) {
					$query_args['tax_query'][] = [
						'taxonomy'         => $slug,
						'field'            => 'slug',
						'terms'            => $filterApplied,
						'include_children' => true,
					];
				}
			}
		}

		$use_global_query = ( isset( $this->attribute( 'query' )['inherit'] ) && $this->attribute( 'query' )['inherit'] );
		if ( $use_global_query ) {
			global $wp_query;

			if ( $wp_query && isset( $wp_query->query_vars ) && is_array( $wp_query->query_vars ) ) {
				// Unset `offset` because if is set, $wp_query overrides/ignores the paged parameter and breaks pagination.
				unset( $query_args['offset'] );
				$query_args = wp_parse_args( $query_args, $wp_query->query_vars );
				if ( $page !== $query_args['paged'] ) {
					$query_args['paged'] = $page;
				}

				if ( empty( $query_args['post_type'] ) && is_singular() ) {
					$query_args['post_type'] = get_post_type( get_the_ID() );
				}
				$query_args['s'] = fndry_get_query_var( "query-$queryId-search" ) ?? $query_args['s'];
			}
		} else {
			// search would be set theoretically in the case of a global search query, so only do it otherwise
			$query_args['s'] = fndry_get_query_var( "query-$queryId-search" );

			if ( $query_args['offset'] > 0 ) {
				$query_args['ignore_sticky_posts'] = true;
			}
		}

		return new WP_Query( $query_args );
	}
}