<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
use ForgeSmith\Blocks\ReusableComponentBlock;

/**
 * Gets content from our blocks, which use serverside rendering and whatnot.
 * @param $blocks
 *
 * @return string
 */
function fndry_extract_text_from_blocks( $blocks ): string {
	$output = [];

	foreach ( $blocks as $block ) {
		if ( strlen( join( ' ', $output ) ) >= 55 ) {
			return join( ' ', $output );
		}

		// skip blocks that may cause recursion
		if ( in_array(
			$block['blockName'],
			[ 'foundry/query-block', 'core/query', 'foundry/related-posts', 'foundry/post-excerpt', 'core/excerpt' ]
		) ) {
			continue;
		}

		if ( ! empty( $block['attrs']['ref'] ) ) {
			/*
			 * if its a "dynamic" block, get its contents.
			 * debating whether core/block is really useful here -
			 * not sure if the content is going to be unique enough.
			*/
			$newContent = get_post_field( 'post_content',
				$block['attrs']['ref'] ?? $block['attrs']['refBlock'],
				'raw' );
			if ( $newContent ) {
				$newBlocks = parse_blocks( $newContent );
				$output[]  = fndry_extract_text_from_blocks( $newBlocks );
			}
		} elseif ( class_exists('ForgeSmith\Blocks\ReusableComponentBlock') && $block['blockName'] === 'foundry/reusable-component-block' && ! empty( $block['attrs']['refBlock'] ) ) {
			/*
			 * if its RCB, create a new RCB block and get the innerBlocks with overrides applied
			 * */
			$wp       = new WP_Block( $block );
			$rcb      = new ReusableComponentBlock( $block['attrs'], '', $wp );
			$output[] = fndry_extract_text_from_blocks( $rcb->innerBlocks );
		} elseif ( ! empty( $block['innerBlocks'] ) ) {
			/*
			 * otherwise, recurse through inner blocks
			 * */
			$output[] = fndry_extract_text_from_blocks( $block['innerBlocks'] );
		} elseif ( ! empty( $block['attrs']['content'] ) ) {
			/*
			 * we dont want no stinking shortcodes
			 * */
			$output[] = strip_shortcodes( $block['attrs']['content'] );
		}
	}

	return join( ' ', $output );
}

/**
 * WP only auto-generates excerpts from classic post content, not blocks. we dont like that, so now it does.
 * doing it this way allows it to appear in WP_JSON excerpt meta, so it'll show up in the editor too;
 * don't do block stuff if we
 *  a) already have an excerpt,
 *  b) don't support excerpts.
 * this is because wordpress likes to grab the post content of featured images for generating
 * said attachment's excerpt/description. This is bad, and causes memory leaks in the editor.
 * */
add_filter( 'get_the_excerpt', function ( $excerpt, $post ) {
	if ( $excerpt || ! post_type_supports( $post->post_type, 'excerpt' ) || ! $post ) {
		return $excerpt;
	}

	$content = get_the_content( null,null,$post->ID );
	if ( has_blocks( $content ) ) {
		$content = fndry_extract_text_from_blocks( parse_blocks( $content ) );
		// If there are blocks in this content, we shouldn't run wpautop() on it later.
		$priority = has_filter( 'the_content', 'wpautop' );
		if ( false !== $priority && doing_filter( 'the_content' ) && has_blocks( $content ) ) {
			remove_filter( 'the_content', 'wpautop', $priority );
			add_filter( 'the_content', '_restore_wpautop_hook', $priority + 1 );
		}
	}

	return wp_trim_words( $content );
}, 10, 2 );

/**
 * Adds our attribute filtering functionality so image attributes can be conditionally included or removed.
 */
add_filter( 'wp_get_attachment_image_attributes', 'fndry_filter_atts' );

/**
 * Removes the "auto" value from image sizes attributes. See: https://core.trac.wordpress.org/ticket/62515
 */
add_filter('wp_img_tag_add_auto_sizes', '__return_false');