<?php

namespace ForgeSmith;

use WP_Block;
use WP_Error;
use WP_Term;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

/**
 * @method getBlockData()
 * @method getAttribute()
 * @property $innerBlocks
 * @property $context
 * @property $providesContext
 */
trait FoundryBlockRenderingTools {

	private static string $anchorPattern = '/<a(.*?(?=href))(?:href=\".*?\")?(.*?)>(.*)<\/a>/';

	/**
	 * Slaps out a list of all the automatically-added CSS classes, plus optional custom classes.
	 * The optional $classes arg is sorta redundant these days, considering fndry_atts is so ubiquitous now.
	 *
	 * @param array $classes
	 * @param bool  $echo
	 *
	 * @return string|null
	 */
	public function rootClasses( array $classes = [], bool $echo = true ): ?string {
		if ( $echo ) {
			echo esc_attr( $this->renderBlockProps( $classes ) );
		} else {
			return $this->renderBlockProps( $classes );
		}

		return null;
	}

	/**
	 * Returns a string of the block's base CSS classes, sort of the same as blockProps.
	 * Optional classes can be provided as an array of strings, these will be parsed together - this allows custom
	 * class
	 * logic to be defined on the php template without requiring separate code to include it in the class attribute.
	 * Attribute props we look at:
	 *      - "baseClass":true => this prop signifies that the attribute adds a css class to our root element;
	 *                            if the attribute type is "string", then the saved value of the attribute is used
	 *                            if available.
	 *      - if the attribute type is "boolean", we look at two other properties:
	 *          - "value" => the css class to use if the attribute is set to true(thy)
	 *          - "modifier": true => signifies that the css class specified in value should be treated as a BEMM
	 *          modifier, and the class will be prepended with {$bemmClass}--
	 *
	 * @return string
	 */
	public function renderBlockProps( $customClasses = [], $returnString = true ): string|array {
		$block      = $this;
		$blockData  = $this->getBlockData();
		$savedAttrs = $this->getAttributes();
		$bemClass   = $this->bemClass ?? null;
		$registered = $blockData->block_type->attributes;

		$customClasses = array_filter( $customClasses );
		// lets filter out conditionals. with this, you can do something like ['css-class-name' => true, 'dont show this' => false]
		$customVals = array_filter( array_values( $customClasses ), function ( $val ) {
			return is_string( $val );
		} );
		$customKeys = array_filter( array_keys( $customClasses ), function ( $val ) {
			return is_string( $val );
		} );

		$classMods = array_merge( [ $bemClass ], $customKeys, $customVals );
		// allow for wp builtin custom class name
		$classMods[] = $savedAttrs['className'] ?? null;
		// only add classes if they have the baseClass property set to true;
		if ( isset( $registered ) ) {
			foreach ( $registered as $attrKey => $attrVal ) {
				if ( isset( $attrVal['baseClass'] ) && $attrVal['baseClass'] ) {
					$isModifier = $attrVal['modifier'] ?? false;
					if ( $attrVal['type'] === 'boolean' ) {
						if ( ! ! $savedAttrs[ $attrKey ] ) {
							// if type is boolean, we use the value specified in the attribute's "modifier" property if boolean === true;
							$classMods[] = $isModifier && $bemClass ? "$bemClass--{$attrVal['value']}" : $attrVal['value'];
						}
					} elseif ( $attrVal['type'] === 'object' ) {
						// are we generating responsive classes alongside our modifiers?
						if ( isset( $attrVal['responsiveUtilityType'] ) ) {
							// responsive = utility, never going to be BEMM modifiers (for now)
							// its fine if this just returns a giant string
							$classMods[] = $this->renderResponsiveProps( $attrKey );
						}
					} else {
						// if the attr type is string, we input the saved data only - we can assume these are either manual entry,
						// or involve a dropdown - in either case, whether a default value is used or not should be dictated by the editor.
						// if the saved attr type is a fndryId, maybeGet it.
						$finalVal = $savedAttrs[ $attrKey ] ?? null;
						if ( isset( $attrVal['isFndryId'] ) ) {
							$finalVal = Foundry::instance()->maybeGetSettingKey( $attrKey, $block );
						}
						$classMods[] = $isModifier && $bemClass ? "$bemClass--{$finalVal}" : $finalVal;
					}
				}
			}
		}

		$mods = array_filter( $classMods );

		return $returnString ? implode( ' ', $mods ) : $mods;
	}

	/**
	 * Renders a list of responsive CSS utility classes.
	 *
	 * @param $attrKey
	 * @param $responsiveType
	 *
	 * @return string|null
	 */
	public function renderResponsiveProps( $attrKey, $responsiveType = null ): ?string {
		// if no saved attr, then we might be looking *ONLY* at context.
		$savedAttr = $this->getAttribute( $attrKey ) ?: $this->getContext( $attrKey );

		if ( ! $savedAttr ) {
			return false;
		}

		$registered = $this->block_type->attributes;

		$prefixes = [
			'padding'   => 'p',
			'margin'    => 'm',
			'width'     => 'col',
			'align'     => 'align',
			'flex'      => 'flex',
			'justify'   => 'justify',
			'grid'      => 'grid',
			'display'   => 'd',
			'textAlign' => 'align-text',
		];

		// allow overriding
		if ( $responsiveType && isset( $prefixes[ $responsiveType ] ) ) {
			$responsiveType = $prefixes[ $responsiveType ];
		} else {
			if ( ! isset( $registered[ $attrKey ] ) ) {
				return false;
			}
			$attrVal        = $registered[ $attrKey ];
			$responsiveType = $prefixes[ $attrVal['responsiveUtilityType'] ];

			if ( isset( $attrVal['useContext'] ) && isset( $block->context[ $attrVal['useContext'] ] ) ) {
				$savedAttr = $this->recursive_array_merge_unique( $savedAttr,
					$block->context[ $attrVal['useContext'] ] );
			}
		}

		$prefix = "fndry-{$responsiveType}";

		$classes = [];

		foreach ( $savedAttr as $bp => $responsiveVal ) {
			if ( $bp === 'undefined' ) {
				continue;
			}
			// our key will always be a breakpoint if responsive => true
			// the difference is whether we're applying multiple classes (e.g. padding for all four sides), or a single class (e.g. align-items--lg-center);

			if ( is_array( $responsiveVal ) ) {
				// we have more to do here, e.g. for padding
				foreach ( $responsiveVal as $prop => $val ) {
					// these should only ever really apply to utility classes, which won't be following
					// BEMM structure.
					// for now, assume these will never be modifiers.
					$bpMod     = $bp !== 'all' ? "{$bp}-{$val}" : $val;
					$classes[] = "$prefix$prop--$bpMod";
				}
			} elseif ( ! ! $responsiveVal ) {
				$bpMod     = $bp !== 'all' ? "{$bp}-{$responsiveVal}" : $responsiveVal;
				$classes[] = "$prefix--$bpMod";
			}
		}

		return $classes ? implode( ' ', array_filter( $classes ) ) : null;
	}

	/**
	 * Allows nested arrays to be merge-able (e.g. overriding default label props)
	 *
	 * @param $a
	 * @param $b
	 *
	 * @return array
	 */
	protected function recursive_array_merge_unique( &$a, $b ): array {
		$a      = (array) $a;
		$b      = (array) $b;
		$result = $b;
		foreach ( $a as $k => &$v ) {
			// check isAssociative because raw arrays are fine (e.g. merging attributes incl taxonomyFilters)
			if ( is_array( $v ) && isset( $result[ $k ] ) && fndry_is_array_assoc( $v ) ) {
				$result[ $k ] = $this->recursive_array_merge_unique( $v, $result[ $k ] );
			} else {
				$result[ $k ] = $v;
			}
		}

		return $result;
	}

	/**
	 * Renders a block's children with some extra special stuff because we're bypassing a lot of native inner block
	 * handling - namely, skip_inner_blocks; We provide our own contexts, including a generic "block index" for all
	 * blocks so that they can have some idea of where they are in the grand scheme of things. Ignore OutputNotEscaped
	 * because we're echoing a block's render function, which *should* be escaped already.
	 *
	 * @param array $contexts
	 * @param bool  $isDynamic
	 *
	 * @return void
	 */
	public function renderInnerBlocks( array $contexts = [], bool $isDynamic = true ): void {
		$index = 0;
		if ( $this->innerBlocks ) {
			foreach ( $this->innerBlocks as $innerBlock ) {
				/*
				because we're doing skip_inner_blocks among other things,
				 contexts  would otherwise not pass over blocks that don't use or provide them,
				 meaning nested blocks relying on contexts one level or higher than ther immediate
				 parent are lost.
				 if only available_context wasn't protected :( see https://github.com/WordPress/gutenberg/pull/22334
				*/
				$contexts                    = array_merge(
					$this->context,
					$this->providesContext,
					$contexts,
					[ 'fndry/innerBlockIndex' => $index ],
				);
				$innerBlock['fndryContexts'] = $contexts;
				$innerBlock                  = new WP_Block( $innerBlock, $contexts );
				echo $innerBlock->render( [ 'dynamic' => $isDynamic ] ); //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				$index ++;
			}
		}
	}

	/**
	 * It's a BEM string output thing. Doesn't chain.
	 *
	 * @param $mod
	 * @param $condition
	 *
	 * @return false|string
	 */
	public function bemMod( $mod, $condition = true ) {
		$bem = $this->bemClass;
		if ( ! $bem || ! $mod || ! $condition ) {
			return false;
		}

		return "{$bem}--{$mod}";
	}

	/**
	 * @param $sub
	 *
	 * @return false|string
	 */
	public function bemSub( $sub ) {
		$bem = $this->bemClass;
		if ( ! $bem || ! $sub ) {
			return false;
		}

		return "{$bem}__{$sub}";
	}

	/**
	 * Renders our very neat background object attributes into actual CSS background properties!
	 * 1.9.x - Added responsive support.
	 *
	 * @param string $attr
	 *
	 * @return array
	 */
	public function renderBackground( string $attr = 'background' ): array {
		$instance = Foundry::instance();
		$bg       = $this->getAttribute( $attr );

		if ( empty( $bg ) ) {
			return [];
		}

		$props = [];

		// default layer's bg color can be set to undefined, so it needs to be initialized.
		foreach (
			[
				'all' => [ 'color' => $bg['color'] ?? 'transparent', 'images' => $bg['images'] ?? [] ],
				'md'  => $bg['md'] ?? false,
				'sm'  => $bg['sm'] ?? false,
			] as $key => $layer
		) {
			$color  = isset( $layer['color'] ) ? $instance->maybeGetSettingKey( $layer['color'] ) : null;
			$images = $layer['images'] ?? false;

			$imgTemp = [];

			if ( $images ) {
				foreach ( $images as $imgItem ) {
					if ( empty( $imgItem ) ) {
						continue;
					}
					$tmp = [];

					$image = $imgItem['image'] ?? false;
					if ( ! $image ) {
						continue;
					}
					if ( ! is_array( $image ) ) {
						$image = wp_get_attachment_image_url( $image, 'full' );
						$image && $tmp[] = "url($image)";
					} else {
						$angle = $image['angle'];
						$stops = $image['stops'];
						if ( $angle !== false && ! ! $stops ) {
							$tmp[] = "linear-gradient({$angle}deg, " .
							         join( ', ',
								         array_map(
									         fn( $stop ): string => join(
										         ' ',
										         [
											         ! empty( $stop['color'] ) ? $instance->maybeGetSettingKey( $stop['color'] ) : 'transparent',
											         $stop['position'],
										         ]
									         ),
									         $stops )
							         ) . ")";
						}
					}
					foreach ( [ 'attachment', 'clip', 'origin', 'repeat' ] as $prop ) {
						if ( isset( $imgItem[ $prop ] ) ) {
							$tmp[] = $imgItem[ $prop ];
						}
					}
					$propTmp = [];

					$propTmp[] = ! empty( $imgItem['position'] ) ? $imgItem['position'] : 'center';
					$propTmp[] = ! empty( $imgItem['size'] ) ? $imgItem['size'] : 'auto';

					! ! $propTmp && $tmp[] = join( '/', $propTmp );
					$imgTemp[] = join( ' ', $tmp );
				}
			}

			$final         = join( ', ', array_filter( [ ...$imgTemp, $color ] ) );
			$props[ $key ] = $final;
		}

		return array_filter( [
			'--fndry-bg'     => $props['all'],
			'--fndry-bg--md' => $props['md'],
			'--fndry-bg--sm' => $props['sm'],
		] );
	}

	/**
	 * Slaps out "position" & position-related CSS properties - z-index, top, left, yadda yadda.
	 *
	 * @param string $attr
	 *
	 * @return string|null
	 */
	public function applyPositioning( string $attr = 'positionProps' ): ?string {
		$posArray = $this->getAttribute( $attr );
		$temp     = [];
		$output   = '';

		if ( ! isset( $posArray ) || ! is_array( $posArray ) ) {
			return $output;
		}
		foreach ( $posArray as $k => $v ) {
			! ! $v and $temp[] = "$k:$v";
		}

		return join( ';', $temp ) ?: null;
	}

	/**
	 * Pass a list of block attributes, get a list of css variables (assuming they're the same name.)
	 * Return values are NOT transformed, so you're getting the raw fndryIds.
	 *
	 * @param array    $attrs
	 * @param          $id
	 * @param          $echo
	 *
	 * @return false|string
	 */
	public function getCSSVars( array $attrs, $id = null, $echo = true ) {
		if ( ! $attrs || ! is_array( $attrs ) ) {
			return false;
		}

		$formattedAttrs = array_map( function ( $key ) use ( $id ) {
			$cssVar = $this->getAttribute( $key, $id );

			return $cssVar ? "--$key:$cssVar;" : false;
		}, $attrs );
		if ( $echo ) {
			echo esc_attr( implode( $formattedAttrs ) );

			return false;
		}

		return implode( $formattedAttrs );
	}

	/**
	 * Sort of like getCSSVars, except we can also pass in desired keys for the CSS variables.
	 *
	 * @param $attrs
	 * @param $echo
	 *
	 * @return false|string|void
	 */
	public function generateCSSVars( $attrs, $echo = false ) {
		if ( ! is_array( $attrs ) ) {
			return false;
		}

		$tmp = [];

		foreach ( $attrs as $var => $attr ) {
			// if just passing a simple array, do the transformations here.
			// otherwise, assume they're explicitly passing a custom css variable name + attribute value for a reason!
			if ( ! is_string( $var ) ) {
				$var  = "--fndry-{$attr}";
				$attr = $this->getAttribute( $attr );
			}
			$tmp[] = "$var:$attr;";
		}

		$tmp = implode( '', array_filter( $tmp ) );

		if ( $echo ) {
			echo esc_attr( $tmp );
		} else {
			return $tmp;
		}
	}

	public function renderResponsiveAttributes( $classes, $echo = true ): ?string {
		// if string, then we're just doin like 1 attr and nothing else
		if ( is_string( $classes ) ) {
			if ( $echo ) {
				echo esc_attr( $this->renderResponsiveProps( $classes ) );
			} else {
				return $this->renderResponsiveProps( $classes );
			}
		} elseif ( is_array( $classes ) ) {
			$tmp = [];

			foreach ( $classes as $class => $type ) {
				$classToRender = $class;
				// if $key is int, then we're not an associative array.
				// if we're associative, we're doing something with context.
				// proper usage would be ['contextName' => 'responsiveType'] e.g. ['foundry/childPadding' => 'padding']
				if ( is_int( $class ) ) {
					$classToRender = $type;
					$type          = null;
				}
				$tmp[] = $this->renderResponsiveProps( $classToRender, $type );
			}

			if ( $echo ) {
				echo esc_attr( implode( ' ', array_filter( $tmp ) ) );
			} else {
				return implode( ' ', array_filter( $tmp ) );
			}
		}

		return null;
	}

	/**
	 * @param array|null  $borders Border attribute. Must be associative array / object.
	 * @param string|null $prefix  Add a custom prefix if you want to use the output as css variables.
	 *
	 * @return string
	 */
	function doBorderStyles( array|null $borders, string $prefix = null ) {
		$test = [];
		if ( $borders && is_countable( $borders ) ) {
			foreach ( $borders as $border => $value ) {
				if ( ! isset( $value['width'] ) || $value['width'] === '' ) {
					continue;
				}
				! isset( $value['style'] ) and $value['style'] = 'solid';
				isset( $value['color'] ) and $value['color'] = Foundry::instance()->maybeGetSettingKey( $value['color'] );
				unset( $value['hoverColor'] );
				
				$test[] = ( "{$prefix}{$border}:" ) . implode( ' ',
						$value );
			}
		}

		return join( ';', $test );
	}

	function doBorderHoverStyles( array|null $borders, string $prefix = null ) {
		$test = [];
		if ( $borders && is_countable( $borders ) ) {
			foreach ( $borders as $border => $value ) {
				if ( ! isset( $value['hoverColor'] ) || $value['hoverColor'] === '' ) {
					continue;
				}
				$hover = Foundry::instance()->maybeGetSettingKey( $value['hoverColor'] );

				$test[] = "{$prefix}{$border}:" . $hover;
			}
		}

		return join( ';', $test );
	}

	/**
	 * @param $filters
	 *
	 * @return array
	 */
	function doCSSFilters( $filters ): array {
		if ( ! is_array( $filters ) ) {
			return [];
		}

		$ret = [];

		foreach ( $filters as $key => $filter ) {
			$unit = "";
			$decimal = true;
			
			switch ($key) {
				case 'blur':
					$unit = 'px';
					break;
				case 'hue-rotate':
					$unit = 'deg';
					$decimal = false;
					break;
				case 'saturate':
					$unit = '%';
					$decimal = false;
					break;
			}
			if ($decimal) {
				$filter = ($filter / 100);
			}
			$ret["--filter-{$key}"] = "{$filter}{$unit};";
		}

		return $ret;
	}

	/**
	 * Renders a button element. Mostly useful because of how we handle the FontAwesome icon shortcodes from the global
	 * ButtonStyle for this.
	 *
	 * @param string|null $text
	 * @param string|null $btnKey  The raw, untransformed fndryId value. Use getAttributes method, as this preserves
	 *                             the original value.
	 * @param array       $atts    It's an array of attributes, same format as doAtts.
	 * @param string      $tagName Tag name!
	 *
	 * @return string
	 */
	public function renderButton(
		string|null $text,
		string|null $btnKey,
		array $atts = [],
		string $tagName = 'button'
	): string {
		$btnClsObj = Foundry::instance()->getOptionById( $btnKey );
		if ( $btnClsObj['hasIcon'] ?? false ) {
			$iconData = $btnClsObj['icon'] ?? false;
			if ( $iconData ) {
				$icon = isset( $btnClsObj['icon']['faId'] ) ? "[icon name='{$btnClsObj['icon']['faId']}' prefix='{$btnClsObj['icon']['faPrefix']}']" : null;
				if ( $icon ) {
					$iconPos                                = $iconData['position'] ?? null;
					$atts['style']['--fndry-btn-icon-size'] = $iconData['size'] ?? $iconData['faSize'] ?? null;
					$text                                   = $iconPos === 'left' ? "$icon $text" : "$text $icon";
				}
			}
		}

		// just in case we get a string for some reason
		if ( ! empty( $atts['class'] ) && ! is_array( $atts['class'] ) ) {
			$atts['class'] = (array) $atts['class'];
		}

		if ( isset( $btnClsObj['key'] ) ) {
			$atts['class'][] = $btnClsObj['key'];
		}
		$atts['class'][] = 'fndry-btn';

		$atts['class'] = array_unique( $atts['class'] );

		$atts = fndry_atts( $atts, false );

		return "<{$tagName} {$atts}>{$text}</{$tagName}>";
	}

	/**
	 * Returns a list of terms that are applicable within the context the current wpQuery.
	 * This excludes terms that do not belong to any post within the current query.
	 * This includes empty terms!
	 *
	 * @param array $args
	 *
	 * @return int[]|string|string[]|WP_Error|WP_Term[]
	 */
	public function getRelatedTerms( array $args ) {
		$query = $this->getContext( 'wpQuery' );
		// just do regular thing if no query object
		if ( ! $query ) {
			return get_terms( $args );
		}

		// ->posts property only contains the max per_page, and we need to check ALL of them.
		// of course, that's only if we don't already have all the possible matching posts ;)
		if (
			! empty( $query->query['posts_per_page'] )
			&& (
				$query->query['posts_per_page'] === - 1
				|| ( $query->query['posts_per_page'] >= $query->found_posts )
			)
		) {
			$posts = $query->posts;
		} else {
			$queryArgs = $query->query;
			unset( $queryArgs['posts_per_page'] );
			$posts = get_posts( wp_parse_args( [ 'numberposts' => - 1 ], $queryArgs ) );
		}

		$foundPosts = array_column( $posts ?: [], 'ID' );

		return get_terms( wp_parse_args( $args, [
			'object_ids' => $foundPosts,
		] ) );
	}
}
