<?php

namespace ForgeSmith;

use ReflectionClass;
use ReflectionException;
use WP_Block;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly
class FoundryBlock {

	use FoundryBlockRenderingTools;

	public $bemClass;

	public  $name            = '';
	public  $block_type;
	public  $innerBlocks     = [];
	public  $context         = [];
	public  $providesContext = []; // TODO - property is only written but never read - do we need this
	public  $templatePath    = '';
	private $shortName       = '';
	private $dirPath         = '';// TODO - property is only written but never read - do we need this
	private $jsonPath        = '';
	private $attributes      = [];
	private $content         = '';
	private $block           = null;
	private $output          = null;

	protected array $deprecations = [];

	protected string $version = 'FREE';

	public function __construct( $attributes, $content = '', $wp_block = '' ) {
		// we need to abort here, otherwise wordpress will try to render foundry blocks as serverside when fetching
		// from post content in the editor.
		if ( is_admin() ) {
			return null;
		}
		/**
		 * Do not do any of this stuff from inside the block editor.
		 * */
		$referer = wp_get_referer();
		if (
			( defined( 'REST_REQUEST' ) && REST_REQUEST ) &&
			(
				// this first condition is for when we're doing stuff in the editor
				( ! empty( $_GET['context'] ) && $_GET['context'] === 'edit' ) ||
				// these last two address autosaves 
				( $referer && str_contains( $referer, 'action=edit' ) ) ||
				( $referer && str_contains( $referer, 'post-new.php' ) )
			)
		) {
			return;
		}

		/* populate props */
		$this->shortName    = ( new ReflectionClass( $this ) )->getShortName();
		$this->dirPath      = ( $this->version === 'PRO' && defined( 'REFOUNDRY_PRO_DIR' ) ? REFOUNDRY_PRO_DIR : REFOUNDRY_CORE_DIR ) . "/src/blocks/$this->shortName";
		$this->jsonPath     = "$this->dirPath/block.json";
		$this->templatePath = "$this->dirPath/template.php";

		// $attributes object is only saved values, but $wp_block->attributes is saved PLUS defaults.
		$this->setAttributes( $wp_block->attributes );
		$this->name        = $wp_block->name;
		$this->innerBlocks = $wp_block->parsed_block['innerBlocks'] ?? [];
		$this->content     = $content; // this is usually empty anyway so??
		$this->block       = $wp_block;
		$this->block_type  = $this->block->block_type;
		$this->generateProvidesContext();
		$this->context = array_merge( $wp_block->context,
			$wp_block->available_context ?? [],
			$wp_block->parsed_block['fndryContexts'] ?? [] );
		! isset( $this->context['postId'] ) and $this->context['postId'] = get_the_ID();
		! isset( $this->context['postType'] ) and $this->context['postType'] = get_post_type();

		$this->handleDeprecations();
		// overload templatePath here and stuff

		/**
		 * Filter to modify Foundry Block after initial __construct,
		 * but before subclasses have done their own __constructs.
		 *
		 * @param FoundryBlock $this The Foundry Block class. Do with it what you will.
		 *
		 * @since 1.0.0
		 */
		apply_filters( 'foundry/foundry-block', $this );
		/**
		 * Filter to modify this specific subclass of Foundry Block after initial __construct,
		 * but before continuing with subclass __construct.
		 *
		 * @param FoundryBlock $this This specific Foundry Block subclass. Do with it what you will.
		 *
		 * @since 1.0.0
		 */
		apply_filters( $this->name, $this );
	}

	/**
	 * If a block has a deprecations array defined, and the saved data doesn't match the "rendered"
	 * (i.e. sanitized) block attributes, then cycle through and apply the deprecation handlers.
	 *
	 * @return void
	 */
	private function handleDeprecations(): void {
		$savedAtts = $this->block->parsed_block['attrs'] ?? [];
		if ( ! empty( $this->deprecations ) && ! empty( $savedAtts ) ) {
			foreach ( $this->deprecations as $dep => $handler ) {
				if ( ! empty( $savedAtts[ $dep ] ) && is_callable( $handler ) ) {
					$this->setAttribute( $dep, call_user_func( $handler, $savedAtts[ $dep ] ) );
				}
			}
		}
	}

	/**
	 * Build our own context for the front-end.
	 * Abort if no block_type, which can happen during REST requests.
	 *
	 * @return void
	 */
	private function generateProvidesContext(): void {
		if ( ! $this->block_type ) {
			return;
		}
		if ( $this->block_type->provides_context ) {
			foreach ( $this->block_type->provides_context as $k => $v ) {
				$this->providesContext[ $k ] = $this->getAttribute( $v );
			}
		}
	}

	/**
	 * Get the raw attribute value.
	 *
	 * @param $key
	 *
	 * @return mixed|null
	 */
	public function getAttribute( $key ) {
		return $this->lodashGet( $key, $this->attributes );
	}

	/**
	 * Use lodash-like notation to get nested properties in multidimensional arrays.
	 *
	 * @param $name
	 * @param $value
	 *
	 * @return mixed|null
	 */
	protected function lodashGet( $name, $value ) {
		$keys = explode( '.', $name );

		$ret = null;

		foreach ( $keys as $key ) {
			if ( $ret === null && isset( $value[ $key ] ) ) {
				$ret = $value[ $key ];
			} elseif ( is_array( $ret ) ) {
				$ret = $ret[ $key ] ?? null;
			}
		}

		return $ret;
	}

	/**
	 * @param                    $attributes
	 * @param string             $content
	 * @param WP_Block|string    $wp_block
	 *
	 * @return mixed
	 * @throws ReflectionException
	 */
	public static function create( $attributes, string $content = '', WP_Block|string $wp_block = '' ) {
		$inst = new ReflectionClass( get_called_class() );
		$inst = $inst->newInstance( $attributes, $content, $wp_block );

		return $inst->renderCallback();
	}

	public function renderCallback() {
		// we're not using render callbacks in the editor, so there's no reason to do any processing
		// unless we're on the frontend.
		if ( is_admin() ) {
			return false;
		}

		$post_id = get_the_ID();

		// todo - we should be hooking into Foundry here for extending.
		$block = $this->getRenderedBlock( $this->attributes, $this->content, $post_id );

		/**
		 * Generic filter to override Foundry Block's render callback.
		 *
		 * @param string       $block Output retrieved from getRenderedBlock.
		 * @param FoundryBlock $this  The Foundry Block class. Do with it what you will.
		 *
		 * @since 1.4.0
		 */
		$block = apply_filters( 'foundry/foundry-block/render-callback', $block, $this );

		/**
		 * Filter to override a Foundry Block subclass's render callback.
		 *
		 * @param string       $block Output retrieved from getRenderedBlock.
		 * @param FoundryBlock $this  The Foundry Block subclass. Do with it what you will.
		 *
		 * @since 1.4.0
		 */
		return apply_filters( $this->name . '/render-callback', $block, $this );
	}

	private function getRenderedBlock( $attributes, $content = '', $post_id = 0, $forceRender = false ) {
		$blockID = null;

		if ( ! isset( $attributes['fndryBlockId'] ) ) {
			$attributes['fndryBlockId'] = uniqid( 'fndry-block' );
			$blockID                    = $attributes['fndryBlockId'];
		}

		if ( $this->output && ! $forceRender ) {
			// just in case we've already rendered, we don't want to re-render again!
			return $this->output;
		}

		// Capture block render output.
		extract( [ $this ] );
		ob_start();
		$this->renderBlock( $attributes, $content, $post_id, $blockID );
		$html = ob_get_clean();

		// Escape "$" character to avoid "capture group" interpretation.
		$content = str_replace( '$', '\$', $content );
		// Store in cache for preloading.
		$this->output = preg_replace( '/<InnerBlocks([\S\s]*?)\/>/', $content, $html );

		return $this->output;
	}

	private function renderBlock(): void {
		if ( file_exists( $this->templatePath ) ) {
			include( $this->templatePath );
		}
	}

	/**
	 * Get the attribute value, transforming fndryIds to usable data.
	 *
	 * @param $fieldName
	 *
	 * @return mixed|string|null
	 */
	public function attribute( $fieldName ) {
		$instance = Foundry::instance();

		// TODO - this should be on block, not instance - block can better determine whether field isFndryId
		$attrVal = $instance->maybeGetSettingKey( $fieldName, $this );

		// Can only do overrides if we have the postId in the context.
		if ( isset( $this->context['postId'] ) ) {
			$overrides = $this->getAttribute( 'overrides' );
			$metaKey   = $this->lodashGet( $fieldName, $overrides );
			if ( ! ! $metaKey && is_string( $metaKey ) ) {
				$overriddenValue = get_post_meta( $this->context['postId'], $metaKey, 1 );
				if ( ! empty( $overriddenValue ) ) {
					$attrVal = $overriddenValue;
				}
			}
		}

		return $attrVal ?? null;
	}

	public function context( $key ) {
		$context = $this->getContext( $key );
		if ( $context ) {
			$instance = Foundry::instance();
			$context  = $instance->maybeGetSettingKey( $context );
		}

		return $context;
	}

	public function getContext( $key ) {
		return $this->lodashGet( $key, $this->context );
	}

	public function getBlockData() {
		return $this->block;
	}

	public function setAttribute( $name, $value ) {
		return $this->attributes[ $name ] = $value;
	}

	public function getAttributes() {
		return $this->attributes;
	}

	public function setAttributes( $attributes ) {
		return $this->attributes = $attributes;
	}

	/**
	 * Transform a block array into a format compatible with WP_Block.
	 *
	 * @param array $block The block array.
	 *
	 * @return array|null Returns a block array or null for invalid blocks.
	 */
	protected function transformBlock( array $block ) {
		if ( isset( $block['name'], $block['attributes'], $block['innerBlocks'] ) ) {
			// Instead of returning a WP_Block instance, return an array with the block data
			return [
				'blockName'   => $block['name'],
				'attrs'       => $block['attributes'],
				'innerBlocks' => array_map( [ $this, 'transformBlock' ], $block['innerBlocks'] ),
			];
		}

		return null;
	}
}
